#!/usr/local/bin/perl -w
#
# Docs are at the bottom below the =head1.
# You could just type perldoc nmap2nagios.pl;
#

use 5.006;
use strict;
use warnings;
 
# For debugging only
use Data::Dumper;
$Data::Dumper::Indent = 1;
$Data::Dumper::Maxdepth = 3;
 
use File::Path;
use File::Basename;
use Getopt::Std;
 
use IO::File;

use XML::Simple;
 
my $data_ref = {};
 
($data_ref->{'Program'},
 $data_ref->{'Path'}) = fileparse($0);
 
$data_ref->{'Version'} = '0.1.2';

$data_ref->{'nmap2nagios_args'} = join(' ', $data_ref->{'Program'}, @ARGV);

my $options_ref = {};
getopts('hvVic:r:o:', $options_ref);
$options_ref->{'configuration_file'}      = $options_ref->{'c'};
$options_ref->{'results_file'}            = $options_ref->{'r'};
$options_ref->{'output_file'}             = $options_ref->{'o'};
$options_ref->{'ignore_unknown_services'} = $options_ref->{'i'};
$options_ref->{'VERBOSE'}                 = $options_ref->{'V'};
$options_ref->{'Verbose'}                 = $options_ref->{'v'} || $options_ref->{'V'};
 
if (defined $options_ref->{'h'} ||
    !$options_ref->{'r'} ||
    !$options_ref->{'o'}) {
  print $data_ref->{'Program'}, ": -h -v -i -r {'nmap_results_file'} -o {'output_file'}\n" .
        "  -i Ignore Unknown Services\n" .
        "  -v Verbose\n" .
        "  -V Serious Verbose\n" .
        "  -h This screen\n";
  exit;
}
 
if ($options_ref->{'output_file'} eq 'hosts.cfg') {
  print "I'm sorry.  It's not that I don't trust you, but I'm not going to possibly overwrite your hosts.cfg\n";
  exit;
}

$options_ref->{'configuration_file'} = $data_ref->{'Path'} . 'nmap2nagios.conf' if (!$options_ref->{'configuration_file'});

# XML::Simple is picky about it's file names beginning with a valid path
$options_ref->{'configuration_file'} = './' . $options_ref->{'configuration_file'} if ($options_ref->{'configuration_file'} !~ /^(\/|\.\/)/);

my $config_ref = load_configuration($options_ref);

if ($config_ref) {
  my $results_ref = load_results($options_ref);
  if ($results_ref) {
    my $hosts_ref = process_results($options_ref, $results_ref);
    if ($hosts_ref) {
      output_netsaint_config($options_ref, $config_ref, $hosts_ref);
    }
  }
}

sub process_results {
  my $options_ref = shift;
  my $results_ref = shift;
  my $hosts_ref = shift;

  $hosts_ref = {} if (!defined $hosts_ref);

  # First we'll get the current time
  my $run_info_ref = split_timestamp();

  # First we'll copy in all the stats
  $run_info_ref->{'nmap2nagios_version'} = 'v' . $data_ref->{'Version'};
  $run_info_ref->{'nmap2nagios_args'} = $data_ref->{'nmap2nagios_args'};
  $run_info_ref->{'nmap_args'} = $results_ref->{'args'};
  $run_info_ref->{'nmap_version'} = $results_ref->{'version'};
  $run_info_ref->{'nmap_start_timestamp'} = $results_ref->{'start'};
  my $start_time_ref = split_timestamp($run_info_ref->{'nmap_start_timestamp'});
  $run_info_ref->{'nmap_start'} = $start_time_ref->{'date_time'};
  $run_info_ref->{'nmap_finish_timestamp'} = $results_ref->{'runstats'}->{'finished'}->{'time'};
  my $finish_time_ref = split_timestamp($run_info_ref->{'nmap_finish_timestamp'});
  $run_info_ref->{'nmap_finish'} = $finish_time_ref->{'date_time'};
  $run_info_ref->{'nmap_duration'} = seconds2minutes($run_info_ref->{'nmap_finish_timestamp'} - $run_info_ref->{'nmap_start_timestamp'});

  $run_info_ref->{'nmap_hosts_up'} = $results_ref->{'runstats'}->{'hosts'}->{'up'};
  $run_info_ref->{'nmap_hosts_down'} = $results_ref->{'runstats'}->{'hosts'}->{'down'};
  $run_info_ref->{'nmap_hosts_total'} = $results_ref->{'runstats'}->{'hosts'}->{'total'};

  $hosts_ref->{'run_info'} = $run_info_ref;

  # host is not an array if only one host is returned by nmap
  $results_ref->{'host'} = [ $results_ref->{'host'} ] if (ref($results_ref->{'host'}) ne 'ARRAY');

  # Then we are read to process the host/service information
  foreach my $nmap_host_ref (@ { $results_ref->{'host'} }) {
    next if ($nmap_host_ref->{'status'}->{'state'} ne 'up');
    # Can we use the nmap results to determine the network and broadcast addresses?
    next if ($nmap_host_ref->{'address'}->{'addr'} =~ /\.0$/);
    next if ($nmap_host_ref->{'address'}->{'addr'} =~ /\.255$/);

    my $host_ref = {};
    $host_ref->{'address'} = $nmap_host_ref->{'address'}->{'addr'};
    $host_ref->{'host_name'} = $nmap_host_ref->{'hostnames'}->{'hostname'}->{'name'};

    $host_ref->{'osmatch'} = $nmap_host_ref->{'os'}->{'osmatch'}->{'name'};

    if (!$host_ref->{'host_name'}) {
      $host_ref->{'host_name'} = $host_ref->{'address'};
      $host_ref->{'host_name'} =~ s/\./-/g;
      $host_ref->{'host_name'} .= '.' . $config_ref->{'domain'};
    }

    print '  Processed Host (', $host_ref->{'host_name'}, ")\n" if ($options_ref->{'Verbose'});

    # Need to try to match a host entry based on the host_name

    # Load up the defaults
    foreach my $field (keys % { $config_ref->{'default_host'} }) {
      $host_ref->{$field} = $config_ref->{'default_host'}->{$field} if (!$host_ref->{$field});
    }

    # Need to default the host_alias
    $host_ref->{'host_alias'} = $host_ref->{'host_name'} if (!$host_ref->{'host_alias'});

    # We at least want connectivity monitoring
    my $port_ref = {
      portid => 0,
      service => {
        name => 'ping'
      }
    };
    push @ { $nmap_host_ref->{'ports'}->{'port'}}, $port_ref;
     
    # Turn the detected port list into something useful
    foreach my $port_ref (@ { $nmap_host_ref->{'ports'}->{'port'} }) {
      my $matched_service_ref;
      # See if we recognize this service
      if (defined $port_ref->{'service'}->{'name'} &&
          defined $config_ref->{'service'}->{$port_ref->{'service'}->{'name'}}) {
        $matched_service_ref = $config_ref->{'service'}->{$port_ref->{'service'}->{'name'}};
      }
      elsif (defined $options_ref->{'ignore_unknown_services'}) {
        # Dump this service if we don't already know about it
        next;
      }
      else {
        # Use the default service definition
        $matched_service_ref = $config_ref->{'service'}->{'default'};
      }

      # Dump this service if they have it disabled
      next if (defined $matched_service_ref->{'disabled'});

      # Load up the defaults
      my $service_ref = {};
      foreach my $field (keys %{$matched_service_ref}) {
	$service_ref->{$field} = $matched_service_ref->{$field} if (!$service_ref->{$field});
      }

      # Need to set some generic info for default checks
      if ($matched_service_ref->{'name'} eq 'default') {
	# In the case where nmap couldn't determine a service name we want to 
	# preface it with 'unknown'
        $port_ref->{'service'}->{'name'} = 'unknown' if (!$port_ref->{'service'}->{'name'});

	$service_ref->{'service_description'} = join('-', $port_ref->{'service'}->{'name'},
                                                          $port_ref->{'protocol'},
                                                          $port_ref->{'portid'});

	$service_ref->{'check_command'} = 'check_' . $port_ref->{'protocol'} . '!' . $port_ref->{'portid'};
      }

      $service_ref->{'port'} = $port_ref;

      $host_ref->{'service'}->{$service_ref->{'port'}->{'portid'}} = $service_ref;
    }

    # Now try to find a hostgroup to put it in based on it's OS type.
    foreach my $hostgroup_ref (@{$config_ref->{'hostgroup'}}) {
      next if ($hostgroup_ref->{'group_name'} eq 'default');
      next if (!defined $host_ref->{$hostgroup_ref->{'match'}->{'field'}});
      if ($host_ref->{$hostgroup_ref->{'match'}->{'field'}} =~ /$hostgroup_ref->{'match'}->{'data'}/i) {
	if ($options_ref->{'Verbose'}) {
	  print '    Matched Hostgroup (', $hostgroup_ref->{'group_name'}, ")\n";
	}

	if ($options_ref->{'VERBOSE'}) {
	  print '      Field: (', $hostgroup_ref->{'match'}->{'field'}, ")\n",
		'      Data: (', $host_ref->{$hostgroup_ref->{'match'}->{'field'}}, ")\n",
		'      Match: (', $hostgroup_ref->{'match'}->{'data'}, ")\n" if ($options_ref->{'VERBOSE'});
	}

	$host_ref->{'hostgroup'} = $hostgroup_ref;
	$hostgroup_ref->{'host'}->{$host_ref->{'host_name'}} = $host_ref;
	last;
      }
    }

    if (!$host_ref->{'hostgroup'}) {
      print "    Unable to match HostGroup setting to (default)\n" if ($options_ref->{'Verbose'});

      $host_ref->{'hostgroup'} = $config_ref->{'default_hostgroup'};
      $config_ref->{'default_hostgroup'}->{'host'}->{$host_ref->{'host_name'}} = $host_ref;
    }

    print "\n" if ($options_ref->{'Verbose'});

    $hosts_ref->{'host'}->{$host_ref->{'address'}} = $host_ref;
  }

  if ($options_ref->{'VERBOSE'}) {
    print "The following HostGroups have been generated\n\n";
    foreach my $hostgroup_ref (@ { $config_ref->{'hostgroup'} }) {
      # Did we find any hosts for this group?
      if (scalar keys % {$hostgroup_ref->{'host'}} ) {
	print '  Hostgroup: ', $hostgroup_ref->{'group_name'}, "\n",
	      '    Alias:         ', $hostgroup_ref->{'group_alias'}, "\n",
	      '    ContactGroups: ', $hostgroup_ref->{'contactgroups'}, "\n",
	      '    Members:       ', join(",\n                   ", sort keys % { $hostgroup_ref->{'host'} }), "\n",
	      "\n";

        # Just it case I decide to add more detailed Member output (like which services each one has)
        #foreach my $host_ref (sort { $a->{'host_name'} cmp
        #                             $b->{'host_name'} } values % { $hostgroup_ref->{'host'} }) {
        #  print Dumper($host_ref) if ($options_ref->{'VERBOSE'});
        #}
      }
    }
  }

  return $hosts_ref;
}

sub output_netsaint_config {
  my $options_ref = shift;
  my $config_ref = shift;
  my $hosts_ref = shift;

  print "Generating '", $options_ref->{'output_file'}, "':\n" if ($options_ref->{'Verbose'});

  my $cfg_fh = new IO::File('>' . $options_ref->{'output_file'});
  if ($cfg_fh) {
    my $header = process($config_ref->{'template'}->{'header'}, $hosts_ref->{'run_info'});
    print $cfg_fh $header;

    foreach my $hostgroup_ref (@ { $config_ref->{'hostgroup'} }) {
      # Did we find any hosts for this group?
      if (scalar keys % {$hostgroup_ref->{'host'}} ) {
        # Generate the hosts field;
	$hostgroup_ref->{'hosts'} = join(',', sort keys % { $hostgroup_ref->{'host'} });

        my $hostgroup_header = process($config_ref->{'template'}->{'hostgroup_header'}, $hostgroup_ref);
        print $cfg_fh $hostgroup_header;

        my $hostgroup_entry = process($config_ref->{'template'}->{'hostgroup_entry'}, $hostgroup_ref);
        print $cfg_fh $hostgroup_entry;

	foreach my $host_ref (sort { $a->{'host_name'} cmp
				     $b->{'host_name'} } values % { $hostgroup_ref->{'host'} }) {
	  my $host_header = process($config_ref->{'template'}->{'host_header'}, $host_ref);
	  print $cfg_fh $host_header;

	  my $host_entry = process($config_ref->{'template'}->{'host_entry'}, $host_ref);
	  print $cfg_fh $host_entry;

	  foreach my $service_ref (sort { $a->{'port'}->{'portid'} <=>
				          $b->{'port'}->{'portid'} } values % { $host_ref->{'service'} }) {
            $service_ref->{'host_name'} = $host_ref->{'host_name'};
	    my $service_header = process($config_ref->{'template'}->{'service_header'}, $service_ref);
	    print $cfg_fh $service_header;

	    my $service_entry = process($config_ref->{'template'}->{'service_entry'}, $service_ref);
	    print $cfg_fh $service_entry;

	    my $service_footer = process($config_ref->{'template'}->{'service_footer'}, $service_ref);
	    print $cfg_fh $service_footer;
	  }

	  my $host_footer = process($config_ref->{'template'}->{'host_footer'}, $host_ref);
	  print $cfg_fh $host_footer;
	}

        my $hostgroup_footer = process($config_ref->{'template'}->{'hostgroup_footer'}, $hostgroup_ref);
        print $cfg_fh $hostgroup_footer;
      }
    }
  }
  else {
    print "Unable to create '", $options_ref->{'output_file'}, "'\n";
  }
}

sub load_configuration {
  my $options_ref = shift;

  my $config_ref = undef;
  if (!-e $options_ref->{'configuration_file'}) {
    print STDERR 'Unable to read (', $options_ref->{'configuration_file'}, ")\n";
  }
  else {
    my @options;
    push @options, $options_ref->{'configuration_file'};
    push @options, 'keyattr' => { service => "+name" };
    push @options, 'forcearray' => ['host', 'hostgroup', 'service'];
    push @options, 'suppressempty' => '';

    eval { $config_ref = XML::Simple::XMLin(@options); };
    if ($@) {
      print STDERR 'The follow error occurred while processing (', $options_ref->{'configuration_file'}, ")\n", $@, "\n";
    }
    else {
      # Find the default host and hostgroup configuration and apply it
      # to all others where necessary
      foreach my $type ('host', 'hostgroup') {
        # First do a little prep work to find the default entries
	# It would be nice to thing the default is always first or last or whatever, but...
	foreach my $conf_ref (@ { $config_ref->{$type} }) {
          my $key = $type . '_name';
          $key = 'group_name' if ($type eq 'hostgroup');

	  if ($conf_ref->{$key} eq 'default') {
	    $config_ref->{'default_' . $type} = $conf_ref;
	    last;
	  }
	}

	if (defined $config_ref->{'default_' . $type}) {
	  # Need to fill in the missing fields with the defaults
	  # I bet there is a better way to do this, but...
	  foreach my $conf_ref (@ { $config_ref->{$type} }) {
	    my $key = $type . '_name';
	    $key = 'group_name' if ($type eq 'hostgroup');

	    next if ($conf_ref->{$key} eq 'default');

	    foreach my $field (keys % { $config_ref->{'default_' . $type} }) {
	      $conf_ref->{$field} = $config_ref->{'default_' . $type}->{$field} if (!$conf_ref->{$field});
	    }
	  }
        }
        else {
	  print 'WARNING: Unable to locate (default) ', $type, ' configuration in (', $options_ref->{'configuration_file'}, ")!\n";
	}
      }

      # Services are easier because they are referenced by a hash
      if (defined $config_ref->{'service'}->{'default'}) {
        # Need to fill in the missing fields with the defaults
        # I bet there is a better way to do this, but...
        foreach my $service_ref (values % { $config_ref->{'service'} }) {
          next if ($service_ref->{'name'} eq 'default');

          foreach my $field (keys % { $config_ref->{'service'}->{'default'} }) {
            $service_ref->{$field} = $config_ref->{'service'}->{'default'}->{$field} if (!$service_ref->{$field});
          }
        }
      }
      else {
        print 'Warning: Unable to locate (default) service configuration in (', $options_ref->{'configuration_file'}, ")!\n";
      }
    }
  }

  return $config_ref;
}

sub load_results {
  my $options_ref = shift;

  my $results_ref = undef;
  if (!-e $options_ref->{'results_file'}) {
    print STDERR 'Unable to read (', $options_ref->{'results_file'}, ")\n";
  }
  else {
    my @options;
    push @options, $options_ref->{'results_file'};
    push @options, 'forcearray' => ['port'];

    eval { $results_ref = XML::Simple::XMLin(@options); };
    if ($@) {
      print STDERR 'The follow error occurred while processing (', $options_ref->{'results_file'}, ")\n", $@, "\n";
    }
  }

  return $results_ref;
}

# To keep from having to include a non-CPAN (I just haven't gotten around to it yet) 
# module in the distribution I pasted the process routine from my Tools::Template module
#
# Another significant difference is that Tools::Template uses <!-- --> as the delimiter
sub process {
  my $template = shift;
  my $data_ref = shift;
  my $skip_file_tags = shift;

  my ($file, $var, $var_tag, $if, $not, $if_not, $if_not_tag);

  if (!$skip_file_tags) {
    while ($template =~ /{--file_/) {
      ($if_not_tag) = $template =~ /{--file_(.*(?=_file--}))_file--}/;
      $var_tag = "file_$if_not_tag" . "_file";

      $if = $if_not = Tools::Template::process($if_not_tag, $data_ref);
      $not = '';
      if ($if_not =~ /!/) {
        ($if, $not) = split(/!/, $if_not);
      }

      #check for the if
      if (-e "$if") {
        $file = Tools::Template::read($if);
      }
      #otherwise get the not
      elsif (-e "$not") {
        $file = Tools::Template::read($not);
      }
      else {
        $file = '';
      }

      $template =~ s|{--$var_tag--}|$file|g;
    }
  }

  if ($template =~ /{--/) {
    foreach $var ( keys % { $data_ref } ) {
      $template =~ s/{--$var--}/$data_ref->{$var}/g if (defined $data_ref->{$var});
    }
  }

  return $template;
}

# Swiped this from Stats::Common
sub seconds2minutes {
  my $time_in_seconds = shift;

  my $minutes = int($time_in_seconds / 60);
  my $seconds = $time_in_seconds - ($minutes * 60);
  my $hours = int($minutes / 60);
  $minutes -= $hours * 60;

  return sprintf("%02d:%02d:%02d", $hours, $minutes, $seconds);
}

# Swiped from Tools::TimeDate
sub split_timestamp {
  my $time_ref = {};

  $time_ref->{'timestamp'} = shift;
  $time_ref->{'timestamp'} = time if (!$time_ref->{'timestamp'});

  ($time_ref->{'sec'},
   $time_ref->{'min'},
   $time_ref->{'hour'},
   $time_ref->{'mday'},
   $time_ref->{'mon'},
   $time_ref->{'year'},
   $time_ref->{'wday'},
   $time_ref->{'yday'},
   $time_ref->{'isdst'}) = localtime($time_ref->{'timestamp'});

  $time_ref->{'mon'} = sprintf('%02d', $time_ref->{'mon'} + 1);
  $time_ref->{'mday'} = sprintf('%02d', $time_ref->{'mday'});

  $time_ref->{'hour'} = sprintf('%02d', $time_ref->{'hour'});
  $time_ref->{'min'} = sprintf('%02d', $time_ref->{'min'});
  $time_ref->{'sec'} = sprintf('%02d', $time_ref->{'sec'});

  $time_ref->{'year'} += 1900;

  $time_ref->{'date'} = $time_ref->{'year'} . $time_ref->{'mon'} . $time_ref->{'mday'};
  $time_ref->{'time'} = join(':', $time_ref->{'hour'}, $time_ref->{'min'}, $time_ref->{'sec'});

  $time_ref->{'date_time'} = join('/', $time_ref->{'mon'}, 
                                     $time_ref->{'mday'},
                                     $time_ref->{'year'}) . ' ' . $time_ref->{'time'};

  return $time_ref;
}
1;
__END__

=head1 NAME

nmap2nagios.pl - Perl program to process nmap XML output into Nagios host/hostgroup/services entries

=head1 SYNOPSIS

  Note: I'm not going to go into the theory of using nmap.  Please read the nmap docs for that.

  ./nmap -sS -O -oX nmap.xml myserver.mydomain.com 

  ./nmap2nagios.pl -v -r nmap.xml -o new.cfg

  That's it.

  What this program attempts to do is make you life easier by building your hostgroup, host and service entries for you.

  It does this by parsing the nmap XML output.

  Here's a sample nmap command:

    nmap -sS -O -oX 192.168.100.1.xml 192.168.100.1

  Which generates this to STDOUT:

    Starting nmap V. 2.54BETA7 ( www.insecure.org/nmap/ )
    Interesting ports on myserver.mydomain.com (192.168.100.1):
    (The 1530 ports scanned but not shown below are in state: closed)
    Port       State       Service
    22/tcp     open        ssh
    25/tcp     open        smtp
    80/tcp     open        http
    3306/tcp   open        mysql

    TCP Sequence Prediction: Class=truly random
			     Difficulty=9999999 (Good luck!)
    Remote operating system guess: Linux 2.0.35-38

    Nmap run completed -- 1 IP address (1 host up) scanned in 1 second

  And is the XML which was generated:

    <?xml version="1.0" ?>
    <!-- nmap (V. 2.54BETA7) scan initiated Tue Feb 20 14:08:25 2001 as: nmap -sS -O -oX 207.15.160.27.xml 207.15.160.27 -->
    <nmaprun scanner="nmap" args="nmap -sS -O -oX 207.15.160.27.xml 207.15.160.27" start="982699705" version="2.54BETA7" xmloutputvers
    ion="1.0">
    <scaninfo type="syn" protocol="tcp" numservices="1534" services="1-1026,1030-1032,1058-1059,1067-1068,1080,1083-1084,1103,1109-111
    0,1112,1127,1155,1178,1212,1222,1234,1241,1248,1346-1381,1383-1552,1600,1650-1652,1661-1672,1723,1827,1986-2028,2030,2032-2035,203
    8,2040-2049,2064-2065,2067,2105-2106,2108,2111-2112,2120,2201,2232,2241,2301,2307,2401,2430-2433,2500-2501,2564,2600-2605,2627,263
    8,2766,2784,3000-3001,3005-3006,3049,3064,3086,3128,3141,3264,3306,3333,3389,3421,3455-3457,3462,3900,3984-3986,4008,4045,4132-413
    3,4144,4321,4333,4343,4444,4500,4557,4559,4672,5000-5002,5010-5011,5050,5145,5190-5193,5232,5236,5300-5305,5308,5432,5510,5520,553
    0,5540,5550,5631-5632,5680,5713-5717,5800-5801,5900-5902,5977-5979,5997-6009,6050,6105-6106,6110-6112,6141-6148,6558,6666-6668,696
    9,7000-7010,7100,7200-7201,7326,8007,8009,8080-8082,8888,8892,9090,9100,9535,9876,9991-9992,10005,10082-10083,11371,12345-12346,17
    007,18000,20005,22273,22289,22305,22321,22370,26208,27665,31337,32770-32780,32786-32787,43188,47557,54320,65301" />
    <verbose level="0" />
    <debugging level="0" />
    <host><status state="up" />
    <address addr="207.15.160.27" addrtype="ipv4" />
    <hostnames><hostname name="zaphod.ionictech.net" type="PTR" /></hostnames>
    <ports><extraports state="closed" count="1530" />
    <port protocol="tcp" portid="22"><state state="open" /><service name="ssh" method="table" conf="3" />
    </port>
    <port protocol="tcp" portid="25"><state state="open" /><service name="smtp" method="table" conf="3" />
    </port>
    <port protocol="tcp" portid="80"><state state="open" /><service name="http" method="table" conf="3" />
    </port>
    <port protocol="tcp" portid="3306"><state state="open" /><service name="mysql" method="table" conf="3" />
    </port>
    </ports>
    <tcpsequence index="9999999" class="truly random" difficulty="Good luck!" />
    <os><portused state="open" proto="tcp" portid="22" />
    <portused state="closed" proto="tcp" portid="1" />
    <osmatch name="Linux 2.0.35-38" accuracy="100" />
    </os></host>
    <runstats><finished time="982699706" /><hosts up="1" down="0" total="1" />
    <!-- Nmap run completed at Tue Feb 20 14:08:26 2001; 1 IP address (1 host up) scanned in 1 second -->
    </runstats></nmaprun>

  Big nasty mess, huh!  Well that's ok because it's not meant for you to read anyway!
  
  Here's what the generated file looks like:

    host[myserver.mydomain.com]=myserver.mydomain.com;192.168.100.1;;check-host-alive;3;5;24x7;1;1;1;
    service[myserver.mydomain.com]=FTP;24x7;3;5;1;unix-admins;5;24x7;1;1;0;;check_ftp
    service[myserver.mydomain.com]=ssh;24x7;3;5;1;unix-admins;5;24x7;1;1;0;;check_ssh
    service[myserver.mydomain.com]=Telnet;24x7;3;5;1;unix-admins;5;24x7;1;1;0;;check_telnet
    service[myserver.mydomain.com]=SMTP;24x7;3;5;1;unix-admins;5;24x7;1;1;0;;check_smtp
    # No known service check for 'sunrpc' on port '111'
    # No known service check for 'printer' on port '515'
    # No known service check for 'X11' on port '6000'

  If you perfer not to generate service entries for 'unknown services' you can use the -i option.
  Also you can disable certain known services in the configuration file by added a <disabled>true</disabled> TAG
  to it's configuration.

=head1 TODO

Host specific settings similar to hostgroup and service settings.  (Just didn't have the time).
Caching of previous runs which would allow for merging new and previous scans into a new output file.
Parseing of host/hostgroup/service entries from existing hosts.cfg for merging with new/previous scans.

=head1 AUTHOR

Todd A. Green <slaribartfast@awardsforfjords.com>

=head1 COPYRIGHT

Copyright (c) 2000-2002 Todd A. Green.  All rights reserved.  This program is
free software; you can redistribute it and/or modify it under the same terms as
Perl itself.  If you do modify it though please let the author know cause he
likes to hear that someone found his work useful. :)

Nagios is a registered trademark of Ethan Galstad.

=head1 DISCLAIMER

It you do something stupid with this software, like wipe out your entire 500 host, 1500 service Netsaint/Nagios configuration,
it's your own fault.  Backups, Backups, Backups, Backups.
Be that as it may, I have beaten the crap out of the code, but I'm sure there is something goofy it 
will do so use it at your own risk.  Please send any bug reports or suggestions to the author.

=head1 SEE ALSO

Nagios @ http://www.nagios.org
nmap @ http://www.insecure.org/nmap/index.html 

=head1 NAGIOS

Nagios and the Nagios logo are registered trademarks of Ethan Galstad. 

=cut
