#!/usr/bin/perl
#
# Josh 
# 07/14/06
#
# 	Use the output of snmpget hrSystemDate.0 to create warning/critical
# signals to Nagios.
#
# LICENSE
# 	Licensed under the GPL w/ absolutely no warranty of any kind =)
#
# BUGS & PATCHES
# 	mailto: joshyost@gmail.com
# 
# VERSIONS
# 1.0.4
# 	- using utils.pm (took a while...); re-directing STDERR to STDOUT on syscalls
# 1.0.3
# 	- re-wrote some code for cleanliness & (hopefully) efficiency
# 	- changed usage handling

use warnings;
use strict;
use Getopt::Std;
use Time::Local;
use lib "/usr/nagios/libexec";
use utils qw ( %ERRORS );

$Getopt::Std::STANDARD_HELP_VERSION = 1;    # So getopts doesn't barf
############################ Functions ###################################
sub usage{
	print "Usage: check_systime.pl [-dhV][-v <num>] -H host -C string\n";
	exit $ERRORS{'WARNING'};
}

sub HELP_MESSAGE{
	my $msg;
	$msg = "\n\tCheck the system time on a host. This plugin assumes\n"
	    . "that you have snmpget on your system & in your path somewhere.\n"
	    . "It gives a critical state if the target has drifted more than 10\n"
	    . "times your allowed variance.  It gives a warning state if it is\n"
	    . "outside of the allowed time variance (but less than 10 times).\n"
	    . "\tIt is currently using SNMP v. 2c - you'll need to edit the\n"
	    . "snmpget call manually (for right now) if you want to change it.\n"
	    . "\tIf it can't find utils.pm, then you'll need to change the path\n"
	    . "in the 'use lib' statement to your nagios plugin location.\n"		
	    . "\n\t-h,--help\tshow this help"
	    . "\n\t-d\t\toutput debugging info"
	    . "\n\t-C <str>\tyour community SNMP string"
	    . "\n\t-H <str>\tyour target host"
	    . "\n\t-v <num>\tthe amount of variance allowed (in seconds)"
	    . "\n\t-V,--version\tshow current version\n";
	print $msg;
	exit $ERRORS{'OK'};
}

sub VERSION_MESSAGE{
	my $version = '1.0.4';
	print "check_systime.pl\t\tv. $version\n";
	exit $ERRORS{'OK'};
}

my %opts;
getopts("dhVH:C:v:", \%opts);
my $verbose = defined ($opts{d});
my $host = $opts{H} if defined ($opts{H});
my $string = $opts{C} if defined ($opts{C});		# SNMP string
my $diff = ($opts{v} || 120);				# number of seconds

HELP_MESSAGE() if defined($opts{h});
VERSION_MESSAGE() if defined($opts{V});
usage() if (!defined($host) || !defined($string));

my $output = `snmpget -v2c -c $string $host hrSystemDate.0 2>&1`;
my $state = $?;

print "output:|$output|,state:$state\n" if $verbose;

if ($state == $ERRORS{'CRITICAL'} || $state > $ERRORS{'CRITICAL'}){
	print $output;
	exit $ERRORS{'CRITICAL'};
}
elsif ($state == $ERRORS{'WARNING'}){
	print $output;
	exit $ERRORS{'WARNING'};
}
elsif ($state == 0){
	# STRING: 2006-7-19,18:16:28.0,-5:0 
	if ($output =~ /STRING\s*:\s*(\d\d\d\d)-(\d+)-(\d+),(\d+):(\d+):(\d{1,2})/){
	
	  my ($time, $sys_time) = (timelocal($6,$5,$4,$3,$2-1,$1-1900),time);
	  my $localtime = sprintf "%02d-%02d-%d, %02d:%02d:%02d",$2,$3,$1,$4,$5,$6;
	  my $abs_time  = abs($time-$sys_time);
	  
	  if ($verbose){
		print "sec:\t$6\nmin:\t$5\nhr:\t$4\nday:\t$3\nmon:\t",$2-1,"\nyr:\t",$1-1900,"\n";
		print "host epoch : $time\tsystem epoch : $sys_time\n";
		print "actual diff: $abs_time sec\tallowed diff: $diff sec\n";
		print "--\n";
	  } 
	
	  # Test epoch times
	  if ($abs_time > ($diff*10)){
		print "CRITICAL - System time is way off ($localtime).\n";
		exit $ERRORS{'CRITICAL'};
	  }
	  elsif ($abs_time > $diff){
		print "WARNING - System time is off by at least $diff sec ($localtime).\n";
		exit $ERRORS{'WARNING'};
	  }
	  else{
		print "System Time OK - $localtime\n";
		exit $ERRORS{'OK'};
	  }
	}
	# No match on the RE
	else{
		print "UNKNOWN - snmpget returned unknown output.\n";
		exit $ERRORS{'UNKNOWN'};
	}
}
elsif ($state < 0){
	print $output;
	exit $ERRORS{'UNKNOWN'};
}
