#!/usr/bin/perl -w
#
# $Id$
#
# check_inivalue.pl - nagios plugin to check the value of a defined key in a defined section of an ini file.
#
# Copyright (C) 2009 Rainer Brinkmller <rainer.brinkmoeller@googlemail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# you should have received a copy of the GNU General Public License
# along with this program (or with Nagios);  if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA
#

use strict;
use Getopt::Long;
use File::Basename;
use Config::IniFiles;

use vars qw(
  $opt_infile
  $opt_section
  $opt_key
  $opt_ok
  $opt_warning
  $opt_critical
  $opt_WarnFloat
  $opt_CritFloat
  $opt_label
  $opt_help

  $PROGNAME
  $VERSION
);

use subs qw(
	myexit
	print_usage
);

$PROGNAME = basename($0);
$VERSION  = '0.2.3';

# default values
$opt_infile    = "";
$opt_section   = "";
$opt_key       = "";
$opt_ok        = "";
$opt_warning   = "";
$opt_critical  = "";
$opt_WarnFloat = "";
$opt_CritFloat = "";
$opt_label  = "";

# get Options
Getopt::Long::Configure('bundling');
GetOptions(
	"infile=s"     => \$opt_infile,
	"i=s"          => \$opt_infile,
	"section=s"    => \$opt_section,
	"s=s"          => \$opt_section,
	"key=s"        => \$opt_key,
	"k=s"          => \$opt_key,
	"ok:s"         => \$opt_ok,
	"o:s"          => \$opt_ok,
	"warning:s"    => \$opt_warning,
	"w:s"          => \$opt_warning,
	"critical:s"   => \$opt_critical,
	"c:s"          => \$opt_critical,
	"WarnFloat:f"  => \$opt_WarnFloat,
	"W:f"          => \$opt_WarnFloat,
	"CritFloat:f"  => \$opt_CritFloat,
	"C:f"          => \$opt_CritFloat,
	"label:s"      => \$opt_label,
	"l:s"          => \$opt_label,
	"help"         => \$opt_help,
	"h"            => \$opt_help,
);

#Verify Options
if ( !$opt_infile || !$opt_section || !$opt_key || $opt_help ) {
	if ( !$opt_help ) {
		myexit( 'UNKNOWN', 'check_inivalues.pl Version ' . $VERSION . ' - no or not enough options!' );
	}
	print_usage();
}

my $status    = "";
my $message   = "";

if (!-e $opt_infile){
	myexit( 'UNKNOWN', 'Ini file ' . $opt_infile . ' not found' );
}

my $infile = new Config::IniFiles( -file => $opt_infile );
my $value = $infile->val( $opt_section, $opt_key );

if ( !$value ) {
	myexit( 'UNKNOWN', 'NOTICE - No value for key ' . $opt_key . ' in section ' . $opt_section . '!' );
}
elsif ( $opt_warning && !$opt_ok && !$opt_critical ) {
	if ( $value eq $opt_warning ) {
		myexit( 'WARNING', $opt_key . '=' . $value );
	}
	else {
		myexit( 'OK', $opt_key . '=' . $value );
	}
}
elsif ( $opt_critical && !$opt_ok && !$opt_warning ) {
	if ( $value eq $opt_critical ) {
		myexit( 'CRITICAL', $opt_key . '=' . $value );
	}
	else {
		myexit( 'OK', $opt_key . '=' . $value );
	}
}
elsif ( $opt_ok && !$opt_warning && !$opt_critical ) {
	if ( $value eq $opt_ok ) {
		myexit( 'OK', $opt_key . '=' . $value );
	}
	else {
		myexit( 'CRITICAL', $opt_key . '=' . $value );
	}
}
elsif ( $opt_ok && $opt_warning && !$opt_critical ) {
	if ( $opt_ok eq $opt_warning ) {
		myexit( 'UNKNOWN', 'NOTICE - Ok and Warning options can not be equal!' );
	}
	elsif ( $value eq $opt_ok ) {
		myexit( 'OK', $opt_key . '=' . $value );
	}
	elsif ( $value eq $opt_warning ) {
		myexit( 'WARNING', $opt_key . '=' . $value );
	}
	else {
		myexit( 'CRITICAL', $opt_key . '=' . $value );
	}
}
elsif ( $opt_ok && !$opt_warning && $opt_critical ) {
	if ( $opt_ok eq $opt_warning ) {
		myexit( 'UNKNOWN', 'NOTICE - Ok and Critical options can not be equal!' );
	}
	elsif ( $value eq $opt_ok ) {
		myexit( 'OK', $opt_key . '=' . $value );
	}
	elsif ( $value eq $opt_critical ) {
		myexit( 'CRITICAL', $opt_key . '=' . $value );
	}
	else {
		myexit( 'WARNING', $opt_key . '=' . $value );
	}
}
elsif ( $opt_ok && $opt_warning && $opt_critical ) {
	if ( $opt_ok eq $opt_warning ) {
		myexit( 'UNKNOWN', 'NOTICE - Ok and Critical options can not be equal!' );
	}
	elsif ( $value eq $opt_ok ) {
		myexit( 'OK', $opt_key . '=' . $value );
	}
	elsif ( $value eq $opt_critical ) {
		myexit( 'CRITICAL', $opt_key . '=' . $value );
	}
	elsif ( $value eq $opt_warning ) {
		myexit( 'WARNING', $opt_key . '=' . $value );
	}
	else {
		myexit( 'UNKNOWN', $opt_key . '=' . $value );
	}
}
elsif ( $opt_warning || $opt_critical ) {
	if ( $opt_warning && $opt_critical && $opt_warning eq $opt_critical ) {
		myexit( 'UNKNOWN', 'NOTICE - Warning and Critical options can not be equal!' );
	}
	elsif ( !$opt_critical && $opt_warning && $value eq $opt_warning ) {
		myexit( 'WARNING', $opt_key . '=' . $value );
	}
	else {
		myexit( 'CRITICAL', $opt_key . '=' . $value );
	}
}
elsif ( $opt_WarnFloat || $opt_CritFloat ) {

	my $perfdatout = "";

	if ( $opt_label ) {
		if ( $opt_WarnFloat && $opt_CritFloat ) {
			$perfdatout = $opt_label . '=' . $value . ";" . $opt_WarnFloat . ";" . $opt_CritFloat . ";" . ";";
		}
		elsif ( $opt_WarnFloat && !$opt_CritFloat ) {
			$perfdatout = $opt_label . '=' . $value . ";" . $opt_WarnFloat . ";" . ";" . ";";
		}
		elsif ( !$opt_WarnFloat && $opt_CritFloat ) {
			$perfdatout = $opt_label . '=' . $value . ";" . ";" . $opt_CritFloat . ";" . ";";
		}
		else {
			$perfdatout = $opt_label . '=' . $value . ";" . ";" . ";" . ";";
		}
	}
	else {
		if ( $opt_WarnFloat && $opt_CritFloat ) {
			$perfdatout = $value . ";" . $opt_WarnFloat . ";" . $opt_CritFloat . ";" . ";";
		}
		elsif ( $opt_WarnFloat && !$opt_CritFloat ) {
			$perfdatout = $value . ";" . $opt_WarnFloat . ";" . ";" . ";";
		}
		elsif ( !$opt_WarnFloat && $opt_CritFloat ) {
			$perfdatout = $value . ";" . ";" . $opt_CritFloat . ";" . ";";
		}
		else {
			$perfdatout = $value . ";" . ";" . ";" . ";";
		}
	}

	if ( $opt_WarnFloat && $opt_CritFloat && $opt_WarnFloat==$opt_CritFloat ) {
		myexit( 'UNKNOWN', 'NOTICE - Warning and Critical threshold can not be equal!' );
	}
	elsif ( $opt_WarnFloat && $opt_CritFloat && $opt_CritFloat>$opt_WarnFloat ) {
		if ( $value>$opt_CritFloat ) {
			myexit( 'CRITICAL', $opt_key . '=' . $value, $perfdatout );
		}
		elsif ( $value>=$opt_WarnFloat && $value<$opt_CritFloat ) {
			myexit( 'WARNING', $opt_key . '=' . $value, $perfdatout );
		}
		else {
			myexit( 'OK', $opt_key . '=' . $value, $perfdatout );
		}
	}
	elsif ( $opt_WarnFloat && $opt_CritFloat && $opt_CritFloat<$opt_WarnFloat ) {
		if ( $value<=$opt_CritFloat ) {
			myexit( 'CRITICAL', $opt_key . '=' . $value, $perfdatout );
		}
		elsif ( $value>$opt_CritFloat && $value<=$opt_WarnFloat ) {
			myexit( 'WARNING', $opt_key . '=' . $value, $perfdatout );
		}
		else {
			myexit( 'OK', $opt_key . '=' . $value, $perfdatout );
		}
	}
	elsif ( !$opt_CritFloat && $opt_WarnFloat ) {
		if ( $value>=$opt_WarnFloat ) {
			myexit( 'WARNING', $opt_key . '=' . $value, $perfdatout );
		}
		else {
			myexit( 'OK', $opt_key . '=' . $value, $perfdatout );
		}
	}
	elsif ( !$opt_WarnFloat && $opt_CritFloat  ) {
		if ( $value>=$opt_CritFloat ) {
			myexit( 'CRITICAL', $opt_key . '=' . $value, $perfdatout );
		}
		else {
			myexit( 'OK', $opt_key . '=' . $value, $perfdatout );
		}
	}
	else {
		myexit( 'UNKNOWN', $opt_key . '=' . $perfdatout );
	}
}
else {
	print_usage();
}

sub myexit {
	my $time;
	my $date;

	my ( $status, $message, $perfdata ) = @_;

	my %STATUS_CODE = ( 'OK' => '0', 'WARNING' => '1', 'CRITICAL' => '2', 'UNKNOWN' => '3' );

	my $out = undef;
	$out .= sprintf('%1$s: %2$s', $status, $message);
	$out .= sprintf('|%1$s', $perfdata) if ($perfdata);
	$out .= chr(10);

	print $out;

	exit $STATUS_CODE{$status};
}

sub print_usage {
	print <<EOU;
    check_inivalues.pl - nagios plugin version $VERSION

    Copyright (c) 2009 Rainer Brinkmller

    This plugin checks the value of a defined key in a defined section of an ini file.
    It comes AS IS with ABSOLUTELY NO WARRANTY.
    You may redistribute copies of the plugin under the terms of the GNU General Public License.

    Usage: $PROGNAME [-i <infile>] [-s <section>] [-k <key>] [-o <OK_Value>] [-w or -W <WARNING_Value>] [-c or -C <CRTICAL_Value> ] [-l <LABEL>]

    Options:

    -i --infile STRING
        Path and file name to the ini/config file.
    -s --section STRING
        String to identify the ini/config Section.
    -k --key STRING
        String to identify the key of the value.
    -o --ok STRING
        Value for OK.
    -w --warning STRING
        Value for WARNING.
    -c --critical STRING
        Value for CRITICAL.
    -W --WarnFloat FLOAT
        Threshold for WARNING.
    -C --CritFloat FLOAT
        Threshold for CRITICAL.
    -l --label STRING
        String for the label of the performance data.
    -h --help
        Use this option to show this help info.

    Attention:

    -i, -s and -k are necessary!

    The options -o and/or -w and/or -c are necessary if you want to check for a string value.
    If only -w or -c are defined all other values end in OK State.
    If -o and/or -w are defined (not -c) all values other then in this defined options end in CRITICAL State.
    If -o and -c are defined (not -w) all values other then in this defined options end in WARNING State.
    If -o and -w and -c are defined all values other then in this defined options end in UNKNOWN State.
    (While checking for a string the options --WarnFloat and --CritFloat will be ignored)!

    The options -W and/or -C are necessary if you want to check for a float value.
    If only -W is defined, all values higher or equal will end in WARNING State.
    If only -C is defined, all values higher or equal will end in CRITICAL State.
    If -W and -C are defined and the value is between, it end in WARNING State.
    If the option for -W is higher then for -C :
    A value higher then the option defined by -W end in OK State.
    Is the value less or equal the option defined by -C it end in CRITICAL State.
    (Do not use -o, -w and/or -c if you want to check for float values)!

EOU

	exit;
}
