/* vim: set ts=2 sw=2: */
/****************************************************************************
 * This file is part of nscam.                                              *
 *                                                                          *
 * nscam is free software: you can redistribute it and/or modify            *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation, either version 3 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * nscam is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with Foobar.  If not, see <http://www.gnu.org/licenses/>.          *
 *                                                                          *
 ****************************************************************************
 * filename : nscam_send.c                                                  *
 *                                                                          *
 * id: $Id$
 *                                                                          *
 * date:    : 2011-02-11                                                    *
 * purpose  : Tool to send nsca multicast messages.                         *
 *                                                                          *
 * author   : Peter Meszaros                                                *
 *            e-mail: hauptadler@gmail.com                                  *
 *                                                                          *
 ****************************************************************************/

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <net/if.h>
#include <libgen.h>
#include <ctype.h>
#include <err.h>

#define MAXLINE 4096

int verbose;
char *version = "1.0, written by hauptadler";
char *progname;

void sighandler(int sig)
{
  printf("Timeout, so quit...\n");
  exit(EXIT_SUCCESS);
}

void Usage(char *progname)
{
  fprintf(stderr, "Usage: %s [-h] [-V] [-v] [-a 1.2.3.4] # Version: %s\n", progname, version);
  fprintf(stderr, "\t-h	help\n");
  fprintf(stderr, "\t-a	multicast address\n");
  fprintf(stderr, "\t-V	version\n");
  fprintf(stderr, "\t-v	verbose\n");

  fprintf(stderr, "\n\
  Service Checks:\n\
  <host_name>[tab]<svc_description>[tab]<return_code>[tab]<plugin_output>[newline]\n\n\
  Host Checks:\n\
  <host_name>[tab]<return_code>[tab]<plugin_output>[newline]\n\n");
}

int main(int argc, char **argv)
{
  int opt;
	int fd;
	int	n, on;
	struct sockaddr_in	sa;
  struct servent *se;
	struct ip_mreq mreq;
  char *addr = "224.0.2.1";
  short port = 5669;
  char *service = "nscadm";
	char line[MAXLINE+1];
	int len;

  progname = basename(argv[0]);

  while ((opt = getopt(argc, argv, "hVa:v")) != -1) {
 		switch (opt) {
      case 'h':
      case 'V':
        Usage(progname);
 				exit(EXIT_SUCCESS);
        break;
      case 'v':
        verbose = 1;
        break;
      case 'a':
        addr = optarg;
        break;
 		  default: /* '?' */
        Usage(progname);
 				exit(EXIT_FAILURE);
    }
  }

  fprintf(stderr, "This progam sends multicast message to %s address,\n", addr);
  fprintf(stderr, "service: %s or port %d if service cannot be found by name.\n", service, port); 
  fprintf(stderr, "  Verbose: %s\n", verbose ? "on" : "off");
  fprintf(stderr, "  Address: %s\n", addr);

  if ((se = getservbyname(service, "udp"))) {
    port = se->s_port;
    fprintf(stderr, "Use port %d from service '%s'\n", ntohs(port), service);
  } else {
    port = htons(port);
    fprintf(stderr, "Use port %d, service '%s' not found\n", ntohs(port), service);
  }

  /* send */
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    err(EXIT_FAILURE, "socket()");

  on = 1;
	if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on))) < 0)
    err(EXIT_FAILURE, "setsockopt() SO_REUSEADDR");

  on = 64;
	if ((setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL, &on, sizeof(on))) < 0)
    err(EXIT_FAILURE, "setsockopt() IP_MULTICAST_TTL");

  memset(&sa, 0, sizeof(struct sockaddr_in));
  sa.sin_family = AF_INET;
  sa.sin_addr.s_addr = inet_addr(addr);
  sa.sin_port = port;

  if ((bind(fd, (struct sockaddr *)&sa, sizeof(struct sockaddr)) < 0))
    err(EXIT_FAILURE, "bind()");

	memcpy(&mreq.imr_multiaddr, &(sa.sin_addr), sizeof(struct in_addr));
	mreq.imr_interface.s_addr = htonl(INADDR_ANY);
	if (setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) < 0) {
    if (errno == EINVAL || errno == ENODEV) {
      fprintf(stderr, "Probably there is no default route in the routing table!\n");
      fprintf(stderr, "Multicast messages cannot be sent without that.\n");
    }
    err(EXIT_FAILURE, "setsockopt() IP_ADD_MEMBERSHIP");
  }

  on = 1;
	if ((setsockopt(fd, IPPROTO_IP, IP_MULTICAST_LOOP, &on, sizeof(on))) < 0)
    err(EXIT_FAILURE, "setsockopt() IP_MULTICAST_LOOP");

	while(fgets(line, MAXLINE, stdin)) {
    len = strlen(line);
		if ((n = sendto(fd, line, len, 0, (struct sockaddr *)&sa, sizeof(struct sockaddr))) < 0)
			err(EXIT_FAILURE, "sendto()");

		if (verbose) 
			fprintf(stderr, "message sent to %s:%d: %s",
					inet_ntoa(sa.sin_addr), htons(sa.sin_port), line);
	}

  return EXIT_SUCCESS;
}

