#!/usr/bin/perl 
###################################################################################################
# Program Name: ovo_to_nagios.pl
# Input       : existing ovo template 
# Output      : various nagios templates
# Dependencies: OVO backups 
# Description : This tool will take an OVO template or template group and create the associated Nagios
#               pieces.    
# Usage       : see USAGE()
# Contact     : Ryan Ash <ryan(dot)c(dot)ash(dot)lu4w(at)statefarm(dot)com>
# Notes       : This isn't intended to gve you a 100% complete template.  It just does the leg work for you
#               and builds out conditions.  You will have to complete the work and check all the logic
# Keywords    : NAGIOS OVO MIGRATION_TOOLS MONITORING
###################################################################################################

#Please read the attached README 
use warnings;
use strict;
use File::Copy;
use Getopt::Std;
use File::Path qw (mkpath);

use vars qw($TG_file %opts %info);
my @valid_types = ("LOGFILE","MONITOR","SCHED","OPCMSG");

getopts ('b:agx:s:d:ho:i:t:z:', \%opts) or USAGE();
$opts{d} = $opts{d} ? $opts{d} : 0;
USAGE() if (($opts{h}) || (!$opts{t}));

if ($opts{a}) {
  #INITIAL ASK MODE QUESTIONS
  print "\nWelcome to the Nagios template developer\n\nThe purpose of this tool is to help you build a framework for a Nagios template according to an existing OVO template.  This will not build a template that is 100% ready to deploy.\n\nDo you understand this disclaimer? (y or n) ";
  exit if (<STDIN> !~ /y/i);

  ask_mode_asker("Would you like to prepend all service names with your input template? (value=$opts{x})", \$info{user}{prepend_desc},"yesno","y"); 
  ask_mode_asker("Would you like to use a check_command for your passive services?", \$info{user}{check_command},"yesno","n"); 
  ask_mode_asker("Do check_logfiles payload need to be truncated?", \$info{user}{truncate},"yesno","n"); 
  if ($info{user}{truncate} eq "y") {
    ask_mode_asker("What is maxlength value for check_logfiles?", \$info{user}{truncate_size},"int","1024"); 
  }
  ask_mode_asker("Do you want to trim large service descriptions?", \$info{user}{trim_desc},"yesno","n"); 
  if ($info{user}{trim_desc} eq "y") {
    print "REMINDER: This will be done all the time even if you approve/edit each description!";
    ask_mode_asker("How many characters should be allowed for the service description?", \$info{user}{trim_size},"int","75"); 
  }
  ask_mode_asker("Do you want to be prompted to approve/edit every service description?", \$info{user}{approve_desc},"yesno","n"); 
  if ($info{user}{approve_desc} eq "y") {
    ask_mode_asker("Do you want us to scrub the desc before it is shown to you?", \$info{user}{prescrub_desc},"yesno","y"); 
  }
  ask_mode_asker("Do you want the nsca to be configured in the check_logfile cfg file?", \$info{user}{nsca_in_cfg},"yesno","n"); 
  ask_mode_asker("Include OVO hints? These show OVO matching rules, etc, helpful in development.", \$info{user}{ovo_hints},"yesno","y"); 
  ask_mode_asker("Do you want to set a default max_check_attempts?", \$info{user}{set_max_checks},"yesno","n"); 
  if ($info{user}{set_max_checks} eq "y") {
    ask_mode_asker("What value do you want to use for max_check_attempts?", \$info{user}{max_checks},"int","1"); 
  }
  ask_mode_asker("Is there reoccuring syntax in the description you want to remove?", \$info{user}{remove_from_desc},"yesno","n"); 
  if ($info{user}{remove_from_desc} eq "y") {
    ask_mode_asker("Enter the string to removed.", \$info{user}{remove_string_desc},"string","Chuck Norris is wimp"); 
  }

}else{
  foreach ("ovo_hints","nsca_in_cfg","prepend_desc","check_command","truncate","trim_desc","approve_desc","set_max_checks","remove_string_desc","remove_from_desc") {
    $info{user}{$_} = "null";
  }
}

$info{search_dir} = "/tmp/mypathtomyovobackups/";
$info{nrpe_handler_cnt} = 1;
@{$info{required_scripts}} = "";
$TG_file = $info{search_dir} . "TEMPLGROUP/templgroup.dat";

validate_opts();
obtain_input_wrapper();
build_nagios_conf(); #current code insists this comes before others
build_plugin_conf_linux() if ($opts{z} eq "LINUX");
build_plugin_conf_win() if ($opts{z} eq "WIN");
build_cron();

######################################################################################
#Build the server side service configs in a single file
sub build_nagios_conf {
  my $nagios_conf = "$opts{o}/$opts{x}.cfg"; 
  #`echo "" > $nagios_conf`; 
  open (NAGOUT,">$nagios_conf") or die "couldn't open PLUGOUT : $!\n"; 

  #------------------------------------------------------------------------
  #  My template
  #  define service {
  #    service_description $info{msg}{$type}{$subcurrent}{desc} (MODIFIED TO REMOVE \s)
  #    display_name $info{msg}{$type}{$subcurrent}{desc} 
  #    servicegroups $opts{x}_service_group
  #    hostgroup_name $opts{x}
  #    use <template> (i.e generic-passive-service-noperf
  #    notes <ci>:<object>:<app>:<ki>:<msggrp>  (essential for our routing logic
  #  }
  #
  #  If you would like to use another "template" you need to modify the code below to print yours as needed. I tried to incorporate 
  #  mind reading into this script but it proved difficult. All data needed <i>should</i> be available... 
  #
  #------------------------------------------------------------------------

  unless ($opts{g}) {
    print NAGOUT "\ndefine servicegroup{\n\tservicegroup_name $opts{x}_service_group\n\talias $info{master_desc}\n\t}\n";
  }

  foreach my $type (keys %{$info{msg}}) {
    foreach my $subcurrent (keys %{$info{msg}{$type}}) {
      print "DEBUG-build_nagios_conf : working $type '$subcurrent'}\n" if ($opts{d});
      if ($opts{d} > 2) {
        foreach my $x (keys %{$info{msg}{$type}{$subcurrent}}) {
          print "\tSUPER-DEBUG: key=$x value=$info{msg}{$type}{$subcurrent}{$x}\n";
        }
      }

      #Inheritance - assume higher level severity if necessary
      if ($info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{severity} && !$info{msg}{$type}{$subcurrent}{severity}) {
        $info{msg}{$type}{$subcurrent}{severity} = $info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{severity};
      }

      unless ($info{msg}{$type}{$subcurrent}{desc} &&
		  		$info{msg}{$type}{$subcurrent}{severity}) {
        print "ERROR-build_nagios_conf : invalid condition '$subcurrent'\n";
        next;
      }

      # handle the same service name for multiple conditions 
      my $ucount = 2;
      while (exists $info{unique}{$info{msg}{$type}{$subcurrent}{descnospace}}) {
        print "ERROR: $info{msg}{$type}{$subcurrent}{descnospace} already existed, appending junk to service desc to maintain uniqueness\n";
        $info{msg}{$type}{$subcurrent}{descnospace} .= "_$ucount";
        $ucount++;
      } 
      $info{unique}{$info{msg}{$type}{$subcurrent}{descnospace}} = 1;

      print NAGOUT "\ndefine service {\n"; 
      printf NAGOUT ("\t\%-25s\%-25s\n", "service_description", $info{msg}{$type}{$subcurrent}{descnospace});
      printf NAGOUT ("\t\%-25s\%-25s\n", "display_name", $info{msg}{$type}{$subcurrent}{desc});
      printf NAGOUT ("\t\%-25s\%-25s\n", "servicegroups", "$opts{x}_service_group") unless ($opts{g});
      printf NAGOUT ("\t\%-25s\%-25s\n", "hostgroup_name", "$opts{x}");
      #printf NAGOUT ("\t\%-25s\%-25s\n", "check_command", "check_nrpe_arg!check_logfiles_arg!$info{msg}{type}{$subcurrent}{descnospace}");
      printf NAGOUT ("\t\%-25s\%-25s\n", "check_command", "your_check_command") if ($info{user}{check_command} =~ /y/i);
      printf NAGOUT ("\t\%-25s\%-25s\n", "use", "generic-passive-service-noperf");
      if ($info{user}{set_max_checks} eq "y") {
        printf NAGOUT ("\t\%-25s\%-25s\n", "max_check_attempts", $info{user}{max_checks});
      }else{
        printf NAGOUT ("\t\%-25s\%-25s\n", "max_check_attempts", $info{dup}{$type}{$subcurrent}{counter}) if ($info{dup}{$type}{$subcurrent}{counter});
      }

      printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ALERT_TEXT", $info{msg}{$type}{$subcurrent}{text_clean}) if ($info{msg}{$type}{$subcurrent}{text_clean});

      if ($info{msg}{$type}{$subcurrent}{_CUSTOM_ARG_FETCH}) {
        if ($info{msg}{$type}{$subcurrent}{condition}{postmatch_arg}) {
          printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ARG_FETCH", $info{msg}{$type}{$subcurrent}{condition}{postmatch_arg});
        }else{
          if ( ${$info{msg}{$type}{$subcurrent}{condition}{arg}}[0]) {
            printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ARG_FETCH", ${$info{msg}{$type}{$subcurrent}{condition}{arg}}[0]);
          }else{
            print "ERROR-build_nagios_conf: _CUSTOM_ARG_FETCH arg not found for $info{msg}{$type}{$subcurrent}{desc}\n" if ($opts{d}); 
          }
        }
      }

      if ($info{user}{ovo_hints} eq "y") {
        printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_TEMPLATE", "$type $info{msg}{$type}{$subcurrent}{template}");
        printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_AUTOACTION", $info{msg}{$type}{$subcurrent}{autoactionfull}) if ($info{msg}{$type}{$subcurrent}{autoactionfull});
       
        #NOTE: if we find that conditions have multiple "TEXT" conditions we will need to loop through this array here to show them all
        #printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_MATCH_0", ${$info{msg}{$type}{$subcurrent}{condition}{text_orig}}[0]) if (($info{msg}{$type}{$subcurrent}{_CUSTOM_ARG_FETCH}) || ($info{msg}{$type}{$subcurrent}{autoaction} =~ /\<.+\>/));
        printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_MATCH_0", ${$info{msg}{$type}{$subcurrent}{condition}{text_orig}}[0]) if (${$info{msg}{$type}{$subcurrent}{condition}{text_orig}}[0]); 
      }
     
      if ($info{msg}{$type}{$subcurrent}{autoactionfull}) {
        #AUTOACTION FOUND
        my $options = "";
        my $server_hackme = $info{msg}{$type}{$subcurrent}{autoaction};
        my $client_hackme = $info{msg}{$type}{$subcurrent}{autoaction};
        my $alias_params = "";
        if ($info{msg}{$type}{$subcurrent}{autoaction} =~ /(\<\S*?\>)/) {
          #--------------------------------------------
          # code to build the script with params in it
          #--------------------------------------------
          my $c = 1;
          foreach my $word (split /\s+/, $info{msg}{$type}{$subcurrent}{autoaction}){
            if ($word =~ /(\<\S*?\>)/) {
              my $x = $1;
              print "DEBUG-build_nagios_conf : replacing $1 with --SF_ARG$c--\n" if ($opts{d} > 2);
              $server_hackme =~ s/$x/--SF_ARG$c--/;
              $client_hackme =~ s/$x/\$ARG$c\$/;
              $alias_params .= "--SF_ARG$c-- ";
              $c++;
            }
          }
        }

        if ($info{msg}{$type}{$subcurrent}{autoaction_actionnode}) {
          if ($info{msg}{$type}{$subcurrent}{autoaction_actionnode} =~ /MGMTSV/) {
            $options .= "server,";
            printf NAGOUT ("\t\%-25s\%-25s\n", "event_handler", "svc_autoaction_wrapper");
            printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_EVENT_PARAMS", $server_hackme);
          }
        }
        unless ($options =~ /server/){
          $options .= "client,";
          #----------------------------------------------
          # For commands that run on the client you will need an NRPE handler entry, this helps with that
          # use a hash so we only have one entry in handlers section for each alias
          #----------------------------------------------
          print "DEBUG-build_nagios_conf : working nrpe alias for $info{nrpe_handler_cnt}\n" if ($opts{d} > 2);
          unless (exists $info{nrpe_handler}{$client_hackme}) {
            $info{nrpe_handler}{$client_hackme} = "nrpe_cmd_alias_$info{nrpe_handler_cnt}";
            $info{nrpe_handler_cnt}++;
          }
          $alias_params =~ s/<\$MSG_NODE_NAME>/--SF_HOSTNAME--/g;
          printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_EVENT_PARAMS", "$info{nrpe_handler}{$client_hackme} $alias_params");
        }

        if ($info{msg}{$type}{$subcurrent}{autoaction_ann}) {
          $options .= "ANNOTATE,";
          #Not needed, we now use the autoaction wrapper to handle this stuff
          #printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ANNOTATIONS", "null");
        }
        $options .= "ACK" if ($info{msg}{$type}{$subcurrent}{autoaction_ack});
        printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_EVENT_OPTIONS", $options);

        #printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ARG_EVENTHANDLER", ${$info{msg}{$type}{$subcurrent}{condition}{arg}}[0]) if ($info{msg}{$type}{$subcurrent}{autoaction} =~ /\<.+\>/);
        if ($info{msg}{$type}{$subcurrent}{condition}{postmatch_arg}) {
          printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ARG_EVENTHANDLER", $info{msg}{$type}{$subcurrent}{condition}{postmatch_arg}) if ($info{msg}{$type}{$subcurrent}{autoaction} =~ /\<.+\>/);
        }else{
          if (${$info{msg}{$type}{$subcurrent}{condition}{arg}}[0]) {
            print "ERROR: postmatch_arg is missing for $info{msg}{$type}{$subcurrent}{desc}, using arg instead, may be non-issue\n" if ($opts{d} > 2);
            printf NAGOUT ("\t\%-25s\%-25s\n", "_CUSTOM_ARG_EVENTHANDLER", ${$info{msg}{$type}{$subcurrent}{condition}{arg}}[0]) if ($info{msg}{$type}{$subcurrent}{autoaction} =~ /\<.+\>/);
          }else{
            print "ERROR: postmatch_arg is missing for $info{msg}{$type}{$subcurrent}{desc}, and cannot use arg, may be non-issue\n" if ($opts{d});
          }
        }
      } 

      #Build notes string;
      my $notes = "-SERVICE_NAME-:-OBJECT-:-APPLICATION-:-KI-:-MSGGRP-";
      for my $key ("service_name", "object", "application", "ki", "msggrp") {
        my $seekme = uc("-$key-");
        if ($info{msg}{$type}{$subcurrent}{$key}) {
          $notes =~ s/$seekme/$info{msg}{$type}{$subcurrent}{$key}/g;
        }elsif ($info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{$key}) {
          $notes =~ s/$seekme/$info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{$key}/g;
        }else{
          $notes =~ s/$seekme//g;
        }
      }
      printf NAGOUT ("\t\%-25s\%-25s\n", "notes", $notes);
      print NAGOUT "}\n";
    }
  }

  close NAGOUT;
  print "\nCreated nagios cfg = $nagios_conf\n";
}

sub validate_opts {
  if ($opts{o}){
    unless (-d "$opts{o}") {
      mkpath ( $opts{o} );
    }
  }

  if ($opts{d}) {
    if ($opts{d} !~ /\d/) {
      $opts{d} = 1;
    }elsif ($opts{d} < 1 || $opts{d} > 3) {
      print "Invalid debug value, must be 1-3, or nothing\n";
      exit;
    } 
  }else{
    $opts{d} = 0;
  }
  unless ($opts{x}) {
    print "ERROR: -x option is required\n";
    exit;
  }else{
    chomp($opts{x}); 
    $opts{x} = uc($opts{x});
    if ($opts{t} !~ /^TG_/ && !$opts{b}) {
      print "ERROR: with -t option you must enter -b option\n";
      exit
    }
    if ($opts{b}) {
      chomp($opts{b});
      unless (grep {/$opts{b}/} @valid_types) {
        print "ERROR: your type '$opts{b}' isn't valid, valid ones=@valid_types\n";
        exit;
      }
    }
  } 
  unless ($opts{z}) {
    print "ERROR: -z option is required\n";
    exit;
  }else{
    unless ($opts{z} eq "WIN" || $opts{z} eq "LINUX") {
      print "ERROR: -z value is invalid : must be WIN or LINUX\n";
      exit;
    }
  } 
  print "DEBUG : search_dir = $info{search_dir}\n" if ($opts{d});

  #Build out user stuff
  $info{user}{check_command} = "";
 
  if ($opts{z} eq "WIN") { 
    ask_mode_asker("Mind me asking if this is W2K3 or W2K8?", \$info{user}{win_version},"string","W2K3");
    $info{user}{win_version} = $info{user}{win_version} ? $info{user}{win_version} : "W2K3";
    unless ($info{user}{win_version} =~ /(W2K3|W2K8)/) {
      print "ERROR: invalid windows version used\n";
      exit;
    }
    #NOTE: appsdir is a unique path we have per OS, windows only stuff
    $info{user}{appsdir} = 'E:/APPS' if ($info{user}{win_version} eq "W2K3");
    $info{user}{appsdir} = 'D:/APPS' if ($info{user}{win_version} eq "W2K8");
  }
  #Set a default
  $info{user}{appsdir} = defined $info{user}{appsdir} ? $info{user}{appsdir} : 'D:/APPS';
}

#-----------------------------------------------------------------------------------------
sub build_cron {
  if (keys %{$info{start}{SCHEDULE}}  || keys %{$info{start}{MONITOR}}) {
    action_cron ("open",1);
    if (keys %{$info{start}{SCHEDULE}}) {
      #SCEDULE TEMPLATE
      foreach my $template (keys %{$info{start}{SCHEDULE}}) {
        my ($h, $m, $wd, $cmd);
        foreach my $key (keys %{$info{start}{SCHEDULE}{$template}}) {
          $h = defined $info{start}{SCHEDULE}{$template}{hour} ? $info{start}{SCHEDULE}{$template}{hour} : "*";
          $m = defined $info{start}{SCHEDULE}{$template}{minute} ? $info{start}{SCHEDULE}{$template}{minute} : "*";
          $wd = defined $info{start}{SCHEDULE}{$template}{weekday} ? $info{start}{SCHEDULE}{$template}{weekday} : "*";
          $cmd = defined $info{start}{SCHEDULE}{$template}{schedprog} ? $info{start}{SCHEDULE}{$template}{schedprog} : "WTF-lost";
          $cmd =~ s/auto_service.pl/nagios_auto_service.pl/g;
          map {s/\s//g} ($h, $m, $wd);
          print "DEBUG-build_cron : $template : $key = $info{start}{SCHEDULE}{$template}{$key}, h=$h, m=$m, wd=$wd prog=$cmd\n" if ($opts{d});
        }

        if ($cmd !~ /APPSDIR/) { 
          action_cron("add","$m $h \* \* $wd $info{user}{appsdir}\\$cmd\n");
        }else{
          action_cron("add","$m $h \* \* $wd $cmd\n");
        }
      } 
    }
    #MONITOR TEMPLATE
    if (keys %{$info{start}{MONITOR}}) {
      foreach my $template (keys %{$info{start}{MONITOR}}) {
        #Common replacements: 
        $info{start}{MONITOR}{$template}{monprog} =~ s/auto_service.pl/nagios_auto_service.pl/g;
        $info{start}{MONITOR}{$template}{monprog} =~ s/\<\$NAME\>/$info{start}{MONITOR}{$template}{descnospace}/g;
        $info{start}{MONITOR}{$template}{monprog} =~ s/-mon \\".*?\\"/---INSERT_SERVICE_NAME_HERE---/;

        action_cron("add","(OVO INTERVAL=$info{start}{MONITOR}{$template}{interval}) * * * * * $info{start}{MONITOR}{$template}{monprog}\n");
      }
    }
    action_cron ("close",1)
  }
}

#-----------------------------------------------------------------------------------------
sub action_cron {
  my ($action, $msg) = @_;
  my $plugin_cron = "$opts{o}/$opts{x}_cron.cfg"; 
  $msg =~ s/$/\n/ unless ($msg =~ /\n$/);

  if ($action eq "add") {
    print PLUGCRON $msg;
  }elsif ($action eq "open") {
    print "DEBUG : Starting cron build..\n" if ($opts{d});
    open (PLUGCRON,">$plugin_cron") or die "couldn't open PLUGCRON : $!\n";
    print PLUGCRON "\#SF_VERSION=\$Revision\$\n";
    if ($opts{z} eq "WIN") {
      print PLUGCRON "\#              field          allowed values
\#              -----          --------------
\#              minute         0-59
\#              hour           0-23
\#              day of month   1-31
\#              month          1-12 
\#              day of week    0-6 (0 is Sunday)
\#
\# minute	hour	day_month	month	day_week	script\n";

    }

  }elsif ($action eq "close") {
    close PLUGCRON;
    print "\nCreated cron cfg file = $plugin_cron\n";
  }
}

#-----------------------------------------------------------------------------------------
sub build_ini_win {
  my ($action, $msg) = @_;
  my $plugin_ini = "$opts{o}/$opts{x}.ini"; 
  $msg =~ s/$/\n/ unless ($msg =~ /\n$/);
 
  if ($action eq "add") {
    print PLUGINI $msg;
  }elsif ($action eq "open") {
    my $req_scripts = join ",", @{$info{required_scripts}}; 

    open (PLUGINI,">$plugin_ini") or die "couldn't open PLUGOUT : $!\n";
    print PLUGINI "
;SF_VERSION=\$Revision\$
;SF_REQUIRED_SCRIPTS=$req_scripts
[modules]
[Settings]
[log]
[NSClient]
[NRPE]
[Check System]
[External Script]
[External Scripts]
[External Alias]
[NSCA Agent]
[NSCA Commands]\n";
  }elsif ($action eq "close") {
    print PLUGINI "\n[NRPE Handlers]\n";
    foreach (sort keys %{$info{nrpe_handler}}) {
      print PLUGINI "$info{nrpe_handler}{$_}=$_\n";
    }
    close PLUGINI;
    print "\nCreated check_logfile ini file = $plugin_ini\n";
  }
}

#-----------------------------------------------------------------------------------------
# This should build the client side cfg file which can drive the check_logfiles plugin
# If the type isn't LOGFILE we have no need to run this sub so return
#
sub build_plugin_conf_win {

  my $plugin_conf = "$opts{o}/$opts{x}_check_logfiles.cfg"; 
  my ($minus); 
  open (PLUGOUT,">$plugin_conf") or die "couldn't open PLUGOUT : $!\n"; 

  #Start INI file
  build_ini_win ("open","bla"); 

  if ($info{user}{nsca_in_cfg}) {
    build_ini_win ("add", "$opts{x}_generic=check_logfiles -config=$info{user}{appsdir}/NSClient/conf/$opts{x}_check_logfiles.cfg"); 
  }

  print PLUGOUT "#SF_VERSION=\$Revision:\$
\$protocolsdir=\"$info{user}{appsdir}/NSClient/tmp\";
\$protocolretention=3;
\$seekfilesdir=\"$info{user}{appsdir}/NSClient/tmp\";
\$scriptpath='$info{user}{appsdir}/NSClient/scripts;$info{user}{appsdir}/perl/bin';
\n\n\@searches = (\n";

  #This drives the $opts{s} option 
  if ($opts{s}) {
    if ($opts{s} !~ /\d+/) {
      close PLUGOUT;
      die "ERROR: -s must be numerical\n"; 
    }
    $minus = (keys(%{$info{msg}{"LOGFILE"}}) + 1) - $opts{s};
  }

  my $c = 1;
  foreach my $type (keys %{$info{msg}}) {
    if ( $type ne "LOGFILE") {
      print "SKIPPING-build_plugin_conf_win : $type not LOGFILE\n";
      next;
    } 
    foreach my $subcurrent (keys %{$info{msg}{$type}}) {
      unless ($info{msg}{$type}{$subcurrent}{desc} &&
		   	$info{msg}{$type}{$subcurrent}{severity}) {
        print "ERROR: invalid condition '$subcurrent'\n";
        next;
      }

      unless ($info{msg}{$type}{$subcurrent}{template}) {
        print "WARNING: $type - $subcurrent doesn't have template defined\n";
      }

      unless ($info{user}{nsca_in_cfg}) {
        build_ini_win ("add", "$info{msg}{$type}{$subcurrent}{descnospace}=check_logfiles -config=$info{user}{appsdir}/NSClient/conf/$opts{x}_check_logfiles.cfg -searches=$info{msg}{$type}{$subcurrent}{descnospace}"); 
      }

      #Last minute fix of tag, found while testing
      $info{msg}{$type}{$subcurrent}{desc} =~ s/\//-/g;

      print PLUGOUT "\t{\n\t\ttag => '$info{msg}{$type}{$subcurrent}{descnospace}',\n";

      if ($info{user}{nsca_in_cfg} eq "y") {
        $info{msg}{$type}{$subcurrent}{options} .= "script,"; 
        $info{msg}{$type}{$subcurrent}{script_value} .= "perl.exe";
        $info{msg}{$type}{$subcurrent}{script_params} = "$info{user}{appsdir}\\NSClient\\scripts\\sf_send_nsca.pl \$CL_HOSTNAME\$--dLm--\$CL_TAG\$--dLm--\$CL_SERVICESTATEID\$--dLm--\$CL_SERVICEOUTPUT\$--dLm--1"; 
      }

      if ($info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{logpath} =~ /^\%.*\%$/) {
        #------------Special section for logfile monitoring against a win eventlog
        
        #Setup output using our standard format for all Win Event Log checks
        $info{msg}{$type}{$subcurrent}{options} .= "eventlogformat=\"type:%t id:%i src:%s %m\","; 
 
        my $eventlogTrimmed = $info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{logpath};
        $eventlogTrimmed =~ s/\%//g;
        $eventlogTrimmed = "Application" if ($eventlogTrimmed =~ /application/i); 
        $eventlogTrimmed = "System" if ($eventlogTrimmed =~ /system/i); 
        $eventlogTrimmed = "Security" if ($eventlogTrimmed =~ /security/i); 
        print PLUGOUT "\t\ttype => 'eventlog',
\t\teventlog => {
\t\t\teventlog => '$eventlogTrimmed',
\t\t\tinclude => {
";
       if ($info{msg}{$type}{$subcurrent}{condition}{application}) {
         print PLUGOUT "\t\t\t\tsource => '$info{msg}{$type}{$subcurrent}{condition}{application}',\n"; 
       }
       if ($info{msg}{$type}{$subcurrent}{condition}{severity}) {
         #I just figured out OVO does some field mappings, this needs to map back to windows language
         my $sev = $info{msg}{$type}{$subcurrent}{condition}{severity};
         $sev = "error" if ($sev =~ /critical/i);
         $sev = "info" if ($sev =~ /information/i);
         $sev = "warning" if ($sev =~ /warning/i);
         print PLUGOUT "\t\t\t\teventtype => '$info{msg}{$type}{$subcurrent}{condition}{severity}',\n"; 
       }
       if ($info{msg}{$type}{$subcurrent}{condition}{eventid}) {
         print PLUGOUT "\t\t\t\teventid => '$info{msg}{$type}{$subcurrent}{condition}{eventid}',\n"; 
       }
       print PLUGOUT "\t\t\t},
\t\t\texclude => {
\t\t\t},
\t\t},\n";
      }else{
        #Standard logfile entry
        print PLUGOUT "\t\tlogfile => '$info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{logpath}',\n";
      }

      if ($info{msg}{$type}{$subcurrent}{severity} =~ /(warning|minor|normal)/i) {
        print PLUGOUT "\t\twarningpatterns => [\n"; 
      }elsif ($info{msg}{$type}{$subcurrent}{severity} =~ /(major|critical)/i) {
        print PLUGOUT "\t\tcriticalpatterns => [\n"; 
      }

      #NOTE: if eventid exists lets skip this and just depend on the eventid for now
      if ($info{msg}{$type}{$subcurrent}{condition}{eventid}) {
        if ($info{msg}{$type}{$subcurrent}{condition}{postmatch}) {
          print PLUGOUT "\t\t\t'$info{msg}{$type}{$subcurrent}{condition}{postmatch}',\n"; 
        }else{
          print PLUGOUT "\t\t\t'.*',\n"; 
        }
      }else{ 
       
        if ($info{msg}{$type}{$subcurrent}{condition}{postmatch}) {
          print PLUGOUT "\t\t\t'$info{msg}{$type}{$subcurrent}{condition}{postmatch}',\n"; 
        }else{
	  foreach my $x (@{$info{msg}{$type}{$subcurrent}{condition}{text}}) {
            print PLUGOUT "\t\t\t'$x',\n"; 
          }							
        } 
      }
      $info{msg}{$type}{$subcurrent}{options} =~ s/,$//;
      print PLUGOUT "\t\t],\n\t\toptions => '$info{msg}{$type}{$subcurrent}{options}',\n";	

      if ($info{msg}{$type}{$subcurrent}{script_value}) {
        print PLUGOUT "\t\tscript => '$info{msg}{$type}{$subcurrent}{script_value}',\n";	
        print PLUGOUT "\t\tscriptparams => '$info{msg}{$type}{$subcurrent}{script_params}',\n";	
      }

      #----- SUPPRESSION LOGIC -----
      if ($opts{s}) {
        if ($c >= $minus) {
          print PLUGOUT "\t\tcriticalexceptions => [\n";
          foreach my $sc (keys %{$info{suppression}{$type}}) {
            foreach my $s (@{$info{suppression}{$type}{$sc}{condition}}) {
              print PLUGOUT "\t\t\t'$s',\n";
            }
          } 
          print PLUGOUT "\t\t],\n";
          print PLUGOUT "\t\twarningexceptions => [\n";
          foreach my $sc (keys %{$info{suppression}{$type}}) {
            foreach my $s (@{$info{suppression}{$type}{$sc}{condition}}) {
              print PLUGOUT "\t\t\t'$s',\n";
            }
          } 
          print PLUGOUT "\t\t],\n";
        }
      }

      print PLUGOUT "\t},\n";
      $c++;
    }
  }

  print PLUGOUT ");";
  close PLUGOUT;

  #Start INI file
  build_ini_win ("close","bla"); 

  unless (`grep tag $plugin_conf`) {
   unlink($plugin_conf); 
  }else{ 
    print "\nCreated check_logfile conf = $plugin_conf\n";
  }
}

#-----------------------------------------------------------------------------------------
sub build_plugin_conf_linux {
  my $plugin_conf = "$opts{o}/$opts{x}_check_logfiles.cfg"; 
  my ($minus); 
  open (PLUGOUT,">$plugin_conf") or die "couldn't open PLUGOUT : $!\n"; 

  print PLUGOUT "\$MACROS = {\n\tCL_NSCA_HOST_ADDRESS => '---SF_MGMTSERV---',
\tCL_NSCA_CONFIG_FILE => '/usr/local/nagios/etc/nsca.cfg',\n};\n\n\@searches = (\n";

  #This drives the $opts{s} option 
  if ($opts{s}) {
    if ($opts{s} !~ /\d+/) {
      close PLUGOUT;
      die "ERROR: -s must be numerical\n"; 
    }
    $minus = (int(keys %{$info{msg}{"LOGFILE"}}) + 1) - $opts{s};
  }

  my $c = 1;
  foreach my $type (keys %{$info{msg}}) {
    foreach my $subcurrent (keys %{$info{msg}{$type}}) {
  
      #This is essentially a LOGFILE section, don't allow MONITOR templates
      if ($type eq "MONITOR" ) {
        print "SKIPPING-build_plugin_conf_linux : for $type $subcurrent because it isn't appropriate\n" if ($opts{d});
        next;
      }

      unless ($info{msg}{$type}{$subcurrent}{desc} &&
		   	$info{msg}{$type}{$subcurrent}{severity}) {
        print "ERROR: invalid condition $type '$subcurrent'\n";
        next;
      }

      #Last minute fix of tag, found while testing
      $info{msg}{$type}{$subcurrent}{desc} =~ s/\//-/g;

      print PLUGOUT "\t{\n\t\ttag => '$info{msg}{$type}{$subcurrent}{descnospace}',\n";
      print PLUGOUT "\t\tlogfile => '$info{start}{$type}{$info{msg}{$type}{$subcurrent}{template}}{logpath}',\n";
      if ($info{msg}{$type}{$subcurrent}{severity} =~ /(warning|minor)/i) {
        print PLUGOUT "\t\twarningpatterns => [\n"; 
      }elsif ($info{msg}{$type}{$subcurrent}{severity} =~ /(major|critical)/i) {
        print PLUGOUT "\t\tcriticalpatterns => [\n"; 
      }
      foreach my $x (@{$info{msg}{$type}{$subcurrent}{condition}{text}}) {
        print PLUGOUT "\t\t\t'$x',\n"; 
      }							
      $info{msg}{$type}{$subcurrent}{options} =~ s/,$//;
      print PLUGOUT "\t\t],\n\t\toptions => '$info{msg}{$type}{$subcurrent}{options}',\n";	

      # SUPPRESSION LOGIC
      if ($opts{s}) {
        if ($c >= $minus) {
          print PLUGOUT "\t\tcriticalexceptions => [\n";
          foreach my $sc (keys %{$info{suppression}{$type}}) {
            foreach my $s (@{$info{suppression}{$type}{$sc}{condition}}) {
              print PLUGOUT "\t\t\t'$s',\n";
            }
          } 
          print PLUGOUT "\t\t],\n";
          print PLUGOUT "\t\twarningexceptions => [\n";
          foreach my $sc (keys %{$info{suppression}{$type}}) {
            foreach my $s (@{$info{suppression}{$type}{$sc}{condition}}) {
              print PLUGOUT "\t\t\t'$s',\n";
            }
          } 
          print PLUGOUT "\t\t],\n";
        }
      }

      if ($info{user}{nsca_in_cfg}) {
        print PLUGOUT "\t\tscript => 'send_nsca',
\t\tscriptparams => '-H \$CL_NSCA_HOST_ADDRESS\$ -c \$CL_NSCA_CONFIG_FILE\$',
\t\tscriptstdin => '\$CL_HOSTNAME\$\\t\$CL_TAG\$\\t\$CL_SERVICESTATEID\$\\t\$CL_SERVICEOUTPUT\$\\n',\n";
      }
      print PLUGOUT "\t},\n";
      $c++;
    }
  }

  print PLUGOUT ");";
  close PLUGOUT;
  print "\nCreated check_logfile conf = $plugin_conf\n";
}

sub obtain_input_wrapper {
  $opts{o} = $opts{o} ? $opts{o} : "/tmp";	
  if ($opts{t} =~ /^TG_(.*)/) {
    $opts{t} = $1;
    print "DEBUG-obtain_input_wrapper : using a template group : $opts{t}\n" if ($opts{d});
    open (TGIN,"<$TG_file") or die "couldn't open input $TG_file : $!\n";
    my $active_TG = "";
    foreach my $line (<TGIN>) {
      if ($line =~ /^TEMPLATE_GROUP\s+"(.*)"/) {
        last if ($active_TG eq $opts{t} && $1 ne $active_TG);
        $active_TG = $1; 
      }
      next unless ($opts{t} eq $active_TG);
      $info{master_desc} = $1 if ($line =~ /^\s+DESCRIPTION\s+(.*)$/);
      if ($line =~ /^\s+MEMBER_(\w+)\s+"(.*)"$/) {
        my ($type, $tem) = ($1, $2);
        validate_load_input($tem,$type);
      }  
    }

    close TGIN;
  }else{
    print "DEBUG-obtain_input_wrapper : using a standard template : $opts{t}\n" if ($opts{d});
    validate_load_input($opts{t},$opts{b});
  }
}


sub validate_load_input {
  my ($template,$type) = @_;
  my $input_f;  

  if($template =~ /TEMPLATE_GROUP/) {
    print "INFORMATION: type $type will be skipped\n";
    return;
  } 

  $type = "SCHEDULE" if ($type eq "SCHED"); 

  if ($opts{i}) {
    $input_f = $opts{i};
  }elsif ( $type eq "TEMPLATE_GROUP") {
    print "SKIPPING : $template is a $type, we cannot process imbedded template_groups\n"; 
    return; 
  }else{
    $input_f = $info{search_dir} . "LOGFILE/logfile.dat" if ($type eq "LOGFILE");
    $input_f = $info{search_dir} . "MONITOR/monitor.dat" if ($type eq "MONITOR");
    $input_f = $info{search_dir} . "SCHEDULE/schedule.dat" if ($type eq "SCHEDULE");
    $input_f = $info{search_dir} . "INTERFACE/interface.dat" if ($type eq "OPCMSG");
  }

  unless ($input_f) {
    print "ERROR: input_f still unknown for type=$type, template=$template\n";
    return;
  } 

  print "DEBUG: input used = $input_f, output dir = $opts{o}, template = $template, type=$type\n" if ($opts{d});

  open (IN,"<$input_f") or die "couldn't open $input_f : $!\n";

  my (%placeholder, $subcurrent, $subsubcurrent);
  my $current = "inactive";

  #----------------------------------------------
  #these help us mark start/end/sub/misc of the template.  We need to know what section we are in so we can
  #properly store the fields, hey look I can do comments!
  #----------------------------------------------
  $placeholder{start}       = '^\s*' . $type . '.*' . $template;
  $placeholder{suppression} = '^\s*SUPPRESSCONDITIONS';
  $placeholder{msg}         = '^\s*MSGCONDITIONS';
  $placeholder{failure}     = '^\s*FAILURE';
  $placeholder{end}         = '^\s*' . $type;

  foreach (keys %placeholder) {
    print "DEBUG: $_ = $placeholder{$_} \n" if ($opts{d} > 1);
  }

  #This is necessary so that we can keep the entries in order, order matters!
  #It also makes it easier to get the $opts{s} functionality to work
  #my $c = 1;

  foreach my $l (<IN>) {
    chomp ($l);
    last                     if ($l =~ /$placeholder{end}/ && $current ne "inactive");
    $current = "start"       if ($l =~ /$placeholder{start}/i);
    $current = "suppression" if ($l =~ /$placeholder{suppression}/ && $current ne "inactive");
    $current = "msg"         if ($l =~ /$placeholder{msg}/ && $current ne "inactive");
    $current = "failure"     if ($l =~ /$placeholder{failure}/ && $current ne "inactive");

    print "DEBUG $current : $l\n" if ($opts{d} > 2 && $current ne "inactive"); 
 
    if ($current eq "start") {
      if ($l =~ /(LOGPATH|READFILE) "(.*)"/) {
        $info{start}{$type}{$template}{logpath}     = $2;
        $info{start}{$type}{$template}{logpath} =~ s#\<\$OvAgentDir.*\\#$info{user}{appsdir}/NSClient/tmp/#g;
      }
      $info{start}{$type}{$template}{interval}    = $1 if ($l =~ /INTERVAL\s+"(.*)"/); 
      $info{start}{$type}{$template}{severity}    = $1 if ($l =~ /SEVERITY\s+(.*)/); 
      $info{start}{$type}{$template}{application} = $1 if ($l =~ /APPLICATION\s+"(.*)"/); 
      $info{start}{$type}{$template}{msggrp}      = $1 if ($l =~ /MSGGRP\s+"(.*)"/); 
      $info{start}{$type}{$template}{object}      = $1 if ($l =~ /OBJECT\s+"(.*)"/); 
      if ($l =~ /DESCRIPTION\s+"(.*)"/) { 
        ($info{start}{$type}{$template}{desc},$info{start}{$type}{$template}{descnospace}) = scrub_desc($1);
      }

      #MONITOR TEMPLATES
      $info{start}{$type}{$template}{monprog}     = $1 if ($l =~ /MONPROG\s+"(.*)"/); 

      #SCHEDULE TEMPALTES
      $info{start}{$type}{$template}{minute}    = $1 if ($l =~ /MINUTE\s+"(.*)"/); 
      $info{start}{$type}{$template}{hour}      = $1 if ($l =~ /HOUR\s+"(.*)"/); 
      $info{start}{$type}{$template}{weekday}   = $1 if ($l =~ /WEEKDAY\s+"(.*)"/); 
      $info{start}{$type}{$template}{user}      = $1 if ($l =~ /USER\s+"(.*)"/); 
      $info{start}{$type}{$template}{schedprog} = $1 if ($l =~ /SCHEDPROG\s+"(.*)"/); 

      if ($info{start}{$type}{$template}{desc} && ( ! defined $info{master_desc})) {
        #This should match when we are running against a template and not a tg, this sets master desc
        $info{master_desc} = $info{start}{$type}{$template}{desc};
      }

    #---------------------------------------
    # Pull suppress information
    #---------------------------------------
    }elsif ($current eq "suppression") {
      $subcurrent = $1 if ($l =~ /^\s*DESCRIPTION "(.*)"$/);  
      next unless ($subcurrent);
      $info{suppression}{$type}{$subcurrent}{desc} = $subcurrent; 
      
      if ($l =~ /^\s*TEXT\s*"(.*)"/) {
        #my ($match, $arg) =  scrub_regex($1);
        #push @{$info{suppression}{$type}{$subcurrent}{condition}}, $match;
        process_match_text ($1 , "suppression", $type, $template, $subcurrent);
      }

      if ($l =~ /^\s*APPLICATION\s*"(.*)"/) {
        $info{suppression}{$type}{$subcurrent}{condition}{application} = $1;
        $info{suppression}{$type}{$subcurrent}{condition}{application} =~ s/\|/,/g;
      }

    }elsif ($current eq "msg") {

      if ($l =~ /^\s*DESCRIPTION "(.*)"$/) {
        $subcurrent = $1;
        $info{msg}{$type}{$subcurrent}{desc} = $subcurrent;
        $info{msg}{$type}{$subcurrent}{desc_orig} = $subcurrent;
        $info{msg}{$type}{$subcurrent}{template} = $template;
        ($info{msg}{$type}{$subcurrent}{desc}, $info{msg}{$type}{$subcurrent}{descnospace}) = scrub_desc($subcurrent);

        if ($info{user}{trim_desc} eq "y") {
          $info{msg}{$type}{$subcurrent}{descnospace} = substr($info{msg}{$type}{$subcurrent}{descnospace}, 0, $info{user}{trim_size});
        }

        $info{msg}{$type}{$subcurrent}{options} .= "noperfdata,preferredlevel=critical,";
        if ($info{user}{truncate} eq "y") {
          $info{msg}{$type}{$subcurrent}{options} .= "maxlength=$info{user}{truncate_size},";
        }
      }

      $subsubcurrent = "SUPP_DUPL_IDENT_OUTPUT_MSG" if ($l =~ /^\s*SUPP_DUPL_IDENT_OUTPUT_MSG\s*$/);
      $subsubcurrent = "CONDITION" if ($l =~ /^\s*CONDITION\s*$/);
      $subsubcurrent = "SET" if ($l =~ /^\s*SET\s*$/);

      next unless ($subcurrent);

      #Duplicate Suppression
      $info{dup}{$type}{$subcurrent}{counter} = 1;
      if ( $subsubcurrent && $subsubcurrent eq "SUPP_DUPL_IDENT_OUTPUT_MSG" && $l =~ /COUNTER_THRESHOLD\s+(\d+)\s+RESET_COUNTER_INTERVAL\s+\"(.*)\"/) {
        $info{dup}{$type}{$subcurrent}{counter} = $1;
        $info{dup}{$type}{$subcurrent}{window} = $2;
      }					    
 
      $info{msg}{$type}{$subcurrent}{resend} = $1 if ($l =~ /\s*(.*" RESEND ".*)$/);

      #------------PULL OUT CONDITION MATCH CRITERIA---------------
      if ( $subsubcurrent && $subsubcurrent eq "CONDITION") {
        if ($l =~ /^\s*TEXT\s*"(.*)"/) {
          #print "pushing $1 to array ref '$subcurrent'\n" if ($opts{d} == 3);
          my $orig = $1;
          push @{$info{msg}{$type}{$subcurrent}{condition}{text_orig}},	$orig;
          my ($match, $arg) = scrub_regex($1);
          push @{$info{msg}{$type}{$subcurrent}{condition}{text}}, $match;	
          push @{$info{msg}{$type}{$subcurrent}{condition}{arg}}, $arg;	

          process_match_text($orig, "msg", $type, $template, $subcurrent); 
        }
        if ($l =~ /^\s*APPLICATION\s*"(.*)"/) {
          $info{msg}{$type}{$subcurrent}{condition}{application} = $1;
          $info{msg}{$type}{$subcurrent}{condition}{application} =~ s/\|/,/g;
        }
        if ($l =~ /^\s*SEVERITY\s*"(.*)"/) {
          $info{msg}{$type}{$subcurrent}{condition}{severity} = $1;
        }
      }

      $info{msg}{$type}{$subcurrent}{options} .= "nocase," if ($l =~ /^\s*TEXT\s*"(.*)".*ICASE/ && $subsubcurrent eq "CONDITION");
      $info{msg}{$type}{$subcurrent}{severity} = $1 if ($l =~ /^\s*SEVERITY (.*)/); 
      $info{msg}{$type}{$subcurrent}{application} = $1 if ($l =~ /^\s*APPLICATION "(.*)"/); 
      $info{msg}{$type}{$subcurrent}{service_name} = $1 if ($l =~ /^\s*SERVICE_NAME "(.*)"/); 
      $info{msg}{$type}{$subcurrent}{ki} = $1 if ($l =~ /^\s*HELPTEXT "(.*)"/); 
      $info{msg}{$type}{$subcurrent}{msggrp} = $1 if ($l =~ /^\s*MSGGRP "(.*)"/); 
      $info{msg}{$type}{$subcurrent}{autoaction} = $1 if ($l =~ /^\s*AUTOACTION "(.*)"/); 
      $info{msg}{$type}{$subcurrent}{autoactionfull} = $1 if ($l =~ /^\s*AUTOACTION (.*)$/); 
      $info{msg}{$type}{$subcurrent}{autoaction_ann} = 1 if ($l =~ /^\s*AUTOACTION ".*".*\sANNOTATE/); 
      $info{msg}{$type}{$subcurrent}{autoaction_ack} = 1 if ($l =~ /^\s*AUTOACTION ".*".*\sACK/); 
      $info{msg}{$type}{$subcurrent}{autoaction_actionnode} = $1 if ($l =~ /^\s*AUTOACTION ".*".*\sACTIONNODE\s*IP\s*0.0.0.0\s*"(.*?)"/); 

      #If the autoaction section is populated lets try and pull a list of scripts to populate the SF_REQUIRED_SCRIPTS portion of the ini
      #NOTE: right now this is only grabbing perl scripts!!!!
      if ($l =~ /^\s*AUTOACTION "(?:%APPSDIR\%.*perl.*bin.*perl.exe -S)\s+(\S+)(\s+.*"|")/) {
        my $script = $1;
        unless (grep {/^$script$/} @{$info{required_scripts}}) { 
          print "DEBUG: adding $script to SF_REQUIRED_SCRIPTS\n" if ($opts{d} > 2);
          push @{$info{required_scripts}}, $script; 
        }
      }

      if ($l =~ /^\s*TEXT "(.*)"/ && $subsubcurrent eq "SET") {
        $info{msg}{$type}{$subcurrent}{text_orig} = $1;
        $info{msg}{$type}{$subcurrent}{text_clean} = scrub_text($1);
        $info{msg}{$type}{$subcurrent}{_CUSTOM_ARG_FETCH} = 1 if ($l =~ /\<\S+\>/); 
      }
    }
  }
  close IN;

  foreach (keys %{$info{start}{$type}{$template}}) {
    print "DEBUG: $_ = $info{start}{$type}{$template}{$_} \n" if ($opts{d} > 2);
  }

  foreach my $key1 (keys %{$info{suppression}{$type}}) {
    if ($info{suppression}{$type}{$subcurrent}{condition}{postmatch}) {
      print "DEBUG : suppression hash tree: $type : $key1 === $info{suppression}{$type}{$subcurrent}{condition}{postmatch}\n" if ($opts{d} > 2);
    }else{
      #print msg
    }
  }
  foreach my $key1 (keys %{$info{msg}{$type}}) {
    print "DEBUG : msg hash tree : $type === $key1\n" if ($opts{d} > 2);
  }
}

sub USAGE{
  my $tmp = join ",", @valid_types;
  warn "\nusage - $0
  -a = ASK MODE, asks you questions to help build files
  -b <type> = TYPE, this is used if you want to migrate a single template, you need to indicate type
		   valid TYPES are $tmp 
       This is required with -t that isn't TG_
  -d <value> = debug values are 1-3, 3 being most debug
  -o <dir> = output directory, several files will be created (default /tmp)
  -i <input file> = Not required, because a default can be coded into script (default <see script>)
  -t <template> = the ovo template to build into nagios (REQUIRED)
     TG_<templateGroup> - If you want it to work off of a templateGroup instead of a template
  -s <digit(s)> = number of wild card conditions at end which need the suppression added, if 0 suppression lines won't be used (linux only)
  -x <name> = name of template, i.e NG-ETL-LINUX-BASE, it will use this name to build cfgs and output files (REQUIRED)
  -g = turns off the build out service group for each condition
  -z <os type> = either WIN or LINUX (REQUIRED)
  -h = help

  Some fields are intentionally left out.  In these situation I will use a place keeper so it is easy to finish the conf with a simple find-replace
  
";

  exit ;
}

sub scrub_regex {
  my $input = shift;
  #-----------------------
  #  Invalid characters
  #-----------------------
  #print "WARNING - '$input' has an invalid character in condition, line will be scrubbed\n" if ($input =~ /'/); 
  $input =~ s/\'/\\S/g; #This can't be used because it is the condition limiter in the check_logfiles logic
  $input =~ s/\*/-staR-/g;  #Every tried to replace regex with regex by matching with regex, ya trust me you don't want *'s floating around
  #$input =~ s/\//-slasH-/g;  #Every tried to replace regex with regex by matching with regex, ya trust me you don't want *'s floating around
  #$input =~ s/\\\(/\\$&/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  #$input =~ s/\\\(/\\$&/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  #$input =~ s/[\(\)]/\\$&/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  $input =~ s/\(/-cIrcS-/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  $input =~ s/\)/-cIrcE-/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  $input =~ s/\[/-sqrS-/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  $input =~ s/\]/-sqrE-/g; #OVO allowed you to use a "(" or ")", here it must be back whacked
  $input =~ s/\|/-bAr-/g; #OVO allowed you to use a "(" or ")", here it must be back whacked

  #-----------------------
  # replace special vendor wildcards with regex equivalent, insert black magic
  #-----------------------
  my %regex_fixer;

  $regex_fixer{'<!-sqrS-(.+?)-sqrE->'} = "dollarone_neg";

  $regex_fixer{'<(\d+)?-staR-(\.\w+)?>'} = '.*';

  $regex_fixer{'<(\d+)?\#(\.\w+)?>'} = '\d+';

  $regex_fixer{'<(\d+)?[_S](\.\w+)?>'} = '\s+';

  $regex_fixer{'<(\d+)?\@(\.\w+)?>'} = '\S+';

  $regex_fixer{'<-sqrS-(.*?)-sqrE-(\.\w+)?>'} = "dollarone";

  my $arg = $input;
  foreach my $exp (keys %regex_fixer) {
    while ($input =~ /$exp/) {
      print "\tMATCH = $input TO $exp\n" if ($opts{d} == 3);
      my ($var1, $var2) = ($1, $2);
      print "\t\tMATCH var1 = $var1\n" if ($var1 && $opts{d} == 3);
      print "\t\tMATCH var2 = $var2\n" if ($var2 && $opts{d} == 3);
      $regex_fixer{$exp} = "-sqrS-$var1-sqrE-" if ($regex_fixer{$exp} eq "dollarone");
      #This is a negative lookahead
      $regex_fixer{$exp} = "(?!$var1)" if ($regex_fixer{$exp} eq "dollarone_neg");
      if ($var1)  {
        #NOTE: turns out that nested quantifiers are not good mojo, just removing for now 
        #$input =~ s/$&/$regex_fixer{$exp}\{$var1\}/g;
        print "\t\tFIXING NESTED QUANTIFIER = substitute '$&' with '$regex_fixer{$exp}'\n" if ($opts{d} == 3);
        $input =~ s/$&/$regex_fixer{$exp}/g;
        if ($var2) {
          $arg =~ s/$&/\($regex_fixer{$exp}\)/g;
        }else{
          $arg =~ s/$&/$regex_fixer{$exp}/g;
        }
      }else{
        print "\t\tMATCHED = $&\n" if ($opts{d} == 3);
        $input =~ s/$&/$regex_fixer{$exp}/g;
        if ($var2) {
          $arg =~ s/$&/\($regex_fixer{$exp}\)/g;
        }else{
          $arg =~ s/$&/$regex_fixer{$exp}/g;
        }
      }
      print "\t\t\tNOW $input\n" if ($opts{d} == 3);
    }
  }
  $input =~ s/-cIrcS-/\(/g;
  $input =~ s/-cIrcE-/\)/g;
  $input =~ s/-sqrS-/\[/g;
  $input =~ s/-sqrE-/\]/g;
  $input =~ s/-bAr-/\|/g; 
  $input =~ s/-/\\-/g; 
  $arg =~ s/-sqrS-/\[/g;    
  $arg =~ s/-sqrE-/\]/g;  
  $arg =~ s/-bAr-/\|/g; 
  $input = '.*' unless ($input);
  print "\tcompleted scrub_regex input = $input\n\tcompleted scrub_regex arg = $arg\n" if ($opts{d} > 2);
  print "---------------------- Finished scrub_regex---------------------------------\n" if ($opts{d} > 2);
  return ($input, $arg);
}

sub scrub_text {
  my $input = shift;
  return $input if ($input !~ /[\<\>]/);
  my $input_cleaned = $input;
  my $c = 1;
  print "\tTEXT scrubbing, Start = $input\n" if ($opts{d} > 2);
  foreach my $word (split /\s+/, $input) {
    #print "\t\tword = $word\n" if ($opts{d} > 2);
    if ($word =~ /(\<\w+\>)/) {
      print "\tTEXT MATCHED - $word ($1)\n" if ($opts{d} > 2);
      $input_cleaned =~ s/$1/--SF_ARG$c--/;
      $c++;
    }
  }
  print "\tTEXT scrubbing, Ended = $input_cleaned\n" if ($opts{d} > 2);
  return $input_cleaned;
}

#-----------------------------------
# The purpose of this is to ask questions of users in ask mode and assign values
# types:
#   yesno = [y|n|yes|no] case insensitive
#   int   = \d+
#   string = \w+
#-----------------------------------
sub ask_mode_asker {
  my ($question, $answer, $type_in, $default) = @_;
  my %type;
  $type{yesno} = "(y|n)";
  $type{int} = "(enter an integer)";
  $type{string} = "(enter string value)";
  return 0 unless ($opts{a}); 

  unless (defined $type{$type_in} ) {
    print "ERROR: type is invalid\n";
    exit;
  }

  $question .= "?" unless ($question =~ /\?$/);
  print "\nQUESTION: $question $type{$type_in} ";
  print "(default=$default) " if ($default);
  print ": ";
  $$answer = <STDIN>; 
  chomp ($$answer);
  $$answer = $default unless ($$answer =~ /\w/);
  print "DEBUG - answer = '$$answer'\n" if ($opts{d} >= 2);
}

sub scrub_desc {
  my ($desc) = shift;
  my $descnospace;
  my $desc_orig;  
  $desc =~ s/:/_/g;
  $desc =~ s/,/_/g;
 
  if ($info{user}{remove_from_desc} eq "y") {
    $desc =~ s/$info{user}{remove_string_desc}//g;
  }
 
  print "DEBUG - desc prior to scrubbing '$desc'\n" if ($opts{d} >= 2);
  while ($desc =~ /[\/\'\.\!\#\*\<\>\[\]\,\:\(\)\%\=]/) {
    $desc =~ s/[\/\'\.\!\#\*\<\>\[\]\,\:\(\)\%\=]//g;
  }
  print "DEBUG - desc after to scrubbing '$desc'\n" if ($opts{d} >= 2);
  $desc =~ s/^\s+//g;
  $desc =~ s/^_//g;
  $desc =~ s/\s+/ /g;
  $desc =~ s/_+/_/g;
  $descnospace = $desc;
  $descnospace =~ s/\s+/_/g;
  $descnospace =~ s/_+/_/g;
  $descnospace =~ s/_-_/-/g;

  if ($info{user}{approve_desc} eq "y") {
    if ($info{user}{prescrub_desc} eq "y") {
      ask_mode_asker("Modify this description '$descnospace'", \$descnospace,"string","$descnospace");
    }else{
      ask_mode_asker("Modify this description '$desc_orig'", \$descnospace,"string","$desc_orig");
    }
  }

  if ($info{user}{prepend_desc} eq "y") {
    $descnospace = $opts{x} . "_" . $descnospace;
  }

  return ($desc, $descnospace);
}

sub process_match_text {
  my ($orig, $category, $type, $template, $subcurrent) = @_;

  # OVO cannot do matches against the actual eventID and check_logfiles can.  Lets use it where possible
  # to use it I have to pull it out of the TEXT, SUMMON THE POWER OF REGEX
  # BTW this drives me crazy and it needs to be fixed because it is such a hack!
  my $trash;
  if ($orig =~ /^\^?Event(?:\<\*\>)?ID:?(?:\<\*\>|.*HEX_EVENT_ID.*?|0x[0-9A-F]*)(\d+)\\?\)?/) {
    $info{$category}{$type}{$subcurrent}{condition}{eventid} = $1;
    ($info{$category}{$type}{$subcurrent}{condition}{postmatch},$info{$category}{$type}{$subcurrent}{condition}{postmatch_arg}) = scrub_regex($');
    print "\tDEBUG - found eventIDv1 = $1\n" if ($opts{d} > 2);
    print "\t\tDEBUG - postmatch eventIDv1 = $info{$category}{$type}{$subcurrent}{condition}{postmatch}\n" if ($opts{d} > 2);
  }elsif ($orig =~ /^\^?Event(?:\<\*\>)?ID:?(?:\<\*\>|.*HEX_EVENT_ID.*?|0x[0-9A-F]*)\<\[(\d+.*?)\]\>.*-?/) {
    $info{$category}{$type}{$subcurrent}{condition}{eventid} = $1;
    ($info{$category}{$type}{$subcurrent}{condition}{postmatch},$info{$category}{$type}{$subcurrent}{condition}{postmatch_arg}) = scrub_regex($');
    print "\tDEBUG - found eventIDv2 = $1\n" if ($opts{d} > 2);
    print "\t\tDEBUG - postmatch eventIDv2 = $info{$category}{$type}{$subcurrent}{condition}{postmatch}\n" if ($opts{d} > 2);
  }elsif ($orig =~ /^\^?Event(?:\<\*\>)?ID:.*?\[(\d+.*?)\]/) {
    my $x = $1;
    my $post = $';
    $x =~ s/\|/,/g;
    $info{$category}{$type}{$subcurrent}{condition}{eventid} = $x;
    ($info{$category}{$type}{$subcurrent}{condition}{postmatch},$info{$category}{$type}{$subcurrent}{condition}{postmatch_arg}) = scrub_regex($post);
    print "\tDEBUG - found eventIDv3 = $x\n" if ($opts{d} > 2);
    print "\t\tDEBUG - postmatch eventIDv3 = $info{$category}{$type}{$subcurrent}{condition}{postmatch}\n" if ($opts{d} > 2);
  }elsif ($orig =~ /^\^?EventID:.*?\(?(\d+)\)?/) {
    $info{$category}{$type}{$subcurrent}{condition}{eventid} = $1;
    ($info{$category}{$type}{$subcurrent}{condition}{postmatch},$info{$category}{$type}{$subcurrent}{condition}{postmatch_arg}) = scrub_regex($');
    print "\tDEBUG - found eventIDv4 = $1\n" if ($opts{d} > 2);
    print "\t\tDEBUG - postmatch eventIDv4 = $info{$category}{$type}{$subcurrent}{condition}{postmatch}\n" if ($opts{d} > 2);
  }elsif ($orig =~ /^\^?EventID:?\s*(.*)/) {
    ($info{$category}{$type}{$subcurrent}{condition}{postmatch},$info{$category}{$type}{$subcurrent}{condition}{postmatch_arg}) = scrub_regex($1);
    print "\t\tDEBUG - postmatch eventIDv5 = $info{$category}{$type}{$subcurrent}{condition}{postmatch}\n" if ($opts{d} > 2);
  }else{
    print "\tMISSED - found eventID = $orig, script needs a human\n" if ($orig =~ /EventID/);
  }
  #push @{$info{suppression}{$type}{$subcurrent}{condition}}, $match;
}
