#include <errno.h>		/* errno		*/
#include <getopt.h>
#include <netdb.h>		/* getaddrinfo(...)	*/
#include <signal.h>		/* sigaction(...)	*/
#include <stdio.h>		/* printf(...)		*/
#include <stdlib.h>		/* exit(...)		*/
#include <string.h>		/* memset(...)		*/
#include <time.h>		/* time(...)		*/
#include <unistd.h>		/* close(...)		*/
#include <arpa/inet.h>		/* inet_pton, inet_ntop */
#include <sys/socket.h>		/* socket(...)		*/
#include <sys/time.h>		/* gettimeofday(...)	*/
#include <netinet/in.h>		/* struct sockaddr_in	*/

#include <netinet/ip6.h>	/* struct ip6_hdr	*/
#include <netinet/icmp6.h>	/* struct icmp6_hdr	*/
#include <stdint.h>		/* uint...		*/

#define	BUF_SIZE		256

static char	*server_port = "3544";
static char	*server_address = NULL;
static double	warning_time = -1.;
static double	critical_time = -1.;
static int	socket_timeout = 10;
static char	*status_codes[] = {"OK", "WARNING", "CRITICAL"};

struct teredo_auth_header {
	uint16_t	proto;
	uint8_t		id_len;
	uint8_t		auth_len;
	uint64_t	nonce;
	uint8_t		confirm;
} __attribute__((__packed__));

struct teredo_origin_header {
	uint16_t	proto;
	uint16_t	xport;
	uint32_t	xaddr;
};

struct teredo_solicit
{
	struct	teredo_auth_header	auth;
	struct	ip6_hdr			ip6;
	struct	nd_router_solicit	rs;
	struct	nd_opt_hdr		opt;
	uint8_t				lladdr[14];
} __attribute__((__packed__));

struct teredo_advert
{
	struct	teredo_auth_header	auth;
	struct	teredo_origin_header	origin;
	struct	ip6_hdr			ip6;
	struct	nd_router_advert	ra;
	uint8_t	options[1280];
} __attribute__((__packed__));

long deltime (struct timeval tv)
{
        struct timeval now;
        gettimeofday(&now, NULL);
        return (now.tv_sec - tv.tv_sec)*1000000 + now.tv_usec - tv.tv_usec;
}

int register_signal(int sig, void (*handler)(int))
{
	struct sigaction act;

	bzero(&act, sizeof(act));
	act.sa_handler = handler;

	return sigaction(sig, &act, NULL);
}

void signal_sigalrm(int sig)
{
	printf("check_teredo_server: time out %d sec\n", socket_timeout);
	exit(2);
}

void malformed_packet(double time)
{
	printf("TEREDO_SERVER CRITICAL - Malformed packet|time=%.6fs\n",
		time);
	exit(2);
}

int teredo_check(char *address, char *port)
{
	struct teredo_solicit data;
	struct teredo_advert advert;
	uint8_t src[] = {0xfe,0x80,0,0,0,0,0,0, 0,0,0xff,0xff,0xff,0xff, 0xff,0xfd};
	uint8_t dst[] = {0xff,0x02,0,0,0,0,0,0, 0,0,0,0,0,0,0,0x02};
	struct addrinfo	*addressi;
	struct addrinfo hints;
	int		result = 0;
	int		sock, len, error;
	char    	my_addr[INET_ADDRSTRLEN];
	char    	my_prefix[INET6_ADDRSTRLEN];
	uint64_t	nonce;
	struct timeval	tv;
	double		elapsed_time;
	uint8_t		*ptr;
	uint8_t		*prefix;
	uint8_t		prefix_len;

	srandom(time(NULL));
	nonce = random();

	memset(&data, 0, sizeof(data));
	memset(&advert, 0, sizeof(advert));

	data.auth.proto = htons(0x01);		// Teredo auth header
	data.auth.nonce = nonce;
	data.ip6.ip6_vfc = 0x60;	// IPv6
	data.ip6.ip6_plen = htons(sizeof(data.rs) + sizeof(data.opt) + sizeof(data.lladdr));
	data.ip6.ip6_nxt = IPPROTO_ICMPV6; 
	data.ip6.ip6_hlim = 0xff;
	memcpy(data.ip6.ip6_src.s6_addr, src, sizeof(src));
	memcpy(data.ip6.ip6_dst.s6_addr, dst, sizeof(dst));
	data.rs.nd_rs_hdr.icmp6_type = ND_ROUTER_SOLICIT;
	data.rs.nd_rs_hdr.icmp6_cksum = htons(0x7c27);
	data.opt.nd_opt_type = ND_OPT_SOURCE_LINKADDR;
	data.opt.nd_opt_len = 2;

	sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (sock < 0) {
		printf("socket failed: %s\n", strerror(errno));
		return 3;
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_INET;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_protocol = IPPROTO_UDP;

	error = getaddrinfo(address, port, &hints, &addressi);
	if(error || addressi == NULL)
	{
		printf("getaddrinfo: %s\n", gai_strerror(error));
		return 3;
	}

	gettimeofday (&tv, NULL);
	sendto(sock, &data, sizeof(data), 0, addressi->ai_addr, addressi->ai_addrlen);

	freeaddrinfo(addressi);

	len = recv(sock, &advert, sizeof(advert), 0);

	elapsed_time = ((double) deltime(tv)) / 1.0e6;
	if (critical_time > 0 && elapsed_time > critical_time)
                result = 2;
        else if (warning_time > 0 && elapsed_time > warning_time)
                result = 1;

	if (len < 109) //13(Auth) + 8(Origin) + 40(IP6) + 16(RA) + 32(PI)
		malformed_packet(elapsed_time);

	if (nonce != advert.auth.nonce)
		malformed_packet(elapsed_time);

	prefix = NULL;
	for(ptr = advert.options; ptr - (uint8_t *)&advert <= (len-8) && *(ptr+1) != 0; ptr += (*(ptr+1))*8)
	{
		if (*ptr == ND_OPT_PREFIX_INFORMATION && ptr - (uint8_t *)&advert + 32 <= len)
		{
			prefix = ptr + 16;
			prefix_len = *(ptr+2);
			break;
		}
	}

	if(!prefix)
		malformed_packet(elapsed_time);

	advert.origin.xport = advert.origin.xport ^ 0xffff;
	advert.origin.xaddr = advert.origin.xaddr ^ 0xffffffff;

	inet_ntop(AF_INET, &advert.origin.xaddr, my_addr,
                                                sizeof(my_addr));
	inet_ntop(AF_INET6, prefix, my_prefix,
                                                sizeof(my_prefix));

	printf("TEREDO_SERVER %s - Origin: %s:%d, Prefix: %s/%d|time=%.6fs\n",
			status_codes[result],
			my_addr, ntohs(advert.origin.xport),
			my_prefix, prefix_len,
			elapsed_time);

	close(sock);
	return result;
}

static int process_arguments(int argc, char *argv[]);
int main(int argc, char *argv[])
{
	int	retval;

	process_arguments(argc, argv);

	register_signal(SIGALRM, signal_sigalrm);
	alarm(socket_timeout);
	retval = teredo_check(server_address, server_port);

	return retval;
}

void print_usage(void)
{
	printf("Usage: check_teredo_server -H host [-p <port>] [-w <warning time>] [-c <critical time>]\n");
}

void print_help(void)
{
	print_usage();

	printf("\n");
	printf(	"Checks if teredo server is working properly. It is done by\n"
		"sending Router Solicitation message to teredo server and\n"
		"waiting for appropriate reply\n\n");

	printf(	"    -H, --hostname=ADDRESS  IPv6 address or domain name of teredo server\n");
	printf(	"    -p, --port=INTEGER      Teredo server port (default: 3544)\n");
	printf(	"    -t, --timeout=INTEGER   Seconds to wait for reply (default: 10)\n");
	printf(	"    -w, --warning=DOUBLE    Response time to result in warning status (seconds)\n");
	printf(	"    -c, --critical=DOUBLE   Response time to result in critical status (seconds)\n");

	printf("\nReport bugs to <julius.kriukas@gmail.com>.\n");
}

static int process_arguments(int argc, char *argv[])
{
	int c;
	int option_index = 0;

	static struct option longopts[] = {
		{"hostname", required_argument, 0, 'H'},
		{"port", required_argument, 0, 'p'},
		{"timeout", required_argument, 0, 't'},
		{"warning", required_argument, 0, 'w'},
		{"critical", required_argument, 0, 'c'},
		{"version", no_argument, 0, 'V'},
		{"help", no_argument, 0, 'h'},
		{0, 0, 0, 0}
	};

	while(1) {
		c = getopt_long (argc, argv, "H:p:w:c:t:Vh",
				longopts, &option_index);

		if (c == -1)
			break;

		switch (c)
		{
		case 'H':
			server_address = optarg;
			break;
		case 'p':
			server_port = optarg;
			break;
		case 't':
			socket_timeout = atoi(optarg);
			if (socket_timeout <= 0)
			{
				printf("Timeout interval must be a positive integer\n");
				exit(3);
			}
			break;
		case 'w':
			warning_time = strtod (optarg, NULL);
			break;
		case 'c':
			critical_time = strtod (optarg, NULL);
			break;
		case 'h':
			print_help();
			exit(0);
			break;
		case 'V':
			printf("check_teredo_server 0.1");
			break;
		}
	}

	if(server_address == NULL)
	{
		printf("You must provide a teredo server address\n");
		print_usage();
		exit(3);
	}

	return 0;
}

