#include <iostream>
#include <getopt.h>
#include <string>
#include <modbus/modbus.h>
#include <errno.h>

using namespace std;

void print_help()
{
    cout 
    << "Check ModBus TCP version 0.01 by Roman Suchkov aka Fineson" << endl 
    << "-h  --help	Print this help" << endl 
    << "-H  --ip=	<IP address>" << endl
    << "-p  --port=	[Port number] Default 502" << endl 
    << "-d  --device=	[Device number] Default 1" << endl
    << "-a  --address=	[Address reference] Default 1" << endl
    << "-f  --function=	<Number of functions>" << endl
    << " 		1 - Read coils" << endl 
    << " 		2 - Read input discretes" << endl
    << " 		3 - Read multiple registers" << endl
    << " 		4 - Read input registers" << endl
    << "-w  --warning=	[Warning range]" << endl
    << "-c  --critical=	[Critical range]" << endl
    << "-n  --null	If the query will get zero, return the critical signal" << endl
    << "-N  --not_null	If the query will get no zero, return the critical signal" << endl << endl
    << " Example: ./check_modbt --ip=192.168.1.123 -d 1 -a 13 -f 4 -w 1234 -c 2345" << endl
    << " Example: ./check_modbt --ip=192.168.1.123 -d 1 -a 15 -f 4 -w 2345 -c 1234" << endl
    << " Example: ./check_modbt --ip=192.168.1.234 -d 2 -a 20 -f 2 -n" << endl
    << endl;
}
        
int main(int argc, char **argv)

{

    modbus_t *mb;
    uint8_t tab_reg8[1];
    uint16_t tab_reg16[1];
    int mport = 502; 		// Port number
    int devnum = 1;		// Device number
    int sad = 1;		// Address
    int nf = 0;			// Number of function
    int warn_range = 0;		// Warning range
    int crit_range = 0;		// Critical range
    int nc = 0;			// Null flag
    int nnc = 0;		// No null flag
    int result = 0;		// Result
    int rc;
    int rs;
    int option_index;
    const char* short_options = "hH:p:d:a:f:w:c:nN";
    const struct option long_options[] = {
        {"help",no_argument,NULL,'h'},
        {"ip",required_argument,NULL,'H'},
        {"port",optional_argument,NULL,'p'},
        {"device",optional_argument,NULL,'d'},
        {"address",optional_argument,NULL,'a'},
        {"function",required_argument,NULL,'f'},
        {"warning",optional_argument,NULL,'w'},
        {"critical",optional_argument,NULL,'c'},
        {"null",no_argument,NULL,'n'},
        {"not_null",no_argument,NULL,'N'},
        {NULL,0,NULL,0}
    };
    char *IPad = NULL; 		// IP address

    if (argc < 2) {
	cout << argv[0] << " : Could not parse arguments" << endl;
	print_help(); return 3; };

     while (1) {
	rs=getopt_long(argc,argv,short_options,long_options,&option_index);
        if (rs == -1) break;

        switch(rs){
	    case 'h':
	        print_help(); return 0;
	    case 'H':
		IPad = optarg;
		break;
	    case 'p':
		mport = atoi(optarg);
		break;
	    case 'd':
		devnum = atoi(optarg);
		break;
	    case 'a':
		sad = atoi(optarg);
		break;
	    case 'f':
		nf = atoi(optarg);
		break;
	    case 'w':
		warn_range = atoi(optarg);
		break;
	    case 'c':
		crit_range = atoi(optarg);
		break;
	    case 'n':
		nc = 1;
		break;
	    case 'N':
		nnc = 1;
		break;
	    case '?': default:
                print_help(); exit (3);
                  };
               };
    
    if (IPad == NULL) {
	cout << argv[0] << " : Could not parse IP address" << endl;
	print_help(); return 3; };

    mb = modbus_new_tcp(IPad, mport);
    if (mb == NULL) {
	cout << "Unable to allocate libmodbus context" << endl;
        return 3;
        }
    if (modbus_connect(mb) == -1) {
	cout << "Connection failed: " << modbus_strerror(errno) << endl;
        modbus_free(mb);
        return 3;
        }

    modbus_set_slave(mb,devnum);
    sad --;

    switch (nf)
    {
    case 1: 
	rc = modbus_read_bits(mb, sad, 1, tab_reg8);
        if (rc == -1) {
    	    cout << "Read failed: " << modbus_strerror(errno) << endl;
	    modbus_close(mb);
    	    modbus_free(mb);
    	    return 3;
	    }
	    result = (int)tab_reg8[0];
    break;
    case 2: 
        rc = modbus_read_input_bits(mb, sad, 1, tab_reg8);
	if (rc == -1) {
	    cout << "Read failed: " << modbus_strerror(errno) << endl;
	    modbus_close(mb);
    	    modbus_free(mb);
    	    return 3;
	    }
	    result = (int)tab_reg8[0];
    break;
    case 3: 
        rc = modbus_read_registers(mb, sad, 1, tab_reg16);
	if (rc == -1) {
	    cout << "Read failed: " << modbus_strerror(errno) << endl;
	    modbus_close(mb);
    	    modbus_free(mb);
    	    return 3;
	    }
	    result = tab_reg16[0];
    break; 
    case 4: 
        rc = modbus_read_input_registers(mb, sad, 1, tab_reg16);
	if (rc == -1) {
	    cout << "Read failed: " << modbus_strerror(errno) << endl;
	    modbus_close(mb);
    	    modbus_free(mb);
    	    return 3;
    	    }
	    result = tab_reg16[0];
    break; 
    default:
	cout << "Invalid function number" << endl;
	    modbus_close(mb);
    	    modbus_free(mb);
    	    return 3;
    }

    if ( (nc == 1 && nnc == 1)||(nc == 0 && nnc == 0)) {} else{
    if ( nc == 1 && result == 0 ) { cout << "Result: " << result << endl; return 2; }
    if ( nnc == 1 && result != 0 ) { cout << "Result: " << result << endl; return 2; }
	};
    if ( warn_range < crit_range) {
	if ( result >= crit_range) {
	    cout << "Critical: " << result << endl; return 2;
	}
	else if ( result >= warn_range) {
	    cout << "Warning: " << result << endl; return 1;
	}
	else {
	    cout << "Ok: " << result << endl; return 0;
	}
    }

    if ( warn_range > crit_range) {
	if ( result <= crit_range) {
	    cout << "Critical: " << result << endl; return 2;
	}
	else if ( result <= warn_range) {
	    cout << "Warning: " << result << endl; return 1;
	}
	else {
	    cout << "Ok: " << result << endl; return 0;
	}
    }
    
    cout << "Result: " << result << endl; return 0;

    modbus_close(mb);
    modbus_free(mb);

    cin.sync();
    return 0;
}
