#!/usr/bin/perl

######
# Written by Bart Busschots - Computer Centre, NUI Maynooth.
# Copyright  2012 National University of Ireland Maynooth.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
######

use strict;
use warnings;
use Carp;
use Getopt::Std; # for commandline argument processing
use Sys::Syslog qw(:DEFAULT setlogsock); # for logging to syslog
use DBI; # for MySQL DB connection
use English qw( -no_match_vars );  # for eval error special variable

#
# Process arguments
#

# get the arguments, and support --help and --version
my %flags; # a hashref to store the arguments in
$Getopt::Std::STANDARD_HELP_VERSION = 1;
sub VERSION_MESSAGE {
    print "check_rsyslogdb.pl version 0.1 (released February 2012)\n";
    return;
}
sub HELP_MESSAGE{
    print <<'END_HELP';

A plugin for checking that a remote rsyslog server which is configured to log
to a MySQL database is successfully receiving and storing syslog messages. This
plugin was orignally written and tested in an environment using rsyslog, but
it should work with any syslog implementation that logs to a MySQL database.

This plugin works by sending a test syslog message to the remote server, and
then, after a short timeout, querying the database to verify that the message
was successfully stored.

Required Flags:
-H      The IP address to send the syslog message to.
-u      The username to connect to the DB with.

Optional Flags:
-v      Verbose output.
-s      Syslog socket to use - 'udp', 'tcp', or 'unix' supported. Default is
        'udp'. Note that when 'unix' is used the message will be sent to the
        local syslog deamon running on the Nagios server. It is up to you to
        configure that local deamon to forward the message to the remote server.
        Your installed OS will determine which sockets are availble to you. E.g.
        On OS X 'udp' seems to work best, and on CentOS 5, 'unix'.
-f      Syslog facility - defaults to 'local0'.
-d      Delay between sending the message and testing for it in the DB in
        seconds - defaults to 5.
-h      Database host - defaults to the value of -H.
-n      Database name - defaults to 'syslog'.
-p      Database password - defaults to an empty string.
-t      Database table - defaults to 'syslog'.
-c      Database column that contains the syslog messages. Default is 'msg'.

Exit Codes:
0       SUCCESS - the log was successfully retried from the DB.
2       CRITICAL - the log was not found in the DB.
3       UNKNOWN - the script encountered an error. E.g. could not connect to
        the Database.

END_HELP
    return;
}
my $args_legal = getopts('H:u:vs:f:d:h:n:p:t:c:', \%flags);
unless($args_legal){
    print "ERROR - invalud arguments received\n";
    exit 3;
}

# process required flags
my $ip = $flags{'H'};
unless($ip && $ip =~ m/^\d{1,3}[.]\d{1,3}[.]\d{1,3}[.]\d{1,3}$/sx){
    print "ERROR - invalid IP\n";
    exit 3;
}

my $db_user = $flags{'u'};
unless($db_user){
    print "ERROR - no DB username given\n";
    exit 3;
}

# process optional flags
my $verbose = 0;
if($flags{'v'}){
    $verbose = 1;
    print "INFO - entering verbose mode\n";
}

my $syslog_socket = q{}; # empty string
if($flags{'s'}){
    $syslog_socket = lc $flags{'s'};
}
unless($syslog_socket =~ m/^(unix)|(tcp)|(udp)$/sx){
    $syslog_socket = 'unix';
}
print "INFO - using syslog socket type $syslog_socket\n" if $verbose;

my $syslog_facility = 'local0';
if($flags{'f'}){
    $syslog_facility = $flags{'f'};
}
print "INFO - using syslog facility $syslog_facility\n" if $verbose;

my $delay = 5;
if($flags{'d'} && $flags{'d'} =~ m/^\d+$/sx){
    $delay = $flags{'d'};
}
print "INFO - set delay to $delay seconds\n" if $verbose;

my $db_host = $ip;
if($flags{'h'}){
    $db_host = $flags{'h'};
}
print "INFO - using DB Host $db_host\n" if $verbose;

my $db_name = 'syslog';
if($flags{'n'}){
    $db_name = $flags{'n'};
}
print "INFO - using DB name $db_name\n" if $verbose;

my $db_pass = q{}; # empty string
if($flags{'p'}){
    $db_pass = $flags{'p'};
}
print "INFO - using DB Password $db_pass\n" if $verbose;

my $db_table = 'syslog';
if($flags{'t'}){
    $db_table = $flags{'t'};
}
print "INFO - using DB table $db_table\n" if $verbose;

my $db_col = 'msg';
if($flags{'c'}){
    $db_col = $flags{'c'};
}
print "INFO - using DB column $db_col\n" if $verbose;

#
# try to send the syslog message
#

print "\nPhase 1: Send log message to server\n" if $verbose;

# generate a log message to test for
my $test_message = 'check_rsyslogdb_'.time;
print "\tINFO - Message: $test_message\n" if $verbose;

# open a syslog connection
{
    local $EVAL_ERROR = undef;
    unless(eval{
        if($syslog_socket eq 'udp' || $syslog_socket eq 'tcp'){
            setlogsock($syslog_socket, $ip) or croak 'Failed to connect to syslog server';
        }else{
            setlogsock($syslog_socket) or croak 'Failed to connect to local syslog service';
        }
        openlog('check_rsyslogdb', q{}, $syslog_facility);
        syslog('info', $test_message);
        closelog();
    }){
        print "CRITICAL - failed to send syslog message to server $ip ($EVAL_ERROR)\n";
        exit 2;
    }
}

#
# Pause for what ever timeout is set
#

print "\nPausing for $delay seconds .... " if $verbose;
sleep $delay;
print "DONE\n" if $verbose;

#
# Try retrieve the message from the DB
#

print "\nPhase 2 - Querying the DB for the log\n" if $verbose;

# connect to the DB
my $db_connectstring = "DBI:mysql:database=$db_name;host=$db_host";
print "\tINFO - using connect string: $db_connectstring\n" if $verbose;
print "\tConnecting to DB ... " if $verbose;
my $dbh = DBI->connect("$db_connectstring", "$db_user", "$db_pass", {'RaiseError' => 0});
if($dbh){
    print "DONE\n" if $verbose;
}else{
    print "\n" if $verbose;
    print "ERROR - failed to connect to the DB\n";
    exit 3;
}

# query for our test message
my $test_message_sql = "SELECT count(*) FROM $db_table WHERE $db_col LIKE '\%$test_message\%'";
print "\tINFO - using query: $test_message_sql\n" if $verbose;
my $test_message_sth = $dbh->prepare($test_message_sql);
unless($test_message_sth){
    print "ERROR - failed to prepare DB query\n";
    exit 3;
}
unless($test_message_sth->execute()){
    print "ERROR - failed to execute DB query\n";
    exit 3;
}
my @test_row = $test_message_sth->fetchrow_array();
my $num_matches = $test_row[0];

# test the result and return accordingly
print "\n" if $verbose;
if($num_matches == 1){
    print "SUCCESS - test log sucessfully retrieved from DB\n";
    exit 0;
}else{
    print "CRITICAL - test log not found in DB\n";
    exit 2;
}