<?PHP
/**
 * Copyright (C) 2006-2012 Heiko Henning <greenrover33@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


require_once($GLOBALS['basepath'] . '/class.sms.php');
require_once($GLOBALS['basepath'] . '/class.nagios.php');

/**
 * error handeler callback function with is sending errors by mail
 * @param int $errno
 * @param string $errstr
 * @param string $errfile
 * @param int $errline
 * @return boolean ever true
 */
function mail_error_handler($errno, $errstr, $errfile, $errline) {
    $errorType = array(
        E_ERROR => 'ERROR',
        E_WARNING => 'WARNING',
        E_PARSE => 'PARSING ERROR',
        E_NOTICE => 'NOTICE',
        E_CORE_ERROR => 'CORE ERROR',
        E_CORE_WARNING => 'CORE WARNING',
        E_COMPILE_ERROR => 'COMPILE ERROR',
        E_COMPILE_WARNING => 'COMPILE WARNING',
        E_USER_ERROR => 'USER ERROR',
        E_USER_WARNING => 'USER WARNING',
        E_USER_NOTICE => 'USER NOTICE',
        E_STRICT => 'STRICT NOTICE',
        E_RECOVERABLE_ERROR => 'RECOVERABLE ERROR'
    );
    if (array_key_exists($errno, $errorType)) {
        $err = $errorType[$errno];
    } else {
        $err = 'CAUGHT EXCEPTION';
    }

    // dont spam
    if ($err == 'NOTICE') {
        return true;
    }

    $errMsg = "$err: $errstr in $errfile on line $errline";
    log_msg($errMsg);

    $smtp = (empty($GLOBALS['mail_smtp'])) ? '' : $GLOBALS['mail_smtp'];
    $mailto = (empty($GLOBALS['mail_mailto'])) ? 'h.henning@educa.ch' : $GLOBALS['mail_mailto'];
    $mailfrom = (empty($GLOBALS['mail_mailfrom'])) ? 'h.henning@educa.ch' : $GLOBALS['mail_mailfrom'];
    $subject = (empty($GLOBALS['mail_subject'])) ? 'nagios_sms error' : $GLOBALS['mail_subject'];

    echo $errMsg . "\n";

    @include('Mail.php');

    if (empty($smtp) || !class_exists('Mail')) {
        $headers = "MIME-Version: 1.0\n";
        $headers .= "X-Priority: 3\n";
        $headers .= "X-MSMail-Priority: Normal\n";
        $headers .= "X-Mailer: php\n";
        $headers .= "From: $mailfrom\n";
        $headers .= "Content-Type: text/plain; charset=\"utf-8\"\n";
        $headers .= "Content-Transfer-Encoding: 8bit\n";

        mail($mailto, $subject, $errMsg, trim($headers)
        );
    } else {
        // Use PEAR Mail
        $headers = array();
        $headers['From'] = $mailfrom;
        $headers['To'] = $mailto;
        $headers['Subject'] = $subject;
        $headers['Content-Type'] = "text/plain; charset=UTF-8";

        $smtpinfo = array();
        $smtpinfo['host'] = $smtp;
        $smtpinfo['port'] = '25';
        $smtpinfo['auth'] = false;

        $mail_object = & Mail::factory('smtp', $smtpinfo);

        $mail = $mail_object->send($mailto, $headers, $errMsg);
    }

    /* Don't execute PHP internal error handler */
    return true;
}

set_error_handler('mail_error_handler');

/**
 * nagios sms handler class
 */
class nagios_sms {

    /**
     * location of the configuration file
     */
    const CONFIG_FILE = '/etc/nagios_sms.conf';

    /**
     * configuration options
     * @var array
     */
    protected $config = array();

    /**
     * nagios log parser
     * @var nagios
     */
    protected $nagios = null;

    /**
     * asp sms object
     * @var sms
     */
    protected $sms = null;

    /**
     * filename of the queue sql lite db
     * @var string
     */
    protected $queue_db_file = 'queue.db';

    /**
     * PDO Databse link for the queue
     * @var PDO
     */
    protected $queue = null;

    /**
     * path to directory where to save the last messages
     * @var string
     */
    protected $last_msg_log_path = '';

    /**
     * time after the last status report will be invalid  seconds
     * @var int
     */
    protected $last_msg_cache_lifetime = 3600; // 1h

    /**
     * delay message by this time  seconds
     * @var int
     */
    protected $notification_delay = 180; // 3 min

    /**
     * error log file
     * @var string
     */
    protected $sms_error_log_file = '/var/log/sms-error.log';

    public function __construct() {
        $this->load_config();

        $GLOBALS['mail_smtp'] = $this->config['credits-smtp'];
        $GLOBALS['mail_mailto'] = $this->config['credits-mailto'];
        $GLOBALS['mail_mailfrom'] = $this->config['credits-mailfrom'];
        $GLOBALS['mail_subject'] = $this->config['credits-subject'];

        $this->last_msg_log_path = $GLOBALS['basepath'] . '/last_msgs/';

        $this->sms = new sms(
                $this->config['userkey'],
                $this->config['password']
            );

        $this->nagios = new nagios($this->config['status-file']);


        if (substr($this->queue_db_file, 0, 1) != '/') {
            $this->queue_db_file = $GLOBALS['basepath'] . '/' . $this->queue_db_file;
        }
        if (!is_file($this->queue_db_file)) {
            trigger_error('nagios_sms::__cosntruct can not find queue db file: ' . $this->queue_db_file, E_USER_ERROR);
        }
        $this->queue = new PDO('sqlite:' . $this->queue_db_file);
    }

    public function __destruct() {
        $this->sms = null;
        $this->nagios = null;
        $this->queue = null;
    }

    /**
     * queue sms with status report to host
     * @param string $mobil_nr
     * @param string $host
     */
    public function queue_sms($mobil_nr, $host) {
        log_msg('queue: ' . $mobil_nr . "\t" . $host);

        $params = array(
            ':host' => $host,
            ':mobile_nr' => $mobil_nr,
            ':time' => time(),
        );
        $sql = 'INSERT INTO send_too (host, mobile_nr, time) VALUES (:host, :mobile_nr, :time) ';
        $sth = $this->queue->prepare($sql);
        $sth->execute($params);
    }

    /**
     * send sms with status report to host
     * @param string $host
     */
    public function send_sms($host) {
        $mobile_nrs = array();

        $params = array(':host' => $host);
        $sql = 'SELECT status FROM status_log WHERE host=:host';
        $sth = $this->queue->prepare($sql);
        $sth->execute($params);
        $last_status = @$sth->fetchColumn();


        $params = array(':host' => $host);
        $sql = 'SELECT mobile_nr FROM send_too WHERE host=:host';
        $sth = $this->queue->prepare($sql);
        $sth->execute($params);
        while ($g = $sth->fetch(PDO::FETCH_ASSOC)) {
            // md5, because we dont want send twice to one person
            $mobile_nrs[md5($g['mobile_nr'])] = $g['mobile_nr'];
        }

        // send the message!
        // clean queue we will send now ;-)
        $params = array(':host' => $host);
        $sql = 'DELETE FROM send_too WHERE host=:host';
        $sth = $this->queue->prepare($sql);
        $sth->execute($params);

        // insert/update last status reported
        $params = array(':host' => $host, ':status' => $this->nagios->get_status($host));
        if (empty($last_status)) {
            $sql = 'INSERT INTO status_log (host, status) VALUES (:host, :status)';
        } else {
            $sql = 'UPDATE status_log SET status=:status WHERE host=:host';
        }
        $sth = $this->queue->prepare($sql);
        $sth->execute($params);

        // get the message text
        $txt = $this->nagios->get_status_text($host);

        log_msg('send: ' . $host . "\t" . implode(',', $mobile_nrs) . "\t" . strtr($txt, array("\n" => "\t")));

        $originator = (empty($this->config['userkey'])) ? 'Nagios' : $this->config['originator'];
        $this->sms->setOriginator($originator);
        foreach ($mobile_nrs AS $mobile_nr) {
            $this->sms->addRecipient($mobile_nr);
        }
        $this->sms->setContent($txt);

        $result = $this->sms->sendSMS();

        if ($result != 1) {
            $sms_error_log_file = (substr($this->sms_error_log_file, 0, 1) == '/') ? $this->sms_error_log_file : $GLOBALS['basepath'] . '/' . $this->sms_error_log_file;

            $fh = fopen($sms_error_log_file, 'a+');
            fwrite($fh, implode(', ', $mobile_nrs) . "\t" . $this->sms->getErrorDescription() . "\n");
            fclose($fh);
        }

        $this->sms->showCredits();
        $credits = $this->sms->getCredits();

        if ((INT) $credits < (INT) $this->config['credits-min']) {
            trigger_error('nagios_sms Credit limit reached. Credits left: ' . $credits, E_USER_NOTICE);
        }
    }

    /**
     * parse config file to array $this->config
     */
    protected function load_config() {
        $config_file = $_ENV['HOME'] . '/.nagios_sms.conf';
        if (!is_file($config_file)) {
            $config_file = self::CONFIG_FILE;
        }
        if (!is_file($config_file)) {
            die("No configuration file found.\n");
        }

        $this->config = array();
        preg_match_all('/([-\w]+)\s*=\s*(.+)/', preg_replace('/\#(.*)/', '', file_get_contents($config_file)
                ), $tmp
        );
        foreach ($tmp[1] AS $i => $key) {
            $key = trim($key);
            $val = trim($tmp[2][$i]);

            $this->config[$key] = $val;
        }
    }

    /**
     * delay next action
     */
    public function sleep_for_notification_delay() {
        sleep($this->notification_delay);
    }

    /**
     * get dalay value notification_delay
     * @return int
     */
    public function get_notification_delay() {
        return $this->notification_delay;
    }

    /**
     * get list of host with have an msg in que to send
     * @return array
     */
    public function find_notifications_to_send() {
        $hosts_status = array();
        $sql = 'SELECT status, host  FROM status_log ';
        $sth = $this->queue->prepare($sql);
        $sth->execute();
        while ($g = $sth->fetch(PDO::FETCH_ASSOC)) {
            $hosts_status[$g['host']] = $g['status'];
        }

        $hosts = array();
        $sql = 'SELECT time, host  FROM send_too ';
        $sth = $this->queue->prepare($sql);
        $sth->execute();
        while ($g = $sth->fetch(PDO::FETCH_ASSOC)) {
            if (isset($hosts[$g['host']])) {
                $hosts[$g['host']] = max($hosts[$g['host']], $g['time']);
            } else {
                $hosts[$g['host']] = $g['time'];
            }
        }

        $delay = $this->get_notification_delay();
        foreach ($hosts AS $host => $last_action) {
            if (($last_action + $delay) >= time()) {
                // host notification is still in delay time
                unset($hosts[$host]);
            } elseif ($hosts_status[$host]=='OK' && $this->nagios->get_status($host)=='OK') {
                // not send anny messages if host is OK and cause of the delay no error messages was send
                unset($hosts[$host]);
            }
        }

        return array_keys($hosts);
    }
}

/**
 * write message to log file
 * @param string $msg
 */
function log_msg($msg) {
    $sms_error_log_file = $GLOBALS['basepath'] . '/msg.log';

    $fh = fopen($sms_error_log_file, 'a+');
    fwrite($fh, strftime('%Y-%m-%d %H:%M:%S', time()) . "\t" . $msg . "\n");
    fclose($fh);
}

?>