#!/usr/bin/perl -w 
############################## check_snmp_dca #################
# Date : May 1, 2012
# Author  : EMC Corporation
# Licence : BSD
# Changelog : Initial Version
# Contributors : Based on existing code from many others 
# Description : Query the EMC Greenplum Data Computing Appliance (DCA) 
#               SNMP MIB
#################################################################
#
# Copyright (c) 2012,EMC Corporation
# All rights reserved.
# Redistribution and use in source and binary forms, with or without modification, 
# are permitted provided that the following conditions are met:
#   o Redistributions of source code must retain the above copyright notice, 
#     this list of conditions and the following disclaimer.
#   o Redistributions in binary form must reproduce the above copyright notice, 
#     this list of conditions and the following disclaimer in the documentation and/or 
#     other materials provided with the distribution.
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS 
# OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
# AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR 
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# Help : ./check_snmp_dca.pl -h
#

use strict;
use Net::SNMP;
use Getopt::Long;
use feature ":5.10";

# Nagios specific

use lib "/usr/local/nagios/libexec";
use utils qw(%ERRORS $TIMEOUT);
#my $TIMEOUT = 15;
#my %ERRORS=('OK'=>0,'WARNING'=>1,'CRITICAL'=>2,'UNKNOWN'=>3,'DEPENDENT'=>4);
my %resp_codes=(1=>'Normal',2=>'Warning',3=>'Error',4=>'Unreachable',5=>'Unknown');
my %resp_states=(1=>'OK',2=>'WARNING',3=>'CRITICAL',4=>'UNKNOWN',5=>'UNKNOWN');

# SNMP Data

# Generic with host-ressource-mib
my $base_dca = "1.3.6.1.4.1.1139.23.1.1.2";   # base oid for DCA MIB

# Version 

my $gpDCAv1Hardware = $base_dca . ".1";   
my $gpDCAv2Hardware = $base_dca . ".2";   
my $gpDCAServices = $base_dca . ".3";   
my $gpDCAIsoVersion = $base_dca . ".4";   
my $gpDCAHadoopVersion = $base_dca . ".5";   

# The following variables help us construct the actual OIDs later on in the program.
# These are the leaf parts of the OIDs which are the same whether you are querying Master, Segment
# or whatever. 

my $gpNodeTable = ".1";

my $gpNodeHostName = $gpNodeTable . ".1.2";
my $gpNodePowerSupply = $gpNodeTable . ".1.3";
my $gpNodeBattery = $gpNodeTable . ".1.4";
my $gpNodeCoolingDevice = $gpNodeTable . ".1.5";
my $gpNodeProcessorDevice = $gpNodeTable . ".1.6";
my $gpNodeCacheDevice = $gpNodeTable . ".1.7";
my $gpNodeOsMemory = $gpNodeTable . ".1.8";
my $gpNodeMemoryDevice = $gpNodeTable . ".1.9";
my $gpNodeNetworkDevice = $gpNodeTable . ".1.10";
my $gpNodeController = $gpNodeTable . ".1.11";
my $gpNodeControllerBattery = $gpNodeTable . ".1.12";
my $gpNodeVirtualDisk = $gpNodeTable . ".1.13";
my $gpNodeDiskUsed = $gpNodeTable . ".1.14";
my $gpNodeDiskUsedPercentage = $gpNodeTable . ".1.15";
my $gpNodeArrayDisk = $gpNodeTable . ".1.16";
my $gpNodeSystemWatts = $gpNodeTable . ".1.17";
my $gpNodeSystemTemperature = $gpNodeTable . ".1.18";
my $gpNodeSystemTemperatureStatus = $gpNodeTable . ".1.19";
my $gpNodeCoreFileStatus = $gpNodeTable . ".1.20";
my $gpNodeLastCoreFileTimestamp = $gpNodeTable . ".1.21";
my $gpNodeCrashFileStatus = $gpNodeTable . ".1.22";
my $gpNodeLastCrashFileTimestamp = $gpNodeTable . ".1.23";
my $gpNodeOverallStatus = $gpNodeTable . ".1.24";
my $gpNodeAvailableRam = $gpNodeTable . ".1.25";
my $gpNodeTotalRam = $gpNodeTable . ".1.26";
my $gpNodeAvailableSwap = $gpNodeTable . ".1.27";
my $gpNodeTotalSwap = $gpNodeTable . ".1.28";
my $gpNodeSystemCpuPct = $gpNodeTable . ".1.29";
my $gpNodeUserCpuPct = $gpNodeTable . ".1.30";
my $gpNodeIdleCpuPct = $gpNodeTable . ".1.31";
my $gpNodeTotalBytesRead = $gpNodeTable . ".1.32";
my $gpNodeTotalBytesWritten = $gpNodeTable . ".1.33";
my $gpNodeTotalReadAccesses = $gpNodeTable . ".1.34";
my $gpNodeTotalWriteAccesses = $gpNodeTable . ".1.34";

# DB Service

my $gpDbServiceTable = $gpDCAServices . ".1.1";

my $gpDbVersion = $gpDbServiceTable . ".1.2";
my $gpDbPrimarySegCount = $gpDbServiceTable . ".1.3";
my $gpDbMirrorSegCount = $gpDbServiceTable . ".1.4";
my $gpDbConnStatus = $gpDbServiceTable . ".1.5";
my $gpDbConnStatusDesc = $gpDbServiceTable . ".1.6";
my $gpDbSegmentDownStatus = $gpDbServiceTable . ".1.7";
my $gpDbSegmentDownCount = $gpDbServiceTable . ".1.8";
my $gpDbChangeTrackingStatus = $gpDbServiceTable . ".1.9";
my $gpDbChangeTrackingCount = $gpDbServiceTable . ".1.10";
my $gpDbResyncModeStatus = $gpDbServiceTable . ".1.11";
my $gpDbResyncModeCount = $gpDbServiceTable . ".1.12";
my $gpDbRoleChangeStatus = $gpDbServiceTable . ".1.13";
my $gpDbRoleChangeCount = $gpDbServiceTable . ".1.14";
my $gpDbMasterMirrorSyncStatus = $gpDbServiceTable . ".1.15";
my $gpDbMasterMirrorSyncStatusDesc = $gpDbServiceTable . ".1.16";


# Admin Switch Values
my $gpAdminSwitchHost = $gpNodeTable . ".1.2";
my $gpAdminSwitchInterface = $gpNodeTable . ".1.3";
my $gpAdminSwitchOverallStatus = $gpNodeTable . ".1.4";

# Interconnect Switch Values
my $gpInterconnectSwitchHost = $gpNodeTable . ".1.2";
my $gpInterconnectOperationalStatus = $gpNodeTable . ".1.3";
my $gpInterconnectOperationalStatusDesc = $gpNodeTable . ".1.4";
my $gpInterconnectSensor = $gpNodeTable . ".1.5";
my $gpInterconnectInterface = $gpNodeTable . ".1.6";
my $gpInterconnectOverallStatus = $gpNodeTable . ".1.7";

# Globals

my $Version='1.0.0';

my $o_host = 	undef; 		# hostname
my $o_community = "public"; 	# community
my $o_port = 	161; 		# port
my $o_help=	undef; 		# want some help ?
my $o_verb=	undef;		# verbose mode
my $o_crit=	undef;		# 
my $o_warn=	undef;		# 
my $o_version=	undef;		# print version
my $o_nodeType=	undef;		# Node Type (Master, Segment, Name Node, etc.)
my $o_nodeIndex=undef;		# Node Index (1..x)
my $o_object=undef;		# The item to get
# End compatibility

# functions

sub p_version { print "check_snmp_dca version : $Version\n"; }

sub print_usage {
    print "Usage: $0 [-v] -H <host> [-C <snmp community>] [-p <port>] -o <Object> -n <node type> -i <table index> [-w <warn level> -c <crit level>] [-V]\n";
   print <<EOT;
-o, --object 
   A quoted string that indicates what information to query from the DCA. It roughly corresponds to OIDs in the MIB 
   but can include several, related OIDs that will be queried as a group. An example might be "Core File Status".
   This actually causes two OIDs to be queried - Core File Status and Last Core File Timestamp. Those results are combined.
   These are essentially generic across multiple node types - Master, Standby, Segmentxx, etc. The specific node is specified
   in the node-type and node-index params.
-n, --node-type
   Indicates whether we want information from the Master, Segment, DIA, Admin Switch, etc. This gets us to the correct table
   in the MIB. Note that the standby is not a separate node-type, it is a Master node. See the node-index param.
-i, --node-index
   Indicates which entry in a MIB table we want to use. Most of the OIDs are in tables which have multiple rows. For example,
   the Master table has two "rows" - one for the Master and one for the Standby master. The Segment table has as many rows
   as there are Segments in the DCA. 
-w, --warning=value,value 
   warning level for various objects
-c, --critical=value,value
   critical level for various objects
-C, --community=COMMUNITY NAME
   community name for the host's SNMP agent. This defaults to 'public'
-p, --port=PORT
   SNMP port (Default 161)

Note that not all the objects will take warning and critical thresholds. Many objects simply look at the status value 
to indicate the state.
EOT
}

sub isnnum { # Return true if arg is not a number
  my $num = shift;
  if ( $num =~ /^(\d+\.?\d*)|(^\.\d+)$/ ) { return 0 ;}
  return 1;
}


# For verbose output
sub verb { my $t=shift; print $t,"\n" if defined($o_verb) ; }

sub check_options {
    Getopt::Long::Configure ("bundling");
    GetOptions(
        'v'     => \$o_verb,            'verbose'       => \$o_verb,
        'h'     => \$o_help,            'help'          => \$o_help,
        'H:s'   => \$o_host,            'hostname:s'    => \$o_host,
        'p:i'   => \$o_port,            'port:i'        => \$o_port,
        'C:s'   => \$o_community,       'community:s'   => \$o_community,
        'V'     => \$o_version,         'version'       => \$o_version,
        'n:s'     => \$o_nodeType,         'node-type'       => \$o_nodeType,
        'i:i'     => \$o_nodeIndex,         'node-index'       => \$o_nodeIndex,
        'o:s'     => \$o_object,         'object'       => \$o_object,
        'c:s'   => \$o_crit,            'critical:s'    => \$o_crit,
        'w:s'   => \$o_warn,            'warn:s'        => \$o_warn,
        );

    if (!defined($o_host) || !defined($o_object)) 
       {print_usage(); exit $ERRORS{"UNKNOWN"}}
}
########## MAIN #######

check_options();


$SIG{'ALRM'} = sub {
 print "No answer from host\n";
 exit $ERRORS{"UNKNOWN"};
};

# Connect to host
my ($session,$error);
# SNMPv2 Login
verb("SNMP v2c login");
($session, $error) = Net::SNMP->session(
	-hostname  => $o_host,
	-version   => 2,
	-community => $o_community,
	-port      => $o_port,
	-maxmsgsize      => 65000,
);

if (!defined($session)) {
   printf("ERROR opening session: %s.\n", $error);
   exit $ERRORS{"UNKNOWN"};
}

my $exit_val=3;
my @req_oids;
my @items;
my $result;
my $value;
my $desc;
my $state;
my $sets;
my $check_above_threshold;
my $nodeType;

# Now call different SNMP query routines based on the OID we want.

# Use the node-type parameter to index into the proper table

if ($o_nodeType ne '') {
   given ($o_nodeType) {
      when("Master") {$nodeType = ".1";}
      when("Segment") {$nodeType = ".2";}
      when("Admin") {$nodeType = ".3";}
      when("Int") {$nodeType = ".4";}
      when("ETL") {$nodeType = ".5";}
      when("HDMaster") {$nodeType = ".6";}
      when("HDWorker") {$nodeType = ".7";}
      when("AggAdmin") {$nodeType = ".8";}
      when("AggInt") {$nodeType = ".9";}
      default {print("Unrecognized Node Type: $o_nodeType\n"); $exit_val = $ERRORS{"UNKNOWN"};}
   }
}

given ($o_object) {

   when("RAM") {
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeTotalRam . "." . $o_nodeIndex);
      push (@items,"Total");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeAvailableRam . "." . $o_nodeIndex);
      push (@items,"Available");
      $check_above_threshold = 0;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }
   when("Swap"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeTotalSwap . "." . $o_nodeIndex);
      push (@items,"Total");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeAvailableSwap . "." . $o_nodeIndex);
      push (@items,"Available");
      $check_above_threshold = 0;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }
   when("Bytes R/W"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeTotalBytesRead . "." . $o_nodeIndex);
      push (@items,"Total Read");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeTotalBytesWritten . "." . $o_nodeIndex);
      push (@items,"Total Written");
      $check_above_threshold = 1;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }
   when("Accesses R/W"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeTotalReadAccesses . "." . $o_nodeIndex);
      push (@items,"Total Read");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeTotalWriteAccesses . "." . $o_nodeIndex);
      push (@items,"Total Written");
      $check_above_threshold = 1;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }
   when("CPU Pct"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeSystemCpuPct . "." . $o_nodeIndex);
      push (@items,"System");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeUserCpuPct . "." . $o_nodeIndex);
      push (@items,"User");
      $check_above_threshold = 1;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }

   when("System Watts" ) {
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeSystemWatts . "." . $o_nodeIndex);
      push (@items,"Watts");
      $check_above_threshold = 1;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }
   when("Temperature" ) {
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeSystemTemperature . "." . $o_nodeIndex);
      push (@items,"Degrees");
      $check_above_threshold = 1;
      $exit_val = checkValues($session,$check_above_threshold,$o_warn,$o_crit,\@req_oids,\@items);
      }
   when("DB Version"){
      push (@req_oids, $gpDbVersion . ".1");
      push (@items,"DB Version");
      $exit_val = getValues($session,\@req_oids,\@items);
      }
   when("Segment Counts") {
      push (@req_oids, $gpDbPrimarySegCount . ".1");
      push (@items,"Primary");
      push (@req_oids, $gpDbMirrorSegCount . ".1");
      push (@items,"Mirror");
      $exit_val = getValues($session,\@req_oids,\@items);
      } 
   when("Overall Status") {
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeOverallStatus . "." . $o_nodeIndex);
      push (@items,"Status");
      $exit_val = getStatus($session,\@req_oids,\@items);
   }
   when("Temperature Status") {
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeSystemTemperatureStatus . "." . $o_nodeIndex);
      push (@items,"Degrees");
      $exit_val = getStatus($session,\@req_oids,\@items);
   }
   when("Core File Status") {
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeCoreFileStatus . "." . $o_nodeIndex);
      push (@items, "Status");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeLastCoreFileTimestamp . "." . $o_nodeIndex);
      push (@items, "Timestamp");
      $exit_val = getStatus($session,\@req_oids,\@items);
      } 
   when("Crash File Status"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeCrashFileStatus . "." . $o_nodeIndex);
      push (@items, "Status");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpNodeLastCrashFileTimestamp . "." . $o_nodeIndex);
      push (@items, "Timestamp");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when("Connection Status") {
      push (@req_oids, $gpDbConnStatus . ".1");
      push (@items, "Status");
      push (@req_oids, $gpDbConnStatusDesc . ".1");
      push (@items, "Desc");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when("Segment Down Status"){
      push (@req_oids, $gpDbSegmentDownStatus . ".1");
      push (@items, "Status");
      push (@req_oids, $gpDbSegmentDownCount . ".1");
      push (@items, "Count");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when ("Change Tracking Status"){
      push (@req_oids, $gpDbChangeTrackingStatus . ".1");
      push (@items, "Status");
      push (@req_oids, $gpDbChangeTrackingCount . ".1");
      push (@items, "Count");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when ("Resync Mode Status"){
      push (@req_oids, $gpDbResyncModeStatus . ".1");
      push (@items, "Status");
      push (@req_oids, $gpDbResyncModeCount . ".1");
      push (@items, "Count");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when ("Role Change Status"){
      push (@req_oids, $gpDbRoleChangeStatus . ".1");
      push (@items, "Status");
      push (@req_oids, $gpDbRoleChangeCount . ".1");
      push (@items, "Count");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when("Master Mirror Sync Status"){
      push (@req_oids, $gpDbMasterMirrorSyncStatus . ".1");
      push (@items, "Status");
      push (@req_oids, $gpDbMasterMirrorSyncStatusDesc . ".1");
      push (@items, "Desc");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when("Battery"){
      push (@items, "Status");
      push (@items, "Desc");
      $exit_val = getCompositeStatus($session,1,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeBattery,\@items);
      }
   when("Power Supply"){
      push (@items, "Status");
      push (@items, "State");
      push (@items, "Amps");
      push (@items, "Volts");
      $exit_val = getCompositeStatus($session,2,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodePowerSupply,\@items);
      }
   when("Cooling Device"){
      push (@items, "Status");
      push (@items, "Desc");
      $sets = $o_nodeType eq "Master" ? 12 : 10;
      $exit_val = getCompositeStatus($session,$sets,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeCoolingDevice,\@items);
      }
   when("Processor Device"){
      push (@items, "Status");
      push (@items, "Desc");
      $exit_val = getCompositeStatus($session,2,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeProcessorDevice,\@items);
      }
   when("Cache Device"){
      push (@items, "Status");
      push (@items, "Desc");
      push (@items, "Size");
      $exit_val = getCompositeStatus($session,6,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeCacheDevice,\@items);
      }
   when("OS Memory"){
      push (@items, "Status");
      push (@items, "Desc");
      $exit_val = getCompositeStatus($session,1,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeCacheDevice,\@items);
      }
   when("Memory Device"){
      push (@items, "Status");
      push (@items, "Desc");
      $exit_val = getCompositeStatus($session,6,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeCacheDevice,\@items);
      }
   when("Network Device"){
      push (@items, "Status");
      push (@items, "Desc");
      $sets = $o_nodeType eq "Master" ? 6 : 4;
      $exit_val = getCompositeStatus($session,$sets,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeNetworkDevice,\@items);
      }
   when("Controller"){
      push (@items, "Status");
      push (@items, "Desc");

      # Name is defined in the MIB but doesn't seem to come back.
      push (@items, "Name");
      $exit_val = getCompositeStatus($session,1,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeController,\@items);
      }
   when("Controller Battery"){
      push (@items, "Status");
      push (@items, "Desc");
      push (@items, "State");
      push (@items, "Desc");
      $exit_val = getCompositeStatus($session,1,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeControllerBattery,\@items);
      }
   when("Virtual Disk") {
      push (@items,"Status");
      push (@items,"Desc");
      push (@items,"Write Policy Status");
      push (@items,"Write Policy Desc");
      push (@items,"Read Policy Status");
      push (@items,"Read Policy Desc");
      push (@items,"State Status");
      push (@items,"State Desc");
      $exit_val = getCompositeStatus($session,3,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeVirtualDisk,\@items);
      } 
# Something seems wrong with the MIB for Master Disk Used. There is only one entry and it looks like
# it's for the second disk. This doesn't really seem to work correctly.
   when("Disk Used") {
      push (@items,"Status");
      push (@items,"Value");
      $exit_val = getCompositeStatus($session,1,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeDiskUsed,\@items);
      } 
   when("Disk Used Percentage") {
      push (@items,"Status");
      push (@items,"Value");
      $exit_val = getCompositeStatus($session,2,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeDiskUsedPercentage,\@items);
      } 
   when("Array Disk") {
      push (@items,"Status");
      push (@items,"Desc");
      push (@items,"Name");
      $exit_val = getCompositeStatus($session,6,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpNodeArrayDisk,\@items);
      } 

   when("Admin Switch Status"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpAdminSwitchOverallStatus . "." . $o_nodeIndex);
      push (@items, "Status");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpAdminSwitchHost . "." . $o_nodeIndex);
      push (@items, "Host");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when("Admin Switch Interface") {
      push (@items,"Status");
      push (@items,"State");
      push (@items,"Desc");
      $exit_val = getCompositeStatus($session,28,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpAdminSwitchInterface,\@items);
      } 

   when("Interconnect Switch Status"){
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpInterconnectOverallStatus . "." . $o_nodeIndex);
      push (@items, "Status");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpInterconnectSwitchHost . "." . $o_nodeIndex);
      push (@items, "Host");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpInterconnectOperationalStatus . "." . $o_nodeIndex);
      push (@items, "Operational Status");
      push (@req_oids, $gpDCAv1Hardware . $nodeType . $gpInterconnectOperationalStatusDesc . "." . $o_nodeIndex);
      push (@items, "Operational Status Desc");
      $exit_val = getStatus($session,\@req_oids,\@items);
      }
   when("Interconnect Interface") {
      push (@items,"Status");
      push (@items,"State");
      push (@items,"Desc");
      $exit_val = getCompositeStatus($session,62,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpInterconnectInterface,\@items);
      } 
   when("Interconnect Sensor") {
      push (@items,"Status");
      push (@items,"Desc");
      $exit_val = getCompositeStatus($session,9,$o_nodeIndex,$gpDCAv1Hardware . $nodeType . $gpInterconnectSensor,\@items);
      } 
   default { print("Unrecognized Object: $o_object\n"); $exit_val = $ERRORS{"UNKNOWN"};}
}

$session->close();
exit $exit_val;

##########################################################################

# Used for OIDS that return values that need to be compared against a threshold
sub checkValues {
   
   my($session,$check_above_threshold,$o_warn,$o_crit,$req_oids_ref,$items_ref) = @_;
   my(@req_oids) = @$req_oids_ref;
   my(@items) = @$items_ref;

   $result = $session->get_request(-varbindlist => \@req_oids,);

   if (!defined $result) {
      printf "ERROR: %s.\n", $session->error();
      $session->close();
      exit 1;
   }
   # Warning and Critical thresholds may be ganged. Split them into an array if so
   my @w_vals = split(',',$o_warn);
   my @c_vals = split(',',$o_crit);
   my $crit_count = 0;
   my $warn_count = 0;

   $state = 'OK';
   # Loop through the returned values and check against the threshold
   for (my $i=0; $i<= $#w_vals; $i++) {

      $value = $result->{$req_oids[$i]};
      # Check if the value returned is not a number (such as "No data available for this instance")
      if ( isnnum($value)) {
	 # We'll set this to OK but maybe it should be Warning
         $state = 'UNKNOWN';
         $desc .= "$items[$i]:$value ";
      # Some thresholds are "don't go above" and some are "don't go below". Figure out which way to compare
      } elsif ($check_above_threshold) {
         if ( $value > $c_vals[$i] ) {
            $crit_count++;
            $desc .= "$items[$i]:$value > $c_vals[$i] ";
         } elsif ( $value > $w_vals[$i] ) {
            $warn_count++;
            $desc .= "$items[$i]:$value > $w_vals[$i] ";
         } else {
            $state = 'OK';
            $desc .= "$items[$i]:$value ";
         }
      } else {
         if ( $value < $c_vals[$i] ) {
            $crit_count++;
            $desc .= "$items[$i]:$value < $c_vals[$i] ";
         } elsif ( $value < $w_vals[$i] ) {
            $warn_count++;
            $desc .= "$items[$i]:$value < $w_vals[$i] ";
         } else {
            $state = 'OK';
            $desc .= "$items[$i]:$value ";
         }
      }
   }
   $state = 'WARNING' if ($warn_count) ;
   $state = 'CRITICAL' if ($crit_count) ;
   printf "%s - %s: %s\n", $state,$o_object,$desc;
   return $ERRORS{$state};
}

##########################################################################
# Used for OIDS that return a status codes (as opposed to just a number or string to show)
sub getStatus {
   
   my($session,$req_oids_ref,$items_ref) = @_;
   my(@req_oids) = @$req_oids_ref;
   my(@items) = @$items_ref;

   $result = $session->get_request(-varbindlist => \@req_oids,);

   if (!defined $result) {
      printf "ERROR: %s.\n", $session->error();
      $session->close();
      exit 1;
   }

   $value = $result->{$req_oids[0]};

   # If we get text back for the status, it means that there is no info for this entry. Set to unknown
   if (isnnum($value)) {$value = 5;}

   $state = $resp_states{$value};
   $desc = "$state - $o_object: $resp_codes{$value}";
   # Some OIDs have status descriptions as well. Get that if appropriate
   if ($#req_oids > 0) {
      $desc .= ", $items[1]: $result->{$req_oids[1]}";
   }
   print "$desc\n";
   return $ERRORS{$state};

}

##########################################################################
# Used for OIDS that return just a value (or values) and not a status code
sub getValues {
   
   my($session,$req_oids_ref,$items_ref) = @_;
   my(@req_oids) = @$req_oids_ref;
   my(@items) = @$items_ref;

   $result = $session->get_request(-varbindlist => \@req_oids,);

   if (!defined $result) {
      printf "ERROR: %s.\n", $session->error();
      $session->close();
      exit 1;
   }

   $exit_val=$ERRORS{'OK'};
   for (my $i=0; $i<= $#req_oids; $i++) {
      $value = $result->{$req_oids[$i]};
      $desc .= "$items[$i]:$value ";
   }

   printf "%s - %s: %s\n", 'OK',$o_object,$desc;
   return $exit_val;

}

##########################################################################
# Used for querying multiple OIDS in a nested structure and forming a composite status 
sub getCompositeStatus {
   
   my($session,$parentElements,$o_nodeIndex,$req_oid,$items_ref) = @_;
   my(@req_oids) = ();
   my(@items) = @$items_ref;
   my($childElements) = $#items + 1;

   for (my $j=1; $j<=$parentElements;$j++) {
      # All of the "status" structures in the MIB start with a status value followed by the "other" information.
      # The Interconnect Interface is the only one where the description comes before the status. We are going to 
      # ask for them in the "correct" order so that we can parse the status in a general way for all the MIBs below.
      # 
      if ($o_object eq "Interconnect Interface") {
         push(@req_oids,$req_oid . ".$j.2." . $o_nodeIndex);
         push(@req_oids,$req_oid . ".$j.3." . $o_nodeIndex);
         push(@req_oids,$req_oid . ".$j.1." . $o_nodeIndex);
      } else {
         for (my $i=1; $i<=$childElements; $i++) {
            push(@req_oids,$req_oid . ".$j.$i." . $o_nodeIndex);
         }
      }
   }
   
   # Ask for all the OIDs
   $result = $session->get_request(-varbindlist => \@req_oids,);

   # Check for errors
   if (!defined $result) {
      printf "ERROR: %s.\n", $session->error();
      $session->close();
      exit 1;
   }

   # Assume the composite status is 1 = Normal
   my $compositeStatus = 1;
   $desc = "";
   # Loop through the returned values and display them
   # Keep track of which group of values we are in so we can show it in the description
   my $set = 0;
   for (my $i=0; $i<= $#req_oids; $i++) {
      # If this is the first element of the child set, it's the status. Check it
      if (($i % $childElements) == 0) {
	 $set++;
         $value = $result->{$req_oids[$i]};

         # If we get text back for the status, it means that there is no info for this entry. Set to warn
         if (isnnum($value)) {$value = 2;}

	 # If it's something other than ok, record it 
	 if ($value > 1){ 
	    # Set the max status we've reached overall
	    $compositeStatus = $value unless ($value < $compositeStatus);
	 }
      } else {
	 # For all the other values, just record them in the description field
	 $desc .= "\n$items[$i % $childElements]($set):$result->{$req_oids[$i]}";
      }
   }

   # Calculate the overall status
   $state = $resp_states{$compositeStatus};
   $desc = "$state - $o_object $desc";
   print "$desc\n";
   return $ERRORS{$state};
}


