#!/usr/bin/perl -w
#
#=============================== SUMMARY =======================================
#
# Program : check_ups_powerware.pl
# Version : 11-15-2012
# Date    : Nov 15 2012
# Author  : Bryan Tolka, Cord Scott, Tom Greaser (WVU) btolka@hsc.wvu.edu
#	  : Code is based on the Dell UPS Wizards by Troy Lea
# Summary : This is a Nagios XI wizard/plugin that checks the status of a Powerware UPS 
#           via a network mananagement card. Specifically this is for the UPS
#			devices manufactured by Eaton.
# Copyright: Bryan Tolka, Cord Scott, Tom Greaser (WVU) btolka@hsc.wvu.edu 2012
#
#
#============================== SETUP NOTES ====================================
#
# Copy this file to your Nagios installation folder in "libexec/". 
#
# You must have enabled SNMP on the UPS and define the community. 
# On the server that will be running the plugin you must have the perl
# "Net::SNMP" module installed.
#
# perl -MCPAN -e shell
# cpan> install Net::SNMP
#
# For SNMPv3 support make sure you have the following modules installed:
#   Crypt::DES, Digest::MD5, Digest::SHA1, and Digest::HMAC
#
#========================== SETUP EXAMPLES =====================================
#
# define command{
#       command_name    check_ups_powerware
#       command_line    $USER1$/check_ups_powerware.pl -H $HOSTADDRESS$ -C $ARG1$ -T $ARG2$ $ARG3$ $ARG4$ $ARG5$ $ARG6$ $ARG7$ $ARG8$
#       }
#
# define service{
#       use                     generic-service
#       host_name               UPS01
#       service_description     Global Status
#       check_command           check_ups_powerware!public!global_status
#       normal_check_interval   3
#       retry_check_interval    1
#       }
#
# define service{
#       use                     generic-service
#       host_name               UPS01
#       service_description     Battery Remaining
#       check_command           check_ups_powerware!public!battery_remaining!-U m!-w 15!-c 8
#       normal_check_interval   3
#       retry_check_interval    1
#       }
#
#================================= REVISION ====================================
#
# Version 2011-03-13
# The first version released. Includes the following checks: battery_monitoring_status, 
# battery_remaining, battery_test_status, firmware_version, global_status, 
# model, output_voltage, output_load, output_current,output_freq,input_voltage,input_freq
# serial_number
#
#
# Version 2011-03-26
# * Updated command example to include $ARG3$ to $ARG8$
# * Updated battery_monitoring_status check to report a CRITICAL status if traps
# 11, 13, 15 or 57 are present (alerting that the UPS is either over/under AC Voltage
# OR Input Frequency is out of range OR UPS is on battery) (if 57 is present it will NOT
# report a CRITICAL status if 61 is ALSO present)
# * Checking for the Net::SNMP Library occurs first and will report an error
# when not in verbose mode
# * Added alarm_status check
# * Added more information in the help text
#
#
# Version 2012-04-23
# * Added firmware_version_nic check that will report the Network Management Card 
# Firmware Version
#
# Version 11-15-2012
# Converted this plugin to work with powerware/eaton style ups.  added support for multi
# outputs and inputs
#=========================== START OF PROGRAM CODE =============================
use Getopt::Long;

use strict;
my $Version='11-15-2012';

my @unit_dependant_checks = ('battery_remaining');
my @unit_types = ('h','m','s');
my $unit_type_check;
my $unit_type_defined;
my $perf_data;
my $perf_data1;
my %check_types = (
	"alarm_status" => [
		'alarmstatus'
	],
	"battery_monitoring_status" => [
		'batteryABMstatus'
	],
	"battery_remaining" => [
		'batterysecondsremaining'
	],
	"battery_test_status" => [
		'batteryteststatus'
	],
	"firmware_version" => [
		'version'
	],
	"firmware_version_nic" => [
		'nicfirmware'
	],
	"global_status" => [
		'productStatusGlobalStatus'
	],
	"model" => [
		'model'
	],
	"output_voltage" => [
		'outputvoltage',
		'outputvoltage2',
		'outputvoltage3'
	],

	"output_load" => [
		'outputload',	
		'outputload2',
                'outputload3'
	],
	
	"output_current" => [
		'outputcurrent',
		'outputcurrent2',
		'outputcurrent3'
	],
	"output_freq" => [
                'outputfreq'
        ],
	"input_current" => [
                'inputcurrent',
                'inputcurrent2',
                'inputcurrent3'
        ],
	"input_voltage" => [
                'inputvoltage',
		'inputvoltage2',
		'inputvoltage3'
        ],
	"input_freq" => [
                'inputfreq'
        ],
	"serial_number" => [
		'serialnumber'
	],


);

my %ups_oids = (


###   UPS-MIB ####
	'nicfirmware'=>'.1.3.6.1.2.1.33.1.1.4.0',
	'model'=>'.1.3.6.1.2.1.33.1.1.2.0',
	'version'=>'.1.3.6.1.2.1.33.1.1.3.0',
	'serialnumber'=>'.1.3.6.1.2.1.33.1.1.5.0',
	'inputvoltage'=>'.1.3.6.1.2.1.33.1.3.3.1.3.1',
        'inputvoltage2'=>'.1.3.6.1.2.1.33.1.3.3.1.3.2',
        'inputvoltage3'=>'.1.3.6.1.2.1.33.1.3.3.1.3.3',
	'inputfreq'=>'.1.3.6.1.2.1.33.1.3.3.1.2.1',
	'inputcurrent'=>'.1.3.6.1.2.1.33.1.3.3.1.4.1',
        'inputcurrent2'=>'.1.3.6.1.2.1.33.1.3.3.1.4.2',
	'inputcurrent3'=>'.1.3.6.1.2.1.33.1.3.3.1.4.3',	
        'outputcurrent'=>'.1.3.6.1.2.1.33.1.4.4.1.3.1',
      	'outputcurrent2'=>'.1.3.6.1.2.1.33.1.4.4.1.3.2',
	'outputcurrent3'=>'.1.3.6.1.2.1.33.1.4.4.1.3.3',
	'outputload'=>'.1.3.6.1.2.1.33.1.4.4.1.5.1',
	'outputload2'=>'.1.3.6.1.2.1.33.1.4.4.1.5.2',
	'outputload3'=>'.1.3.6.1.2.1.33.1.4.4.1.5.3',
        'outputfreq'=>'.1.3.6.1.2.1.33.1.4.2.0',
        'outputvoltage'=>'.1.3.6.1.2.1.33.1.4.4.1.2.1',
        'outputvoltage2'=>'.1.3.6.1.2.1.33.1.4.4.1.2.2',
        'outputvoltage3'=>'.1.3.6.1.2.1.33.1.4.4.1.2.3',
        'alarmstatus'=>'.1.3.6.1.2.1.33.1.6.1.0',
        'batteryteststatus'=>'.1.3.6.1.2.1.33.1.7.3.0',
        'batterysecondsremaining'=>'.1.3.6.1.2.1.33.1.2.3.0',


### XUPS-MIB
#       'inputfreq'=>'.1.3.6.1.4.1.534.1.3.1',
# 	'alarmstatus'=>'.1.3.6.1.4.1.534.1.7.1',
	'batteryABMstatus'=>'1.3.6.1.4.1.534.1.2.5.0'


);

my %ERRORS=('OK'=>0,'WARNING'=>1,'CRITICAL'=>2,'UNKNOWN'=>3);
my $o_host=     undef;          # hostname
my $o_community= undef;         # community
my $o_port=     161;            # SNMP port
my $o_help=     undef;          # help option
my $o_verb=     undef;          # verbose mode
my $o_version=  undef;          # version info option
my $o_unit= 	undef;         	# Time Unit to use for reporting
my $o_warn=     undef;          # warning level option
my $o_crit=     undef;          # Critical level option
my $o_reportonly=	undef;      # Report Only option
my $o_timeout=  5;              # Default 5s Timeout
my $o_version2= undef;          # use snmp v2c
# SNMPv3 specific
my $o_login=    undef;          # Login for snmpv3
my $o_passwd=   undef;          # Pass for snmpv3
my $o_attr=     undef;          # What attribute(s) to check (specify more then one separated by '.')
my @o_attrL=    ();             # array for above list
my $o_unkdef=   2;              # Default value to report for unknown attributes
my $o_type=     undef;          # Type of system to check 
my $j=		0;
my $k= 		0;
my $phases= 	undef;		# Number of phases

sub print_version { print "\n$0\nVersion $Version\n" };

sub print_usage {
	print "Usage: $0 [-v] -H <host> -C <snmp_community> [-2] | (-l login -x password) [-P <port>] [-T <check_type> alarm_status|battery_monitoring_status|battery_remaining|battery_test_status|firmware_version|firmware_version_nic|global_status|model|output_voltage|output_load|output_current|output_freq|input_voltage|input_freq|input_current|serial_number] [-U <unit> h|m|s] [-w <warning value>] [-c <critical value>] -r [-t <timeout>] [-V] -O <Number of Phases> \n";
	print "\nFor detailed help type \"check_ups_powerware.pl --help | more\"\n";
}

# Return true if arg is a number
sub isnum {
	my $num = shift;
	if ( $num =~ /^(\d+\.?\d*)|(^\.\d+)$/ ) { return 1 ;}
	return 0;
}

sub help {
	print "\nPowerwareUPS Plugin for Nagios version ",$Version,"\n";
	print " by Cord Scott, Tom Greaser and Bryan Tolka - btolka\@hsc.wvu.edu\n\n";
	print_usage();
	print "\n";
	print <<EOD;
-v, --verbose
    print extra debugging information
-h, --help
    print this help message
-H, --hostname=HOST
    name or IP address of UPS to check
-C, --community=COMMUNITY NAME
    community name for the host's SNMP agent (implies v 1 protocol)
-2, --v2c
    use SNMP v2 (instead of SNMP v1)
-P, --port=PORT
    SNMP port (Default 161)
-t, --timeout=INTEGER
    timeout for SNMP in seconds (Default: 5)
-V, --version
    prints version number
-U, --unit=TIME UNIT
	What unit of measurement to use when reporting time values
	Required for battery_remaining
	s = seconds
	m = minutes
	h = hours
-w, --warn=WARNING VALUE
	Warning value for tiggering alerts, works with:
		battery_remaining, output_load, output_current, input_current
-c, --crit=CRITICAL VALUE
	Critical value for tiggering alerts, works with:
		battery_remaining, output_load, output_current, input_current
-r, --reportonly
	If warning or critical values have been defined, only use them for
	performance data, status code will be returned as OK
-T, --type
		This is the type of check you want to perform against the UPS
	alarm_status 
		Current traps that are active (only odd traps are listed)
		 Will always return an OK status
	battery_monitoring_status 
		Advanced battery monitoring status
		 Will report a CRITICAL status if traps 11, 13, 15 or 57 are 
		 present (alerting that the UPS is either over/under AC Voltage
		 OR Input Frequency is out of range OR UPS is on battery)
		 (if UPS is On Battery it will NOT report a CRITICAL status if 
		 a Battery Test is in progress)
	battery_remaining
		How much time the battery has remaining before empty
	battery_test_status
		Report the current battery test status
	firmware_version
		Report the full firmware version of the Dell UPS
	firmware_version_nic
		Report the full firmware version of the Network Management Card
	global_status
		Current status of the UPS
		UPS Result	Nagios Status Code
		Other		OK
		Unknown		UNKNOWN
		OK		OK
		Aborted		OK
		Non-Critical	WARNING
		Critical	CRITICAL
		Non-Recoverable	CRITICAL
	model
		Report the model of the Dell UPS
	output_voltage
		VA consumption
	output_load
		Output Load
	output_current
		Output Current
	output_freq
		Output Freq
	input_voltage
                Input Voltage
        input_freq
                Input Freq
	input_current
		Input Current	
	serial_number
		Report the serial number of the Dell UPS

-O  	Number of phases to monitor (this was added to deal with snmp issues with 9170+
		
Examples:
	check_ups_powerware.pl -H 192.168.5.1 -C public -T serial_number
	check_ups_powerware.pl -H 192.168.5.1 -C public -T battery_remaining -U m -w 15 -c 8
	check_ups_powerware.pl -H 192.168.5.1 -C public -T 2output_load -w 1500 -c 2000
	
EOD
}
#===============================================================================
# For verbose output - don't use it right now
sub verb { my $t=shift; print $t,"\n" if defined($o_verb) ; }
# Get the alarm signal (just in case snmp timout screws up)
$SIG{'ALRM'} = sub {
	print ("ERROR: Alarm signal (Nagios time-out)\n");
	exit $ERRORS{"UNKNOWN"};
};
sub check_options {
    Getopt::Long::Configure ("bundling");
    GetOptions(
		'v'		=> \$o_verb,			'verbose'		=> \$o_verb,
		'h'		=> \$o_help,			'help'			=> \$o_help,
		'H:s'	=> \$o_host,			'hostname:s'	=> \$o_host,
		'P:i'	=> \$o_port,			'port:i'		=> \$o_port,
		'C:s'	=> \$o_community,		'community:s'	=> \$o_community,
		'l:s'	=> \$o_login,			'login:s'		=> \$o_login,
		'x:s'	=> \$o_passwd,			'passwd:s'		=> \$o_passwd,
		't:i'	=> \$o_timeout,			'timeout:i'		=> \$o_timeout,
		'V'		=> \$o_version,			'version'		=> \$o_version,
		'2'		=> \$o_version2,		'v2c'			=> \$o_version2,
		'U:s'	=> \$o_unit,			'unit:s'		=> \$o_unit,
		'w:i'	=> \$o_warn,			'warn:i'		=> \$o_warn,
		'c:i'	=> \$o_crit,			'crit:i'		=> \$o_crit,
		'T:s'	=> \$o_type,			'type:s'		=> \$o_type,
		'r'		=> \$o_reportonly,		'reportonly'	=> \$o_reportonly,
		'O:i'	=> \$phases,                    'phases:i'                => \$phases
    );
    # Print help text
	if (defined($o_help) ) { help(); exit $ERRORS{"UNKNOWN"}; }
    
	# Print version
	if (defined($o_version)) { print_version(); exit $ERRORS{"UNKNOWN"}; }
    
	# Check if the host has been defined
	if (!defined($o_host) ) 
        { print "\nYou have not defined the UPS to check (IP Address or FQDN)\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; }
    
	# check snmp information
    if (!defined($o_community) && (!defined($o_login) || !defined($o_passwd)) )
        { print "\nYou need to define SNMP connection settings\n\n"; print_usage(); exit $ERRORS{"UNKNOWN"}; }
    
	if (!defined($o_type)) { print "\nCheck type has not been defined\n\n"; print_usage(); exit $ERRORS{"UNKNOWN"}; }
	
	# Critical Value
	if (defined($o_crit)) {
		#print "$o_crit \n";
		# Check to see if the value is blank
			if ($o_crit eq '') {
				print "\nThe \"crit\" value cannot be left blank\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
			} 
	}
	
	# Warning Value
	if (defined($o_warn)) {
		#print "$o_warn \n";
		# Check to see if the unit is blank
			if ($o_warn eq '') {
				print "\nThe \"warn\" value cannot be left blank\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
			}  
	}
	
	if (defined ($o_type)) {
		if (!defined($check_types{$o_type}))  {
			print "\nUnknown check type \"$o_type\"\n\n"; print_usage(); exit $ERRORS{"UNKNOWN"}; 
        }
		# Check to see if the Unit value has been correctly defined
		$unit_type_defined = 0;
		foreach (@unit_dependant_checks) {
			#print "$_ \n";
			# See if we have a check that uses the unit value
			if ($_ eq $o_type) {
				#print "$_ \n";
				# Check to see if the unit has been defined
				if (!defined($o_unit)) {
					print "\nThe \"$_\" check requires the unit type to be defined\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
				} 
				# Check to see if the unit is blank
				elsif ($o_unit eq '') {
					print "\nThe \"$_\" check unit type cannot be left blank\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
				} 
				# Check to see if the unit value is an allowed type
				foreach $unit_type_check (@unit_types) {
					if ($o_unit eq $unit_type_check) {
						$unit_type_defined = 1;
					}
				}
				# Report error if unit value is not allowed
				if ($unit_type_defined != 1) {
					print "\nThe unit \"$o_unit\" is not an allowed type\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
				}
			} 
		}
	}
}
########## MAIN #######
eval "use Net::SNMP";
if ($@) {
  print "\nERROR: You do NOT have the Net:".":SNMP library \n";
  print "  Install it by running: \n";
  print "  perl -MCPAN -e shell \n";
  print "  cpan[1]> install Net::SNMP \n";
  exit 1;
} else {
  verb("The Net:".":SNMP library is available on your server \n");
}


check_options();
# Check global timeout if something goes wrong
if (defined($o_timeout)) {
  verb("Alarm at $o_timeout");
  alarm($o_timeout);
} else {
  verb("no timeout defined : using 5s");
  alarm (5);
}


# SNMP Connection to the host
my ($session,$error);
if (defined($o_login) && defined($o_passwd)) {
  # SNMPv3 login
  verb("SNMPv3 login");
  ($session, $error) = Net::SNMP->session(
      -hostname         => $o_host,
      -version          => '3',
      -username         => $o_login,
      -authpassword     => $o_passwd,
      -authprotocol     => 'md5',
      -privpassword     => $o_passwd,
      -timeout          => $o_timeout
   );
} else {
   if (defined ($o_version2)) {
     # SNMPv2 Login
         ($session, $error) = Net::SNMP->session(
        -hostname  => $o_host,
            -version   => 2,
        -community => $o_community,
        -port      => $o_port,
        -timeout   => $o_timeout
     );
   } else {
    # SNMPV1 login
    ($session, $error) = Net::SNMP->session(
       -hostname  => $o_host,
       -community => $o_community,
       -port      => $o_port,
       -timeout   => $o_timeout
    );
  }
}

# next part of the code builds list of attributes to be retrieved
my $i;
my $oid;
my $line;
my $resp;
my $attr;
my $key;
my %varlist = ();
my $result;
my $statusinfo = "";
my $statuscode = "";
my $unit_string = "";
my $unit_value = "";
my $unit_decimalpoints = "";
my $exit_no_status = 0;



verb("SNMP responses...");
for $attr ( @{ $check_types{$o_type} } ) {
	$result = $session->get_request( 
    -varbindlist => [$ups_oids{$attr}]
	);
 	
	# Checking if verbose mode is enabled
	if (defined $o_verb) {
		if (!defined($result->{$ups_oids{$attr}})) {
			verb("\nRESULT: $attr \n$ups_oids{$attr} = \n");
		}
		else { 
			verb("\nRESULT: $attr \n$ups_oids{$attr} = $result->{$ups_oids{$attr}}\n");
		}
	}
	
	# Check for an empty result
	if (!defined($result->{$ups_oids{$attr}}) & $o_type ne "output_load" & $o_type ne "input_voltage" & $o_type ne "output_voltage" & $o_type ne "input_current" & $o_type ne "output_current") {
		print "\nTesting for no result\n"; 
		$statusinfo = "No response from UPS";
		$statuscode = "UNKNOWN";
		$perf_data = "";
	} 
	

# Alarm Status
	elsif ($o_type eq "alarm_status") {
		$statuscode = "OK";
		if ($result->{$ups_oids{$attr}} eq '') {
			$statusinfo = "No Active Traps";
		}
		else { 
			$statusinfo = "Active Traps:" if (!$statusinfo);
			#print "\n$result->{$ups_oids{$attr}}\n\n"; 
			my @result_split = split(',',$result->{$ups_oids{$attr}});
			foreach (@result_split) { 
				#print "\n$_\n\n"; 
				if ($_ == 3) {
					$statusinfo .= " {3: On Battery}";
				}
				elsif ($_ == 4) {
                                        $statusinfo .= " {4: Low Battery}";
                                }
				elsif ($_ == 5) {
					$statusinfo .= " {5: Utility Power Restored}";
				}
				elsif ($_ == 6) {
                                        $statusinfo .= " {6: Return From Low Battery}";
                                }
				elsif ($_ == 7) {
					$statusinfo .= " {7: Output Overload}";
				}
				elsif ($_ == 8) {
                                        $statusinfo .= " {8: Internal Failure}";
                                }
				elsif ($_ == 9) {
					$statusinfo .= " {9: Battery Discharged}";
				}
				elsif ($_ == 10) {
                                        $statusinfo .= " {10: Inverter Failure}";
                                }
				elsif ($_ == 11) {
					$statusinfo .= " {11: OnBypass}";
				}
				elsif ($_ == 12) {
                                        $statusinfo .= " {12: Bypass Not Available}";
                                }
				elsif ($_ == 13) {
					$statusinfo .= " {13: Output Off}";
				}
				elsif ($_ == 14) {
                                        $statusinfo .= " {14: Input Failure}";
                                }
				elsif ($_ == 15) {
					$statusinfo .= " {15: Building Alarm}";
				}
				elsif ($_ == 16) {
                                        $statusinfo .= " {16: Shutdown Imminent}";
                                }
				elsif ($_ == 17) {
					$statusinfo .= " {17: On Inverter}";
				}
				elsif ($_ == 20) {
					$statusinfo .= " {20: Breaker Open}";
				}
				elsif ($_ == 21) {
					$statusinfo .= " {21: Alarm Entry Added}";
				}
				elsif ($_ == 22) {
                                        $statusinfo .= " {22: Alarm Entry Removed}";
                                }
				elsif ($_ == 23) {
					$statusinfo .= " {23: Alarm Battery Bad}";
				}
				elsif ($_ == 24) {
                                        $statusinfo .= " {24: Output Off As Requested}";
                                }
				elsif ($_ == 25) {
					$statusinfo .= " {25: Diagnostic Test Failed}";
				}
				elsif ($_ == 26) {
                                        $statusinfo .= " {26: Communications Lost}";
                                }
				elsif ($_ == 27) {
					$statusinfo .= " {27: Ups Shutdown Pending}";
				}
				elsif ($_ == 28) {
                                        $statusinfo .= " {28: Alarm Test In Progress}";
                                }
				elsif ($_ == 29) {
					$statusinfo .= " {29: Ambient Temp Bad}";
				}
				elsif ($_ == 30) {
                                        $statusinfo .= " {30: Loss Of Redundancy}";
                                }
				elsif ($_ == 31) {
					$statusinfo .= " {31: Alarm Temp Bad}";
				}
				elsif ($_ == 32) {
                                        $statusinfo .= " {32: Alarm Charger Failed}";
                                }
				elsif ($_ == 33) {
					$statusinfo .= " {33: Alarm Fan Failure}";
				}
				elsif ($_ == 34) {
                                        $statusinfo .= " {34: Alarm Fuse Failure}";
                                }
				elsif ($_ == 35) {
					$statusinfo .= " {35: Power Switch Bad}";
				}
				elsif ($_ == 36) {
                                        $statusinfo .= " {36: Module Failure}";
                                }
				elsif ($_ == 37) {
					$statusinfo .= " {37: On Alternate Power Source}";
				}
				elsif ($_ == 38) {
                                        $statusinfo .= " {38: Alt Power Not Available}";
                                }
				elsif ($_ == 39) {
					$statusinfo .= " {39: Notice Condition}";
				}
				elsif ($_ == 40) {
                                        $statusinfo .= " {40: Remote Temp Bad}";
                                }
				elsif ($_ == 41) {
					$statusinfo .= " {41: Remote Humidity Bad}";
				}
				elsif ($_ == 42) {
					$statusinfo .= " {42: Alarm Output Bad}";
				}
				elsif ($_ == 43) {
				 	$statusinfo .= " {43: Alarm Awaiting Power}";
				}
				elsif ($_ == 44) {
					$statusinfo .= " {44: On Maintenance Bypass}";
				}
			}  # ends foreach
		} #  ends else
		$perf_data = "";
	}
	
	# Battery Monitoring Status
	elsif ($o_type eq "battery_monitoring_status") {
		$statusinfo = "Battery Monitoring Status: " if (!$statusinfo);
		#print "\n$result->{$ups_oids{$attr}}\n\n"; 
		my @result_split = split(',',$result->{$ups_oids{$attr}});
		foreach (@result_split) { 
			#print "\n$_\n\n"; 
			if ($_ == 1) {
				$statusinfo .= "Charging";
				$statuscode = "OK";
			}
			elsif ($_ == 2) {
				$statusinfo .= "Discharging";
				$statuscode = "OK";
			}
			elsif ($_ == 3) {
				$statusinfo .= "Floating";
				$statuscode = "OK";
			}
			elsif ($_ == 4) {
				$statusinfo .= "Resting";
				$statuscode = "OK";
			}
			elsif ($_ == 5) {
				$statusinfo .= "Off";
				$statuscode = "CRITICAL";
			}
			elsif ($_ == 11) {
				$statusinfo .= " {Input AC Over Voltage Threshold}";
				$statuscode = "CRITICAL";
			}
			elsif ($_ == 13) {
				$statusinfo .= " {Input AC Under Voltage Threshold}";
				$statuscode = "CRITICAL";
			}
			elsif ($_ == 15) {
				$statusinfo .= " {Input Frequency Out Of Range}";
				$statuscode = "CRITICAL";
			}
			elsif ($_ == 57) {
				my @result_split_again = split(',',$result->{$ups_oids{$attr}});
				my $test_in_progress = 0;
				foreach (@result_split_again) {
					if ($_ == 61) {
						$test_in_progress = 1;
					}
				}
				if ($test_in_progress == 0) {
					$statusinfo .= " {UPS on battery}";
					$statuscode = "CRITICAL";
				}
			}
		}
		$perf_data = "";
	}
	
	# Battery Remaining	
	elsif ($o_type eq "battery_remaining") {
		$statusinfo = "Battery Remaining: " if (!$statusinfo);
		my $battery_remaining = $result->{$ups_oids{$attr}};
		if ($o_unit eq "s") { 
			$unit_string = "Seconds";
			$unit_value = 1;
			$unit_decimalpoints = "%.0f";
			}
		if ($o_unit eq "m") { 
			$unit_string = "Minutes";
			$unit_value = 60;
			$unit_decimalpoints = "%.2f";
			}
		if ($o_unit eq "h") { 
			$unit_string = "Hours";
			$unit_value = 3600;
			$unit_decimalpoints = "%.2f";
			}
		$battery_remaining = $battery_remaining*60;
		$battery_remaining = $battery_remaining/$unit_value;
		$battery_remaining = sprintf("$unit_decimalpoints", $battery_remaining);
		$statusinfo .= "$battery_remaining";	
		$statusinfo .= " $unit_string";
		$statuscode = "OK";
		# Check for warning value
		no warnings;
		if (defined($o_warn)) {
			# Check for critical value
			if (defined($o_crit)) {
				# Ensure warning is a larger value than critical
				if ($o_warn < $o_crit) {
					print "\nThe \"warn\" value must be larger than the \"crit\" value\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
				}
				# Check if critical value has been exceeded
				elsif ($battery_remaining <= $o_crit) {
					$statuscode = "CRITICAL";
				}
				# Otherwise check if warning value has been exceeded
				elsif ($battery_remaining <= $o_warn) {
					$statuscode = "WARNING";
				}
			}
			# If no critical value was defined check if warning value has been exceeded
			elsif ($battery_remaining <= $o_warn) {
					$statuscode = "WARNING";
			}
		}
		# Check for critical value
		if (defined($o_crit) && !defined($o_warn)) {
			# Check if critical value has been exceeded
			if ($battery_remaining <= $o_crit) {
				$statuscode = "CRITICAL";
			}
		}
	$perf_data = "|BatteryRemaining=$battery_remaining"."$unit_string;$o_warn;$o_crit;;";
	use warnings;
	}

	# Battery Test Status	
	elsif ($o_type eq "battery_test_status") {
		$statusinfo = "Battery Test Status:" if (!$statusinfo);
		if ($result->{$ups_oids{$attr}} == 1) {
			$statusinfo .= "test";
			$statuscode = "OK";
		}
		elsif ($result->{$ups_oids{$attr}} == 2) {
			$statusinfo .= "Warning";
			$statuscode = "WARNING";
		}
		elsif ($result->{$ups_oids{$attr}} == 3) {
			$statusinfo .= "Error";
			$statuscode = "CRITICAL";
		}
		elsif ($result->{$ups_oids{$attr}} == 4) {
			$statusinfo .= "Aborted";
			$statuscode = "OK";
		}
		elsif ($result->{$ups_oids{$attr}} == 5) {
			$statusinfo .= "In Progress";
			$statuscode = "OK";
		}
		elsif ($result->{$ups_oids{$attr}} == 6) {
			$statusinfo .= "No Test Initiated";
			$statuscode = "OK";
		}
		elsif ($result->{$ups_oids{$attr}} == 7) {
			$statusinfo .= "Test Scheduled";
			$statuscode = "OK";
		}
		$perf_data = "";
	}
	
	# Firmware Version
	elsif ($o_type eq "firmware_version") {
		$statusinfo = "Firmware Version: " if (!$statusinfo);
		$statusinfo .= "." if ($statusinfo ne "Firmware Version: ");
		$statusinfo .= "$result->{$ups_oids{$attr}}";
		$statuscode = "OK";
		$perf_data = "";
		$exit_no_status = 1;
	}

	# Firmware Version NIC
	elsif ($o_type eq "firmware_version_nic") {
		$statusinfo = "NIC Firmware Version: " if (!$statusinfo);
		$statusinfo .= "$result->{$ups_oids{$attr}}";
		$statuscode = "OK";
		$perf_data = "";
		$exit_no_status = 1;
	}

	# Global Status
	elsif ($o_type eq "global_status") {
		$statusinfo = "Global Status: " if (!$statusinfo);
		if ($result->{$ups_oids{$attr}} == 1) {
			$statusinfo .= "Other";
			$statuscode = "OK";
		}
		elsif ($result->{$ups_oids{$attr}} == 2) {
			$statusinfo .= "Unknown";
			$statuscode = "UNKNOWN";
		}
		elsif ($result->{$ups_oids{$attr}} == 3) {
			$statusinfo .= "OK";
			$statuscode = "OK";
		}
		elsif ($result->{$ups_oids{$attr}} == 4) {
			$statusinfo .= "Non-Critical";
			$statuscode = "WARNING";
		}
		elsif ($result->{$ups_oids{$attr}} == 5) {
			$statusinfo .= "Critical";
			$statuscode = "CRITICAL";
		}
		elsif ($result->{$ups_oids{$attr}} == 6) {
			$statusinfo .= "Non-Recoverable";
			$statuscode = "CRITICAL";
		}
		$perf_data = "";
	}

	# Model
	elsif ($o_type eq "model") {
		$statusinfo = "Model: $result->{$ups_oids{$attr}}";
		$statuscode = "OK";
		$perf_data = "";
		$exit_no_status = 1;
	}
	
	
	
	# Output Load
	elsif ($o_type eq "output_load") {
		#print "\n$result->{$ups_oids{$attr}}\n\n"; 
                my @result_split = split(',',$result->{$ups_oids{$attr}});
		foreach (@result_split) {
			$k = $j++; 
                        if (defined($phases)){last if ($j > $phases)};
			#print "\n$_\n\n"; 
			$statusinfo .= " Output Load $j: ";
	                $statusinfo .= "$_";
                	$statusinfo .= "";
                	$statuscode = "OK";
                		# Check for warning value
                		if (defined($o_warn)) {
                 			# Check for critical value
                        		if (defined($o_crit)) {
                                		# Ensure warning is a smaller value than critical
                                		if ($o_warn > $o_crit) {
                                        	print "\nThe \"warn\" value must be smaller than the \"crit\" value\n\n";print_usage(); exit $ERRORS{"UNKNOWN"};
                                		}
                                	# Check if critical value has been exceeded
                                	elsif ($_ >= $o_crit) {
                                        $statuscode = "CRITICAL";
                                	}
                                	# Otherwise check if warning value has been exceeded
                                	elsif ($_ >= $o_warn) {
                                        $statuscode = "WARNING";
                               		}
                        	}
                        	# If no critical value was defined check if warning value has been exceeded
                        	elsif ($_ >= $o_warn) {
                                        $statuscode = "WARNING";
                        	}
                		}
                		# Check for critical value
                		if (defined($o_crit) && !defined($o_warn)) {
                        		# Check if critical value has been exceeded
                        		if ($_ >= $o_crit) {
                                	$statuscode = "CRITICAL";
                        		}
                		}

			
				$perf_data1 .= "'Output Load $j'=$_".";$o_warn;$o_crit;;";
		}   # ends foreach
                no warnings;
                use warnings;

        	### checks for null values
              #  if (defined($perf_data1)){$perf_data = "| $perf_data1";}
              #  else
              #  {
              #  $statusinfo = "No response from UPS";
              #  $statuscode = "UNKNOWN";
              #  $perf_data = "";
              #  }
		$perf_data = "| $perf_data1";
	}




		
	# Output Current
	elsif ($o_type eq "output_current") {

		#print "\n$result->{$ups_oids{$attr}}\n\n";
                my @result_split = split(',',$result->{$ups_oids{$attr}});
                foreach (@result_split) {
                        $k = $j++;
			if (defined($phases)){last if ($j > $phases)};
			#print "\n$_\n\n";
                	$_ = $_/10;
		        $statusinfo .= " Output Current $j: ";
                        $statusinfo .= "$_";
                        $statusinfo .= " AC Amps";
                        $statuscode = "OK";

		# Check for warning value
		if (defined($o_warn)) {
			# Check for critical value
			if (defined($o_crit)) {
				# Ensure warning is a larger value than critical
				if ($o_warn < $o_crit) {
					print "\nThe \"warn\" value must be larger than the \"crit\" value\n\n";print_usage(); exit $ERRORS{"UNKNOWN"}; 
				}
				# Check if critical value has been exceeded
				elsif ($_ <= $o_crit) {
					$statuscode = "CRITICAL";
				}
				# Otherwise check if warning value has been exceeded
				elsif ($_ <= $o_warn) {
					$statuscode = "WARNING";
				}
			}
			# If no critical value was defined check if warning value has been exceeded
			elsif ($_ <= $o_warn) {
					$statuscode = "WARNING";
			}
		}
		# Check for critical value
		if (defined($o_crit) && !defined($o_warn)) {
			# Check if critical value has been exceeded
			if ($_ <= $o_crit) {
				$statuscode = "CRITICAL";
			}
		}
			
			$perf_data1 .= "'Output Current $j'=$_"."ACAmps;$o_warn;$o_crit;;";
		}  #end foreach 
		no warnings;
		use warnings;
		### checks for null values
                if (defined($perf_data1)){$perf_data = "| $perf_data1";}
                else
                {
                $statusinfo = "No response from UPS";
                $statuscode = "UNKNOWN";
                $perf_data = "";
                }

	
	}   ## ends elsif


        # Input Current
        elsif ($o_type eq "input_current") {

                #print "\n$result->{$ups_oids{$attr}}\n\n";
                my @result_split = split(',',$result->{$ups_oids{$attr}});
                foreach (@result_split) {
                        $k = $j++;
			if (defined($phases)){last if ($j > $phases)};
                        #print "\n$_\n\n";
                        $_ = $_/10;
                        $statusinfo .= " Input Current $j: ";
                        $statusinfo .= "$_";
                        $statusinfo .= " AC Amps";
                        $statuscode = "OK";

                # Check for warning value
                if (defined($o_warn)) {
                        # Check for critical value
                        if (defined($o_crit)) {
                                # Ensure warning is a larger value than critical
                                if ($o_warn < $o_crit) {
                                        print "\nThe \"warn\" value must be larger than the \"crit\" value\n\n";print_usage(); exit $ERRORS{"UNKNOWN"};
                                }
                                # Check if critical value has been exceeded
                                elsif ($_ <= $o_crit) {
                                        $statuscode = "CRITICAL";
                                }
                                # Otherwise check if warning value has been exceeded
                                elsif ($_ <= $o_warn) {
                                        $statuscode = "WARNING";
                                }
                        }
                        # If no critical value was defined check if warning value has been exceeded
                        elsif ($_ <= $o_warn) {
                                        $statuscode = "WARNING";
                        }
                }
                # Check for critical value
                if (defined($o_crit) && !defined($o_warn)) {
                        # Check if critical value has been exceeded
                        if ($_ <= $o_crit) {
                                $statuscode = "CRITICAL";
                        }
                }

                        $perf_data1 .= "'Input Current $j'=$_"."ACAmps;$o_warn;$o_crit;;";
                }  #end foreach
                no warnings;
                use warnings;
		### checks for null values
                if (defined($perf_data1)){$perf_data = "| $perf_data1";}
		else 
		{
		$statusinfo = "No response from UPS";
                $statuscode = "UNKNOWN";
                $perf_data = "";
		}
        }   ## ends elsif



   # Output Freq
        elsif ($o_type eq "output_freq") {
                $statusinfo = "Output Freq: " if (!$statusinfo);
                my $output_freq = $result->{$ups_oids{$attr}};
                $output_freq = $output_freq/10;
		$statusinfo .= "$output_freq";
                $statusinfo .= "Hertz";
                $statuscode = "OK";
		$o_warn = 45;
		$o_crit = 65;
                
		# Check for frequency to be between 45 and 65
                        # Check if critical value has been exceeded
                        if ($output_freq < 45 or $output_freq> 65) {
                                $statuscode = "CRITICAL";
                        }
                no warnings;
                $perf_data = "|Output Freq=$output_freq"."Hertz;$o_warn;$o_crit;;";
                use warnings;
        }

   # Input Freq
        elsif ($o_type eq "input_freq") {
                $statusinfo = "Input Freq: " if (!$statusinfo);
                my $input_freq = $result->{$ups_oids{$attr}};
                $input_freq = $input_freq/10;
		$statusinfo .= "$input_freq";
                $statusinfo .= "Hertz";
                $statuscode = "OK";
                $o_warn = 45;
                $o_crit = 65;
		
		# Check for frequency to be between 45 and 65
                        # Check if critical value has been exceeded
                        if ($input_freq < 45 or $input_freq> 65) {
                                $statuscode = "CRITICAL";
                        }

		no warnings;
                $perf_data = "|Input Freq=$input_freq"."Hertz;$o_warn;$o_crit;;";
                use warnings;
        }

   # Input Voltage
        elsif ($o_type eq "input_voltage") {

                #print "\n$result->{$ups_oids{$attr}}\n\n";
		$o_warn = 108;
                $o_crit = 132;
                my @result_split = split(',',$result->{$ups_oids{$attr}});
                foreach (@result_split) {
                        $k = $j++;
                       	if (defined($phases)){last if ($j > $phases)};
			#print "\n$_\n\n";
                        	$statusinfo .= " Input Voltage $j: ";
                        	$statusinfo .= "$_";
                        	$statusinfo .= " VAC";
                        	$statuscode = "OK";
                  	
				# Check for frequency to be between 108 and 132
                        	# Check if critical value has been exceeded
                        	if ($_ < 108 or $_> 132) {
                           		$statuscode = "CRITICAL";
                        	}

			$perf_data1 .= "'Input Voltage $j'=$_"."VAC;$o_warn;$o_crit;;";

		}   # ends foreach
                no warnings;
                use warnings;
        	### checks for null values
		if (defined($perf_data1)){$perf_data = "| $perf_data1";}
                else
                {
                $statusinfo = "No response from UPS";
                $statuscode = "UNKNOWN";
                $perf_data = "";
                }

	}


        # Output Voltage
        elsif ($o_type eq "output_voltage") {

		#print "\n$result->{$ups_oids{$attr}}\n\n";
		$o_warn = 108;
                $o_crit = 132;
                my @result_split = split(',',$result->{$ups_oids{$attr}});
                foreach (@result_split) {
                        $k = $j++;
                        if (defined($phases)){last if ($j > $phases)};
			#print "\n$_\n\n";
				$statusinfo .= " Output Voltage $j: ";
        	                $statusinfo .= "$_";
                	        $statusinfo .= " VAC";
                        	$statuscode = "OK";
        			# Check for frequency to be between 118 and 121
                        	# Check if critical value has been exceeded
                        	if ($_ < 108 or $_> 132) {
                           	$statuscode = "CRITICAL";
                        	}

                        $perf_data1 .= "'Output Voltage $j'=$_"."VAC;$o_warn;$o_crit;;";

                }   # ends foreach
                no warnings;
                use warnings;
        	### checks for null values
		if (defined($perf_data1)){$perf_data = "| $perf_data1";}
                else
                {
                $statusinfo = "No response from UPS";
                $statuscode = "UNKNOWN";
                $perf_data = "";
                }
 
	}





	
	# Serial Number
	elsif ($o_type eq "serial_number") {
		$statusinfo = "Serial Number: $result->{$ups_oids{$attr}}";
		$statuscode = "OK";
		$perf_data = "";
		$exit_no_status = 1;
	}

}


$session->close();

my $final_output = "";

# Check to see if the Report Only option has been chosen   
if (defined($o_reportonly)) {
	$statuscode = "OK";
}

if ($exit_no_status == 1) {
	$final_output = "$statusinfo" . "$perf_data\n";
}
else
{$final_output = "$statuscode: " . "$statusinfo" . "$perf_data\n";
}

printf($final_output);

verb("\nEXIT CODE: $ERRORS{$statuscode} STATUS CODE: $statuscode\n");

exit $ERRORS{$statuscode};

