#!/usr/bin/perl

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

use warnings;
use strict;

# Checking if Nagios::Plugin is installed or not
eval 'require Nagios::Plugin';
if ($@) {
	print
"You dont have Nagios::Plugin installed in your system. Try to install it using:\n";
	print "perl -MCPAN -e shell\n";
	print "install Nagios::Plugin\n";
	print "Otherwise, go to http://www.cpan.org and follow these steps:\n";
	print "Download nagios-Plugin-0.36.tar.gz and extract it to /tmp\n";
	print "perl Makefile.PL\n";
	print "make\n";
	print "make test\n";
	print "make install\n";
}

else {
use Nagios::Plugin::Functions;

# We dont want to have dependecies of external modules as round function from Math module
#use Math::Round qw/round/;

	my $np = Nagios::Plugin->new(
		'usage' => 'Usage: %s -C <config_file> -S <server> -p <volume path>
[-w|--warning=<threshold>] [-c|--critical=<threshold>] [-u|--unit <unit type>]'
	);
	$np->add_arg(
		spec => 'config_file|C=s',
		help =>
"-C, --config=FILE. It defines the config file whith credentials for each server: IP/Hostname User Password",
		required => 1
	);

	$np->add_arg(
		spec => 'server|S=s',
		help =>
"-S, --server=IP/HOST. It defines the IP or hostname we want to check",
		required => 1
	);

	$np->add_arg(
		spec => 'volume_path|p=s',
		help =>
"-p, --path=VOLUME PATH. It defines the volume we want to monitorize its space",
		required => 1
	);

	$np->add_arg(
		spec     => 'warning|w=s',
		help     => "-w, --warning=PERCENT. Percent used when to warn\n",
		required => 0
	);
	$np->add_arg(
		spec     => 'critical|c=s',
		help     => "-c, --critical=PERCENT. Percent used when critical\ne",
		required => 0
	);

	$np->add_arg(
		spec => 'unit|u=s',
		help =>
"-u, --unit [Kb|Mb|Gb]. Sets output in specific format: Kb, Mb or Gb. Default Mb",
		required => 0
	);

	$np->getopts;

	# Initializating variables
	#--------------------------
	my $vi_server = $np->opts->server;

	#Setting unit value. If none is chosen, Mb by default
	my $unit = ( $np->opts->unit ) ? $np->opts->unit : "Mb";

	my ( $vi_username, $vi_password, $volumepath );

	my ( $totalsizeB, $freesizeB, $usedsizeB, $totalsize, $freesize, $usedsize )= 0;

	my ( $percentused, $percentfree ) = 0;

# Getting Server Ip, username and password from config_file.txt to connect to Esxi Server
	sub get_data_server() {

		my $config_file = $np->opts->config_file;

		if ( -e $config_file ) {

			open FILE, $config_file or nagios_die($!);

			while (<FILE>) {
				if ( $_ =~ /$vi_server/ ) {
					( $vi_server, $vi_username, $vi_password ) = split(/ /);
				}
				else {
					nagios_die(
						"There is no such server (IP/Host) in $config_file\n");
				}
			}

		}
		else {
			nagios_die("Configuration file $config_file doesnt exist\n");
		}

	# If warning and critical are defined,checking warning % is under critical %
		if ( $np->opts->warning && $np->opts->critical ) {
			if ( $np->opts->warning > $np->opts->critical ) {
				nagios_die(
"Please adjust your warning/critical thresholds. The warning must be lower than the critical level!\n"
				);
			}
		}
	}

	sub get_volume_size() {
		my $err_log = '/usr/local/nagios/var/check_vmfs.err';
		unlink($err_log);

		$volumepath = $np->opts->volume_path;

`vmkfstools --username '$vi_username' --password '$vi_password' --server '$vi_server' --P '$volumepath' >/dev/null 2>$err_log`;

		if ( $? != 0 ) {
			my $output = `cat $err_log`;

			# We send a CRITICAL status with the error_log file content
			$np->nagios_exit(CRITICAL, $output );
		}
		else {

			# It's mandatory to set variables between ' ' or command is cut
			my $volumesize =`vmkfstools --server '$vi_server' --P '$volumepath' --username '$vi_username' --password '$vi_password'|grep 'Capacity'`;

			# Deleting any trailing string
			chomp($volumesize);
			
			# Output of vmkfstools is like this one:
			# Capacity : 1631819137024, 1047140499456 avail
			# Extracting integer numbers with split & grep into @list array
			my @list=(grep(/(\d)+/,split(/(\d+)/,$volumesize)));
			
			# Volume size in Bytes
			# Total size in bytes, element 0 in the @list array
			$totalsizeB=$list[0];
			# Free size in bytes, element 1 in the @list array
			$freesizeB=$list[1];
			
			
			$usedsizeB = $totalsizeB - $freesizeB;

			
			#To use or not to use round function from Math Library
			# as root:
			# perl -MCPAN -e shell
			# install Math::Round
			#$percentused = round( $usedsizeB * 100 / $totalsizeB );

			# Otherwise you can do it manually, increasing int value in 0.99
			
			# Volume size in %
			$percentused = int( ( $usedsizeB * 100 / $totalsizeB ) + 0.99 );
			$percentfree = 100 - $percentused;
		}

	}

	sub do_unit_conversion() {
		my $multiplier=1;
		
		if ( $unit =~ /kb/i ) {
			$multiplier=1024;	
		}
		elsif ( $unit =~ /mb/i ) {
			$multiplier=1048776;
		}
		elsif ( $unit =~ /gb/i ) {
			$multiplier=1073741824;
		}
		else {
			nagios_die("Unknown unit type: $unit\n");
		}
		
		$totalsize=sprintf '%.2f',$totalsizeB/$multiplier;
		$freesize=sprintf '%.2f',$freesizeB/$multiplier;
		$usedsize=sprintf '%.2f',$usedsizeB/$multiplier;
		
	}

	sub do_output() {
		my $code = $np->check_threshold(
			check    => $percentused,
			warning  => $np->opts->warning,
			critical => $np->opts->critical
		);

		my $output =
"$volumepath - total: $totalsize $unit - used: $usedsize $unit ($percentused%) - free: $freesize $unit ($percentfree%)";

		my $threshold = $np->set_thresholds(
			warning  => $np->opts->warning,
			critical => $np->opts->critical
		);

		$np->add_perfdata(
			label     => $volumepath,
			value     => $percentused,
			threshold => $threshold,
			uom       => "%"
		);

		$np->nagios_exit( $code, $output );

	}

	get_data_server();
	get_volume_size();
	do_unit_conversion();
	do_output();

}

