#!/usr/bin/perl -wT

#
# 04-17-2002 Mike Arndt Version 1.1
#
# bbnt2ns.pl  This script listens for connections on port 1984 (the standard
#             port used by Big Brother) from the Big Brother NT Client.
#             It then process connections and updates the Netsaint external
#             command file as a passive service check.
#

use IO::Socket;
use strict;
use Time::Local;
use Getopt::Std;
use Fcntl ':flock';
use POSIX;
use vars qw($opt_d);

#
# Set location of netsaint command check file.
#
my $cmd_file = "/usr/local/netsaint/var/rw/netsaint.cmd";

#
# Set the port number that Big Brother is using.
#
my $port = "1984";

#
# You should not need to change anything after this point.
#

#
# check command line
# 
help() unless getopts("d:");

#
# global variables
#
my ($server,$client,$debug,$msg_type,@buf);
$debug = 0;
$debug = $opt_d if ($opt_d);

#
# Become a daemon, unless we are in debug mode.
#
if (! $debug) {
	my $pid = fork;
	exit if $pid;
	die "Couldn't fork: $!\n" unless defined($pid);
	POSIX::setsid() or die "Can not start a new session: $!\n";
}

#
# setup socket
#
print "*** setting up socket. ***\n" if ($debug > 1);
$server = IO::Socket::INET->new(LocalPort => $port,
                                Type      => SOCK_STREAM,
                                Reuse     => 1,
                                Listen    => 10 )
	or die "Error setting up socket: $!\n";

#
# wait for connections from the Big Brother NT client, and
# handle them as they come in.
#
print "*** listening on port $port. ***\n" if ($debug > 1);
while (1) {
	print "*** in while loop. ***\n" if ($debug > 1);
	$msg_type = "";
	undef @buf;
	#
	# The script just sits here until a new connection is made.
	#	
	$client = $server->accept();
	while (<$client>) {
		print if ($debug > 1);
		push(@buf,$_);
	}
	for (@buf) {
		$msg_type = $1 if /^status\s.*?\.(\w+)\s+/;
	}
	if ($msg_type =~ /^cpu|procs|disk|msgs$/) {
		check_event(\@buf,$msg_type);
	} else {
		print "Bad msg_type: $msg_type.  Discarding.\n" if ($debug > 1);
	}
}
print "*** closing socket. ***\n" if ($debug > 1);
close($server);
print "*** socket closed. ***\n" if ($debug > 1);
exit 0;

#
# Check a message from Big Brother.  The $type should be cpu, disk, procs,
# or msg.
#
sub check_event {
	my ($buf,$type) = (@_);
	print "\n*** checking $type message. ***\n" if ($debug > 1);
	my ($line,$timestamp,$host,$rv,$info,$color);
	$line = $timestamp = $host = $info = $color = "";
	$rv = -1;
	for (@$buf) {
		($host,$color) = ($1,$2) if /^status\s+(.*?)\.${type}\s+(\w+)/;
	}
	$timestamp = get_timestamp($buf);
	$info = get_info($buf,$type);
	# 
	# If the get_info subroutine fails, netsaint would discard the
	# service check due to the empty description, so make sure we
	# have something.
	#	
	$info = "?" if (! $info);
	if ($color eq "green") {
		$rv = 0;
	} elsif ($color eq "yellow") {
		$rv = 1;
	} elsif ($color eq "red") {
		$rv = 2;
	} else {
		$rv = -1;
	}
	$line = "[" . $timestamp . "] ";
	$line .= join ';',"PROCESS_SERVICE_CHECK_RESULT",$host,$type,$rv,$info;
	print "LINE: $line\n" if ($debug);
	write_cmd_file($line);
	return 1;
}

#
# Extract relevant information about the Big Brother message depending on
# what type of message it is.
#
sub get_info {
	my ($buf,$type) = (@_);
	my $info = "?";
	if ($type eq "cpu") {
		for (@$buf) {
			$info = $1 if /(up:.*?\))/;
		}
		return $info;
	}
	if ($type eq "procs") {
		my (@down_procs,@tmp);
		$info = "Down processes: ";
		for (@$buf) {
			if (/DOWN/) {
				@tmp = split;
				$info .= "$tmp[1]; ";
			}
		}
		$info =~ s/;\s+$//;
		if ($info eq "Down processes: ") {
			$info = "All processes OK";
		}
		return $info;
	}
	if ($type eq "disk") {
		my (@tmp,$high,$fs,$topfs,%capacity);
		for (@$buf) {
			next unless /^\w\s+/;
			(@tmp) = split;
			if (($tmp[0] =~ /^\w$/) && ($tmp[4] =~ /\%/)) {
				$tmp[4] =~ s/\%//;
				$capacity{$tmp[0]} = $tmp[4];
			}
			$info = $1 if /(up:.*?\))/;
		}
		$high = 0;
		for $fs (keys %capacity) {
			if ($capacity{$fs} > $high) {
				$topfs = $fs;
				$high = $capacity{$fs};
			}
		}
		$info = "Largest disk is $topfs at $capacity{$topfs}\% usage";
		return $info;
	}
	if ($type eq "msgs") {
		#
		# We just slurp the whole message into a one-liner, but
		# then limit it to 128 chars.
		#
		$info = "";
		for (@$buf) {
			chomp;
			s/status.*?\]//;
			$info .= " $_";
		}
		$info =~ s/^\s+//;
		if ((length($info)) > 128) {
			$info =~ s/^(.{125}).*/$1/;
			$info .= "...";
		}
		return $info;
	}
	return $info;
}

#
# simple subroutine to write results to the netsaint external command file
#
sub write_cmd_file {
	my ($line) = (@_);
	# Check if the cmd_file is a named pipe, as it needs to be
	# opened slightly differently if it is.
	if (-p "$cmd_file") {
		open FILE, ">$cmd_file" or return 1;
	} else {
		open FILE, ">>$cmd_file" or return 1;
		flock FILE, LOCK_EX or return 1;
	}
	print FILE "$line\n";
	close FILE;
	return 1;
}

#
# determine unix time from date/time included in Big Brother message.
#
sub get_timestamp {
	my ($buf) = (@_);
	my ($ts,$mon,$day,$hour,$min,$sec,$year,%months,@tmp);
	%months = qw(Jan 0 Feb 1 Mar 2 Apr 3 May 4 Jun 5
	             Jul 6 Aug 7 Sep 8 Oct 9 Nov 10 Dec 11);
	for (@$buf) {
		@tmp = split;
		if ($tmp[0] eq "status") {
			$mon  = $months{$tmp[4]};
			$day  = $tmp[5];
			$year = $tmp[8];
			$year = $year - 1900;
			($hour,$min,$sec) = (split /:/, $tmp[6]);
			last;
		}
	}
	$ts = timelocal $sec,$min,$hour,$day,$mon,$year;
	return $ts;
}

#
# help subroutine
#
sub help {
	print <<END;
Usage: bbnt2ns.pl [-d <1,2>]

  By default, bbnt2ns.pl will daemonize.  Use the -d switch to keep
  bbnt2ns.pl in the foreground and print out debugging info.  Use 1
  for a little debugging info, 2 for a lot of debugging info.

END
	exit 0;
}
