#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <time.h>

#include "conf.h"
#include "log.h"


static FILE *log_stream = NULL;

void log_init(char *program) {
    switch (config->log_dest) {
	case _LOG_SYSLOG:
	    if (!config->debug) {
		openlog(program, LOG_PID, LOG_DAEMON);
	    }
	    break;
	case _LOG_STDOUT:
	    log_stream = stdout;
	    break;
	case _LOG_FILE:
	    log_stream = fopen(config->log_file, "a");
	    break;
	case _LOG_STDERR:
	case _LOG_NONE:
	default:
	    log_stream = stderr;
	    break;
    }
}

void log_close(void) {
    if (config->log_dest == _LOG_FILE) {
	fclose(log_stream);
    } else if (config->log_dest == _LOG_SYSLOG) {
	closelog();
    }
}

static void _log(int level, char *fmt, va_list args)
{
    char buffer[2048];
    vsnprintf(buffer, 2048, fmt, args);
    if (config->log_dest != _LOG_SYSLOG) {
    	time_t t = time(NULL);
	char time_str[256];
	char *p = NULL;
	ctime_r(&t, time_str);
	
	p = strchr(time_str, '\n');
	if (p) (*p) = '\0';
	
	fprintf(log_stream ? log_stream : stderr, "%s: %s\n", time_str, buffer);
	fflush(log_stream ? log_stream : stderr);
    } else {
        syslog(level, "%s", buffer);
    }
}

void log_error(char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    _log(LOG_ERR, fmt, args);
    va_end(args);
}

void log_warning(char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    _log(LOG_WARNING, fmt, args);
    va_end(args);
}

void log_info(char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    _log(LOG_INFO, fmt, args);
    va_end(args);
}

void log_debug(debug_section section, char *fmt, ...)
{
    va_list args;

    if ((config->debug & section) == section) {
	va_start(args, fmt);
	_log(LOG_DEBUG, fmt, args);
	va_end(args);
    }
}

