#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <pthread.h>
#include <sys/types.h>

#include "snmp.h"
#include "conf.h"
#include "snmppd.h"
#include "common.h"

void snmp_init(config_t *config) {
	setenv("MIBS", "all", 1);
	setenv("SNMP_PERSISTENT_FILE", "/tmp/snmppd.tmp", 1);

	init_snmp("snmppd");
	if (config->output_options)
	    snmp_out_toggle_options(config->output_options);
}

void snmp_stop(config_t *config) {
	snmp_shutdown("snmppd");
	unlink("/tmp/snmppd.tmp");
}

void snmp_init_session(config_t *config, struct snmp_session *s) {
	snmp_sess_init(s);
	s->version = SNMP_VERSION_1;
	s->retries = config->snmp_retries ? config->snmp_retries : SNMP_DEFAULT_RETRIES;
	s->timeout = config->snmp_timeout ? config->snmp_timeout : SNMP_DEFAULT_TIMEOUT;
	s->remote_port = SNMP_DEFAULT_REMPORT;
}

void snmp_close_session(void *sessp) {
	if (sessp && snmp_sess_session(sessp)) {
	    snmp_sess_close(sessp);
	}
}

int parse_request_type(char *type) {
    if (!strcasecmp(type, "GET")) {
	return SNMP_MSG_GET;
    } else if (!strcasecmp(type, "GETNEXT")) {
	return SNMP_MSG_GETNEXT;
    } else if (!strcasecmp(type, "SET")) {
	return SNMP_MSG_SET;
    }
    return 0;
}

/* Perform the actual SNMP calls and return the retrieved SNMP value */
int snmp_poll_target(config_t *config, target_t * target, struct snmp_session *session, char **buf, int *buf_size)
{
    oid name[MAX_OID_LEN];
    size_t name_length = MAX_OID_LEN;
    void *sessp = NULL;
    struct snmp_pdu *pdu = NULL, *response = NULL;
    struct variable_list *vars = NULL;
    int status;
    object_t *object = NULL;
    int objects = 0;
    int failures = 0;

    session->peername = target->hostname;
    session->community = (u_char *) target->community;
    session->community_len = strlen((char *) target->community);

    sessp = snmp_sess_open(session);

    if (sessp == NULL) {
	log_error("thread %d, fd %d: Couldn't open snmp descriptor when polling %s@%s", target->tid, target->sock, target->objects->objid, target->hostname);
	return (-2);
    }
    pdu = snmp_pdu_create(target->request_type);

    object = target->objects;
    while (object){
	name_length = MAX_OID_LEN;
	if (!snmp_parse_oid(object->objid, name, &name_length)) {
	    log_error("thread %d, fd %d: Invalid object identifier on %s: [%s]\n", target->tid, target->sock, target->hostname, target->objects->objid);
	    failures++;
	}else{
	    snmp_add_null_var(pdu, name, name_length);
	}
	object = object->next;
	objects++;
    }

    if (failures > 0){
	snmp_free_pdu(pdu);
	snmp_sess_close(sessp);
	log_error("thread %d, fd %d: Found failures while polling %s@%s", target->tid, target->sock, target->objects->objid, target->hostname);
	return (-2);
    }
    status = snmp_sess_synch_response(sessp, pdu, &response);

    /* No Response */
    if (status == STAT_TIMEOUT) {
	log_error("poll_target(): Timed out getting result on '%s'", target->hostname);
	snmp_free_pdu(response);
	snmp_sess_close(sessp);
	return (-1);
    }
    /* Error */
    else if (status != STAT_SUCCESS) {
	char *err_str = NULL;
	snmp_sess_error(sessp, NULL, NULL, &err_str);
	log_error("poll_target(): Error on '%s': %s", target->hostname, err_str);
	free(err_str);
	snmp_free_pdu(response);
	snmp_sess_close(sessp);
	return (-2);
    }
    /* Received a response */
    else {
	if (response->errstat != SNMP_ERR_NOERROR) {
	    /* Error with the packet. */
	    log_error("thread %d, fd %d: SNMP Object Error: %s\n", target->tid, target->sock, 
		      snmp_errstring(response->errstat));
	    log_error("thread %d, fd %d: [host=%s] [oid=%s]\n", target->tid, target->sock, target->hostname,
		      target->objects->objid);
	    snmp_free_pdu(response);
	    snmp_sess_close(sessp);
	    return (-2);
	} else {
	    int offset = 0;
	    if (config->debug) log_debug(DEBUG_SNMP, "thread %d, fd %d: Received SNMP response from %s", target->tid, target->sock, target->hostname);
	    vars = response->variables;
	    while (vars){
		    char *var_buf = NULL;
		    size_t var_len = 0;
		    size_t out_len = 0;
		    if (!sprint_realloc_variable((u_char **)&var_buf, &var_len, &out_len, 1, vars->name, vars->name_length, vars)) {
			continue;
		    }
		    *buf = realloc(*buf, offset + out_len + 1 + 1);
		    (*buf)[offset] = '\0';
		    strncat(*buf, var_buf, out_len);
		    free(var_buf);
	    	    offset += out_len;
		    (*buf)[offset] = '\n';
		    offset++;
		    (*buf)[offset] = '\0';
		    vars = vars->next_variable;
	    }
	    *buf_size = offset + 1;
	    if (config->debug) log_debug(DEBUG_SNMP, "thread %d, fd %d: Returning to client: '%s'", target->tid, target->sock, *buf);
	    snmp_free_pdu(response);
	    snmp_sess_close(sessp);
	    return (0);
	}
    }
}
