#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <pthread.h>

#define __USE_XOPEN
#include <sys/poll.h>
#include <sys/types.h>
//#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <pwd.h>
	      
#include <splitter.h>

#include "snmp.h"
#include "common.h"
#include "snmppd.h"
#include "utils.h"
#include "control.h"
#include "queue.h"
#include "threads.h"
#include "log.h"


char *conf_file = NULL;

static int is_running = 1;

static unsigned int idle_threads;

static void term_handler() {
    log_info("Terminating on signal...");
    is_running = 0;
}

static void cleanup(config_t *config)
{
	wnode *mywork;

	while (queue_size(&wq.work)) {
	    log_debug(DEBUG_FLOW, "%s:%d Waiting for queue to be empty", __FILE__, __LINE__);
	    sleep(1);
	}
	log_debug(DEBUG_THREADS, "%s:%d deactivating work queue...", __FILE__,
			  __LINE__);
	control_deactivate(&wq.control);
	while ((mywork = (wnode *) queue_get(&wq.work)) != NULL) {
		free(mywork);
	}
	join_all_threads();
	cleanup_structs();
	snmp_stop(config);
	unlink(config->pid_file);
	log_close();
	free_conf(&config);
}

static void parse_options(config_t *config, int argc, char ** argv) {
	int ch;
	while ((ch = getopt(argc, argv, "u:c:fvhd:Xl:")) != EOF) {
		switch ((char) ch) {
		case 'c':
			conf_file = optarg;
			break;
		case 'u':
			config->user = optarg;
			break;
		case 'f':
			config->nofork = 1;
			break;
		case 'v':
			config->verbose = 1;
			break;
		case 'd':
			config->verbose = 1;
			config->debug = atoi(optarg);
			break;
		case 'p':
			config->port = atoi(optarg);
			break;
		case 'X':
			config->verbose = 1;
			config->nofork = 1;
			config->debug = 65535;
			config->log_dest = _LOG_STDOUT;
			break;
		case 'O':
			config->output_options = optarg;
			break;
		case 'l':
			if (!strcmp(optarg, "syslog")) {
			    config->log_dest = _LOG_SYSLOG;
			} else if (!strcmp(optarg, "stdout")) {
			    config->log_dest = _LOG_STDOUT;
			} else if (!strcmp(optarg, "stderr")) {
			    config->log_dest = _LOG_STDERR;
			} else {
			    config->log_dest = _LOG_FILE;
			    config->log_file = optarg;
			}
			break;
		case 'h':
		default:
			usage(argv[0]);
			break;
		}
	}
}

static target_t *new_target(config_t *config, int csockfd, char *buf) {
	target_t *entry = NULL;
	char **params = NULL;
	int j = 0, k = 0;
	object_t *obj_tail = NULL;
		    
	log_debug (DEBUG_FLOW, "fd %d: Serving client with request '%s'", csockfd, buf);

	params = split(buf, " ", SPLIT_INFINITE, SPLIT_REPEAT);
	j = 0;
	while (params[j] != NULL){
	    j++;
	}
	if ( j > 0 && !strncasecmp(params[0], "KEEPALIVE", 9)) {
	    k = 1;
	}

	if (j < 3 + k) {
	    log_debug(DEBUG_FLOW, "fd %d: Incomplete parameters", csockfd);
	    write(csockfd, "ERROR\n", 6);
	    split_free(params);
	    return NULL;
	}

	if (config->debug & DEBUG_FLOW) {
	    j = 3 + k;
	    while (params[j] != NULL){
		log_debug(DEBUG_FLOW, "fd %d: Got param %d: %s", csockfd, j - 3 - k, params[j]);
		j++;
	    }
	}
	entry = malloc(sizeof(target_t));
	memset(entry, 0, sizeof(target_t));
	entry->sock = csockfd;
	if (k > 0) entry->keepalive = 1;
	entry->request_type = parse_request_type(params[0 + k]);
	entry->hostname = strdup(params[1 + k]);
	entry->community = strdup(params[2 + k]);
	j = 3 + k;
	while ((params[j] != NULL) && (params[j][0] != '\0')){
	    object_t *tmp_obj = NULL;

	    tmp_obj = (object_t *)malloc(sizeof(object_t));
	    memset(tmp_obj, 0, sizeof(object_t));
			
	    strncpy(tmp_obj->objid, params[j], MAX_SNMPPD_OID_LEN);
	    log_debug(DEBUG_TARGETS, "fd %d: Added object id to target struct: %s", csockfd, tmp_obj->objid);
	    tmp_obj->next = NULL;
	    if (!obj_tail){
		entry->objects = tmp_obj;
		obj_tail = tmp_obj;
	    } else {
		obj_tail->next = tmp_obj;
		obj_tail = obj_tail->next;
	    }
	    j++;
	}
	split_free(params);

	return (entry);
}

int main(int argc, char *argv[])
{
    wnode *mywork;
    char buf[MAX_REQUEST_LEN];
    struct sockaddr_in addr;
    int ssockfd = 0;
    int ctot, i, n, nready;
    int one = 1;
    int cycle = 0;
    struct pollfd *pollfds = NULL;
    
    config = malloc(sizeof(config_t));
    memset(config, 0, sizeof(config_t));

    parse_options(config, argc, argv);

    if (!conf_file) conf_file = CONFIG_FILE;

    if (!read_conf(conf_file, config)) {
	exit(1);
    }

    if (config->log_dest == _LOG_NONE) config->log_dest = _LOG_STDERR;
    if (!config->port) config->port = SNMPPD_PORT;
    if (!config->start_threads) config->start_threads = NUM_CONSUMERS;
    if (!config->min_threads) config->min_threads = NUM_CONSUMERS;
    if (!config->max_threads) config->max_threads = NUM_CONSUMERS * 2;
    if (!config->min_idle_threads) config->min_idle_threads = 0;
    if (!config->max_idle_threads) config->max_idle_threads = 1;
    if (!config->pid_file) config->pid_file = strdup("/var/run/snmppd/snmppd.pid");

    log_init("snmppd");

    memset(buf, 0 , MAX_SNMPPD_OID_LEN);

    signal(SIGINT, term_handler);
    signal(SIGQUIT, term_handler);
    signal(SIGTERM, term_handler);

    if ((ssockfd = socket(PF_INET, SOCK_STREAM, 0)) == -1){
	log_error("Can't create socket: %s", strerror(errno));
	exit(1);
    }

    if (setsockopt(ssockfd, SOL_SOCKET, SO_REUSEADDR,
        (char *)&one, sizeof(int)) < 0) {
	log_error("Can't setsockopt(SO_REUSEADDR)");
	close(ssockfd);
    }    

    memset(&addr, 0 ,sizeof(addr));
    addr.sin_addr.s_addr = htonl (INADDR_LOOPBACK);
    addr.sin_family = AF_INET;
    addr.sin_port = htons(config->port);
    
    if (bind(ssockfd, (struct sockaddr *) &addr, sizeof(addr)) == -1) {
	log_error ("Can't bind: %s", strerror(errno));
	exit(1);
    }

    if (listen(ssockfd, 5) == -1){
	log_error ("Can't listen: %s", strerror(errno));
	exit(1);
    }

    /* Initialize pollfds array with server, client info */
    ctot = 0;			// Just the server initially, no clients

    pollfds = malloc(sizeof(struct pollfd) * config->max_threads);

    if (!pollfds) {
	log_error ("Can't malloc: %s", strerror(errno));
	exit(1);
    }

    memset(pollfds, 0, sizeof(struct pollfd) * config->max_threads);
    for (i = 0; i < config->max_threads; i++) {
	if (i == 0) {
	    pollfds[i].fd = ssockfd;
	    pollfds[i].events = POLLRDNORM;
	} else {
	    pollfds[i].fd = -1;
	}
    }
    if (check_pid(config)) {
	exit(1);
    }
    /* Drop privileges */
    if (!config->debug && config->user && strcmp(config->user, "root")) {
	struct passwd *pwd = getpwnam(config->user);
	if (pwd) {
	    setuid(pwd->pw_uid);
	} else {
	    log_error("User %s doesn't exist.", config->user);
	    exit(1);
	}
    }

    if (!config->nofork) {
	if (fork() != 0) {
	    exit(0);
	}
	fclose(stdin);
	if (config->log_dest != _LOG_STDOUT) {
	    fclose(stdout);
	}
	if (config->log_dest != _LOG_STDERR && config->log_dest != _LOG_NONE) {
	    fclose(stderr);
	}
	chdir("/");
	setsid();
    }

    if (write_pid(config))
	exit(1);

    snmp_init(config);

    initialize_structs();

    if (create_threads(config, config->start_threads)) {
	log_error("Error starting threads... cleaning up.");
	join_all_threads();
	exit(1);
    }

    while (is_running) {
	cycle++;
	if (config->debug) log_debug (DEBUG_FLOW, "server waiting");
	nready = poll (pollfds, ctot + 1, -1);

      /* Server, need to add new client */
	if (pollfds[0].revents & POLLRDNORM){
	    int csockfd = 0;
	    socklen_t clen = 0;
	    struct sockaddr_in caddr;

	    if (config->debug) log_debug (DEBUG_FLOW, "new client: %d", ctot + 1);
	    clen = sizeof (caddr);
	    csockfd = accept (ssockfd, (struct sockaddr *) &caddr, &clen);
	    for (i = 1; i < config->max_threads; i++)
		if (pollfds[i].fd < 0){
			pollfds[i].fd = csockfd;
			break;
		}

	    if (i == config->max_threads) {
		log_error("Too many clients (%u), increase MaxThreads in config file", i);
		i--;
//		exit (1);
	    }

	    pollfds[i].events = POLLRDNORM;
	    if (i > ctot)
		ctot = i;

	    if (--nready <= 0)
		continue;
	}

        log_debug(DEBUG_THREADS, "ctot = %u, idle_threads = %u, max_idle_threads = %u, min_idle_threads = %u",
		ctot, idle_threads, config->max_idle_threads, config->min_idle_threads);
	if (ctot > idle_threads) {
    	    log_debug(DEBUG_THREADS, "Adding %u threads\n", ctot - idle_threads);
	    create_threads(config, ctot - idle_threads);
	} else if (idle_threads - ctot > config->max_idle_threads) {
    	    log_debug(DEBUG_THREADS, "Deleting %u threads\n", idle_threads - ctot - config->min_idle_threads);
	    del_threads(config, idle_threads - ctot - config->min_idle_threads);
	}

      /* Now service clients in ascending order */
	for (i = 1; i <= ctot; i++) {
	    int csockfd = 0;
	    if ((csockfd = pollfds[i].fd) < 0)
		continue;

	    if (config->debug & DEBUG_FD) log_debug(DEBUG_FD, "fd %d, i %d: ", csockfd, i);
	    if (pollfds[i].revents & (POLLRDNORM | POLLERR)){
		if (config->debug & DEBUG_FD) log_debug (DEBUG_FD, "cycle %d, fd %d: Reading", cycle, csockfd);
		if ((n = read (csockfd, &buf, MAX_REQUEST_LEN)) < 0){
		    if (config->debug & DEBUG_FD) log_debug(DEBUG_FD, "fd %d: Error on socket (%s)", csockfd, strerror(errno));
		    close (csockfd);
		    pollfds[i].fd = -1;
		    continue;
		}
		else if (n == 0){
		    close (csockfd);
		    if (config->debug & DEBUG_FLOW) log_debug (DEBUG_FLOW, "fd %d: Removing client", csockfd);
		    pollfds[i].fd = -1;
		    continue;
		}
		else {
		    char *p = NULL;
		    int k = 0;
		    target_t *entry = NULL;
		    buf[n] = '\0';
		    while ((p = strchr(buf, '\n')) != NULL){
			*p = '\0';
		    }
		    while ((p = strchr(buf, '\r')) != NULL){
			*p = '\0';
		    }
		    if (!strncasecmp(buf, "QUIT", 5)) {
			write(csockfd, "GOOD BYE\n", 9);
			close (csockfd);
			pollfds[i].fd = -1;
			continue;
		    }

		    entry = new_target(config, csockfd, buf);

		    if (!entry) {
			continue;
		    }
		    k = entry->keepalive;

		    pthread_mutex_lock(&wq.control.mutex);
		    mywork = malloc(sizeof(wnode));
		    memset(mywork, 0, sizeof(wnode));

		    if (!mywork) {
			log_error("ouch! can't malloc!");
			break;
		    }
		    //c++;
		    mywork->next = NULL;
		    mywork->entry = entry;
		    queue_put(&wq.work, (my_node *) mywork);

		    if (!k) pollfds[i].fd = -1;

		    if (config->debug & DEBUG_FLOW) log_debug(DEBUG_FLOW, "%d elements in queue after put", queue_size(&wq.work));
		    pthread_mutex_unlock(&wq.control.mutex);
		    if (config->debug & DEBUG_FLOW) log_debug(DEBUG_FLOW, "Sending signal to workers");
		    pthread_cond_signal(&wq.control.cond);
		}

		if (--nready <= 0)
			break;

	    }

	}
    }
    if (config->debug & DEBUG_FLOW) log_debug (DEBUG_FLOW, "Parent exiting");
    cleanup(config);
    free(pollfds);
    pollfds = NULL;
    exit(0);
}



void *thread_poll(void *myarg)
{
    wnode *mywork;
    cnode *mynode;
    char *buf = NULL;
    int buf_len = 0;
    int result;
    struct snmp_session thread_session;
    unsigned long long counter = 0ULL;

    sigset_t set;
    sigemptyset(&set);
    sigaddset(&set, SIGHUP);
    sigaddset(&set, SIGINT);
    sigaddset(&set, SIGQUIT);
    sigaddset(&set, SIGPIPE);
    pthread_sigmask(SIG_BLOCK, &set, NULL);

    mynode = (cnode *) myarg;

    snmp_init_session(config, &thread_session);

    pthread_mutex_lock(&wq.control.mutex);
    idle_threads++;
    while (wq.control.active && get_thread_status(mynode->threadnum)) {
	while (wq.work.head == NULL && wq.control.active && get_thread_status(mynode->threadnum)) {
	    if (config->debug) log_debug(DEBUG_FLOW, "Thread %d is waiting for control condition",
		      mynode->threadnum);
	    pthread_cond_wait(&wq.control.cond, &wq.control.mutex);
	}
	if (!get_thread_status(mynode->threadnum))
	    break;
	if (!wq.control.active)
	    break;
	//we got something!
	mywork = (wnode *) queue_get(&wq.work);
	if (config->debug) log_debug(DEBUG_FLOW, "%d elements remains in queue after get", queue_size(&wq.work));
	idle_threads--;
	pthread_mutex_unlock(&wq.control.mutex);
	if (config->debug) log_debug(DEBUG_FLOW, "Thread number %d got job", mynode->threadnum);
	//perform processing...
	mywork->entry->tid = mynode->threadnum;

	if (config->debug) log_debug(DEBUG_FLOW, "Thread number %d processing job %s@%s",
		  mynode->threadnum, mywork->entry->objects->objid,
		  mywork->entry->hostname);
	result = snmp_poll_target(config, mywork->entry, &thread_session, &buf, &buf_len);
	if (result != 0) {
	    if (config->debug) log_debug(DEBUG_FLOW, "thread %d, fd %d: Error getting result", mywork->entry->tid, mywork->entry->sock);
	    write(mywork->entry->sock, "ERROR\n", 6);
	} else {
	    if (config->debug) log_debug(DEBUG_FLOW, "thread %d, fd %d: Got result '%s'", mywork->entry->tid, mywork->entry->sock, buf);
	    write(mywork->entry->sock, buf, buf_len);
	}

	if (!mywork->entry->keepalive) {
	    shutdown(mywork->entry->sock, SHUT_RDWR);
	    close(mywork->entry->sock);
	}
	counter++;
	free_entry(&mywork->entry);
	free(mywork);
	if (buf) {
	    free(buf);
	    buf = NULL;
	    buf_len = 0;
	}
	pthread_mutex_lock(&wq.control.mutex);
	idle_threads++;
    }
    pthread_mutex_unlock(&wq.control.mutex);
    pthread_mutex_lock(&cq.control.mutex);
    queue_put(&cq.cleanup, (my_node *) mynode);
    idle_threads--;
    pthread_mutex_unlock(&cq.control.mutex);
    pthread_cond_signal(&cq.control.cond);
    if (config->verbose)
    log_info("thread %d shutting down after %llu polls.",
	     mynode->threadnum, counter);
    if (config->debug) log_debug(DEBUG_FLOW, "thread %d shutting down...", mynode->threadnum);
    return NULL;


}

void free_entry(target_t **entry){
    while ((*entry)->objects){
        object_t *tmp_obj = (*entry)->objects;
        (*entry)->objects = (*entry)->objects->next;
	free(tmp_obj);
    }
    free((*entry)->hostname);
    free((*entry)->community);
    free(*entry);
    *entry = NULL;
}

void usage(char *prog)
{
	printf("%s v%s\n", PACKAGE, VERSION);
	printf("Usage: %s [-u <user>] [-d][-f][-v] [-c <config_file>] \n", prog);
	printf("       -u: run with <user> privileges\n");
	printf("       -d: enable debug\n");
	printf("       -f: stay foreground\n");
	printf("       -v: be more verbose\n");
	exit(-1);
}
