#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>

#include "threads.h"
#include "control.h"
#include "queue.h"
#include "conf.h"
#include "log.h"

static unsigned int numthreads = 0;
work_queue wq;
cleanup_queue cq;

static char *thread_map = NULL;

unsigned int get_num_threads(void) {
    return numthreads;
}

static void set_thread_status(unsigned int thread) {
    pthread_mutex_lock(&wq.control.mutex);
    if (thread_map) thread_map[thread] = 1;
    pthread_mutex_unlock(&wq.control.mutex);
}

static void clear_thread_status(unsigned int thread) {
    pthread_mutex_lock(&wq.control.mutex);
    if (thread_map) thread_map[thread] = 0;
    pthread_mutex_unlock(&wq.control.mutex);
}

int get_thread_status(unsigned int thread) {
    log_debug(DEBUG_THREADS, "thread %d has status %d", thread, thread_map[thread]);
    if (thread_map) return thread_map[thread];
    return 0;
}

static void join_threads(int num_threads_to_join)
{
	int threads_joined = 0;
	cnode *curnode;
	log_debug(DEBUG_THREADS, "Joining %u threads...", num_threads_to_join);
	while (numthreads && threads_joined < num_threads_to_join) {
		pthread_mutex_lock(&cq.control.mutex);
		/* below, we sleep until there really is a new cleanup node.  This
		   takes care of any false wakeups... even if we break out of
		   pthread_cond_wait(), we don't make any assumptions that the
		   condition we were waiting for is true.  */
		while (cq.cleanup.head == NULL) {
			pthread_cond_wait(&cq.control.cond, &cq.control.mutex);
		}
		/* at this point, we hold the mutex and there is an item in the
		   list that we need to process.  First, we remove the node from
		   the queue.  Then, we call pthread_join() on the tid stored in
		   the node.  When pthread_join() returns, we have cleaned up
		   after a thread.  Only then do we free() the node, decrement the
		   number of additional threads we need to wait for and repeat the
		   entire process, if necessary */
		curnode = (cnode *) queue_get(&cq.cleanup);
		pthread_mutex_unlock(&cq.control.mutex);
		pthread_join(curnode->tid, NULL);
		log_debug(DEBUG_THREADS, "joined with thread %d",
				  curnode->threadnum);
		free(curnode);
		numthreads--;
		threads_joined++;
		log_debug(DEBUG_THREADS,"%u threads joined, waiting for %u, %u remains",
			threads_joined, num_threads_to_join - threads_joined, numthreads);
	}
}

void join_all_threads(void) {
    join_threads(numthreads);
}

static int create_thread(config_t *config, int thread_num, pthread_attr_t *attr) {
	cnode *curnode;
	curnode = malloc(sizeof(cnode));
	if (!curnode)
		return 1;
	curnode->threadnum = thread_num;
	set_thread_status(curnode->threadnum);
	if (pthread_create(&curnode->tid, attr, thread_poll, (void *) curnode)) {
		clear_thread_status(curnode->threadnum);
		return 1;
	}
	log_debug(DEBUG_THREADS, "created thread %d", thread_num);
	numthreads++;
	return 0;
}

int create_threads(config_t *config, unsigned int num_threads_to_add) {
	unsigned int x;
	unsigned int last_thread = numthreads + num_threads_to_add;
	pthread_attr_t attr;
	pthread_attr_init(&attr);
	for (x = numthreads; x <= last_thread; x++) {
	    if (x >= config->max_threads) {
		log_error("Can not add thread number %u, increase MaxThreads in config file", x);
		return 1;
	    }
	    if (create_thread(config, x, &attr)) {
		return 1;
	    }
	}
	return 0;
}

int del_threads(config_t *config, unsigned int num_threads_to_del) {
    int x;
    int threads_deleted = 0;
    for (x = numthreads - 1; x > numthreads - num_threads_to_del - 1; x--) {
	if (x > config->min_threads) {
	    log_debug(DEBUG_THREADS, "deleting thread %d", x);
	    clear_thread_status(x);
	    threads_deleted++;
	}
    }
    pthread_cond_broadcast(&wq.control.cond);
    join_threads(threads_deleted);
    return 0;
}


void initialize_structs(void)
{
	if (control_init(&wq.control)) {
		log_error("%s:%d Aborting...", __FILE__, __LINE__);
		exit(1);
	}
	queue_init(&wq.work);
	if (control_init(&cq.control)) {
		control_destroy(&wq.control);
		log_error("%s:%d Aborting...", __FILE__, __LINE__);
		exit(1);
	}
	queue_init(&cq.cleanup);
	control_activate(&wq.control);
	thread_map = malloc(sizeof(char) * config->max_threads);
	memset(thread_map, 0, sizeof(char) * config->max_threads);
}

void cleanup_structs(void)
{
	control_destroy(&cq.control);
	control_destroy(&wq.control);
	free(thread_map);
	thread_map = NULL;
}
