#!/bin/sh
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

PROGNAME=`basename $0`
VERSION="Version 1.1,"
AUTHOR="2014, Javier Polo Cozar"

ST_OK=0
ST_WR=1
ST_CR=2
ST_UK=3

SNMPGET=`which snmpget`
GREP=`which grep`
CUT=`which cut`

# OID for HP Procurve Switches. Change it to satisfy your needs.

OID_TOTALMEM="1.3.6.1.4.1.11.2.14.11.5.1.1.2.1.1.1.5.1"
OID_FREEMEM="1.3.6.1.4.1.11.2.14.11.5.1.1.2.1.1.1.6.1"
OID_ALLOCMEM="1.3.6.1.4.1.11.2.14.11.5.1.1.2.1.1.1.7.1"

# We need bc application  for working with fractionnary numbers,
#  used in unit conversion tasks
BC=`which bc`

print_version() {
    echo "$VERSION $AUTHOR"
}

print_help() {
    print_version $PROGNAME $VERSION
    echo ""
    echo "$PROGNAME is a Nagios plugin to monitor HP Procurve Switches Memory (total, free, used) through SNMP"
    echo ""
    echo "$PROGNAME  -h IP address [-v SNMP version] [-C SNMP community] [-u/--unit <unit measure>] [-w/--warning <warning limit %>] [-c/--critical <critical limit %>] [-u/--unit <unit type>]"
    echo ""
    echo "Examples: $PROGNAME -h 10.71.8.30 -v 2c -C public"
    echo " $PROGNAME -h 10.71.8.30 -C public -w 75 -c 90 -u Mb"
    echo ""
    echo "Options:"
    echo " -h IP Address: sets IP address of the HP switch"
    echo "  -v [1|2c]: sets SNMP version to use. Default is 2c" 
    echo "  -C COMMUNITY: sets the community string. Default is public"
    echo "  --help: displays this help message"
    echo "  --warning|-w <warning limit %>): Sets a warning level for size. Default is: off"
    echo "  --critical|-c <critical limit %>): Sets a critical level for size. Default is: off"
    echo "  --unit|-u [Kb|KB|Mb|MB|Gb|GB]: Sets output in specific format: Kb, Mb or Gb. Default is: Mb"
    echo ""

    exit $ST_UK
}

if test -z "$1" 
then
	echo "No command-line arguments."
	print_help
	exit $ST_UK 

else
# By default, output is given in MBytes
unit=Mb

# By default, SNMP version is 2c
version=2c

#By default, community name is public
community=public

while test -n "$1"; do
    case "$1" in
        --help)
            print_help
            exit $ST_UK
            ;;
        --version)
            print_version $PROGNAME $VERSION
            exit $ST_UK
            ;;
        -v)
            version=$2
            shift
            ;;
	-C)
	    community=$2
	    shift
	    ;;
	-h)
	    host=$2
	    shift
	    ;;
	--unit|-u)
	    unit=$2
	    shift
	    ;;
        --warning|-w)
            warn=$2
            shift
            ;;
        --critical|-c)
            crit=$2
            shift
            ;;
        *)
            echo "Unknown argument: $1"
            print_help
            exit $ST_UK
            ;;
    esac
    shift
done
fi

val_wcdiff() {
    if [ ${warn} -gt ${crit} ]
    then
        wcdiff=1
    fi
}

check_host() {
IP_OK=$(echo "$host" | grep -Ec '^(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[1-9])\.(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[1-9]|0)\.(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[1-9]|0)\.(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[0-9])')

if [ $IP_OK -eq 0 ]; then
	echo "Wrong IP address format: $host\n"
	exit $ST_UK
	
fi
}


get_volume_size() {

$SNMPGET -v $version -c $community $host -m ALL $OID_TOTALMEM >/dev/null 2>&1
if [ $? -ne 0 ]; then
echo "Timeout: No response from $host"
exit $ST_UK
else

totalmemsize=`$SNMPGET -v $version -c $community $host -m ALL $OID_TOTALMEM`
freememsize=`$SNMPGET -v $version -c $community $host -m ALL $OID_FREEMEM`
allocmemsize=`$SNMPGET -v $version -c $community $host -m ALL $OID_ALLOCMEM`

# mem size in Bytes
totalsizeB=`echo $totalmemsize|$CUT -f 4 -d":"`
freesizeB=`echo $freememsize|$CUT -f 4 -d":"`
usedsizeB=`echo $allocmemsize|$CUT -f 4 -d":"`


#usedsizeB=`expr $totalsizeB - $freesizeB`

# Volume size in %
percentused=`expr $usedsizeB \* 100 / $totalsizeB`
percentfree=`expr 100 - $percentused`
fi

}

# Conversion to chose unit
do_unit_conversion() {

case "$unit" in
	Kb|KB)		
	totalsize=$(echo "scale=2; $totalsizeB / 1024"|$BC)
	freesize=$(echo "scale=2; $freesizeB / 1024"|$BC)
	usedsize=$(echo "scale=2; $usedsizeB / 1024"|$BC)
	;;

	Mb|MB)
	totalsize=$(echo "scale=2; $totalsizeB / 1048576"|$BC)
	freesize=$(echo "scale=2; $freesizeB / 1048576"|$BC)
	usedsize=$(echo "scale=2; $usedsizeB / 1048576"|$BC)
	;;

	Gb|GB)
	totalsize=$(echo "scale=2; $totalsizeB / 1073741824"|$BC)
	freesize=$(echo "scale=2;  $freesizeB / 1073741824"|$BC)
	usedsize=$(echo "scale=2; $usedsizeB / 1073741824"|$BC)
	;;

	*)
	echo "Unknown unit type: $unit"
            print_help
            exit $ST_UK
            ;;

esac
}

# Setting warning and critical level in unit selected
do_warning_critical_settings() {

if [ -n "$warn" -a -n "$crit" ]
then
#In Bytes
warningsize=$(echo "scale=2; $warn * $totalsizeB / 100"|$BC)
criticalsize=$(echo "scale=2;  $crit * $totalsizeB / 100"|$BC)

case "$unit" in

	Kb|KB)
        warningsize=$(echo "scale=2; $warningsize / 1024"|$BC)
        criticalsize=$(echo "scale=2; $criticalsize / 1024"|$BC)
        ;;

        Mb|MB)
        warningsize=$(echo "scale=2; $warningsize / 1048576"|$BC)
        criticalsize=$(echo "scale=2; $criticalsize / 1048576"|$BC)
        ;;

        Gb|GB)
        warningsize=$(echo "scale=2; $warningsize / 1073741824"|$BC)
        criticalsize=$(echo "scale=2; $criticalsize / 1073741824"|$BC)
        ;;
esac
fi
}

do_output() {

	output="Memsize - total: $totalsize $unit - used: $usedsize $unit ($percentused%)- free: $freesize $unit ($percentfree%)"
}

do_perfdata() {
	
	perfdata="Memsize=${usedsize}$unit;$warningsize;$criticalsize;;$totalsize"
#	perfdata="$vmfs=${percentused}%;$warn;$crit;;100"
}

if [ "$SNMPGET" = "" ]
then
	echo"\nsnmpget must be previously installed to access HP Procurve Switches. Try to execute this command if you are in a debian based linux system:\n"
	echo "$sudo apt-get install snmp\n"
	echo "\nIf you are in a red hat based linux, you could try with:\n"
	echo "$sudo yum install snmp\n"
	exit $ST_UK
fi


if [ "$BC" = "" ]
then
        echo "\nbc application must be previously installed. Try to execute next command if you are in a debian based linux:\n"
        echo "$ sudo apt-get install bc\n"
        echo "\nIf you are in a red hat based linux, you could try with:\n"
        echo "$ sudo yum install bc\n"
        exit $ST_UK
fi

if [ -n "$warn" -a -n "$crit" ]
then
    val_wcdiff
    if [ "$wcdiff" = 1 ]
    then
		echo "Please adjust your warning/critical thresholds. "
		echo "The warning must be lower than the critical level!"
        exit $ST_UK
    fi
fi

check_host
get_volume_size
do_unit_conversion
do_warning_critical_settings
do_output
do_perfdata

if [ -n "$warn" ] && [ -n "$crit" ]; then
    if [ "$percentused" -ge "$warn" -a "$percentused" -lt "$crit" ]
    then
		echo "WARNING - ${output} | ${perfdata}"
        exit $ST_WR
    elif [ "$percentused" -ge "$crit" ]
    then
		echo "CRITICAL - ${output} | ${perfdata}"
        exit $ST_CR
    else
		echo "OK - ${output} | ${perfdata}"
        exit $ST_OK
    fi
else
      if [ -n "$warn" ]; then
	if [ "$percentused" -ge "$warn" ]; then
		echo "WARNING - ${output} | ${perfdata}"
		exit $ST_WR
	else 
		echo "OK - ${output}|${perfdata}"
		exit $ST_OK
	fi
      elif [ -n "$crit" ]; then
	if [ "$percentused" -ge "$crit" ]; then
		echo "CRITICAL - ${output} |${perfdata}"
		exit $ST_CR
	else
	echo "OK - ${output} | ${perfdata}"
    	exit $ST_OK
	fi
     else
	echo "OK - ${output}|${perfdata}"
	exit $ST_OK
     fi
	
fi
