

    use Carp;
    use strict;
    use Data::Dumper;
    use Asterisk::AMI;
    use Time::HiRes qw/gettimeofday/;

    # Constants
    my $DOT  = q{.};
    my $NULL = q{};

=item get_fax_stats( )

Query the AMI and issue a "fax show stats" command. For Asterisk 1.8 and above
and SpanDSP. Returns a hashref.

$hashref  = get_fax_stats( );

=cut


    sub get_fax_stats {

        my $hashref = shift;
        my %params  = %$hashref;

        # If $DEBUG is set to 1 for testing, then values will be randomly 
        # generated by calling gen_rand_num( ). The value will be an integer
        # between 'min' and 'max' inclusive.
        my $DEBUG = '0';

        my $astman  = Asterisk::AMI->new( PeerAddr => "$params{ ami }{ host    }",
                                          PeerPort => "$params{ ami }{ port    }",
                                          Username => "$params{ ami }{ manager }",
                                          Secret   => "$params{ ami }{ secret  }"
        );

        croak "Unable to connect to asterisk" unless $astman;
        my $actionid = $astman->send_action( { Action  => 'Command',
                                               Command => 'fax show stats',
                                               $params{ ami }{ timeout } } );
        my $response = $astman->get_response( $actionid );
        my $arrayref = $response->{ CMD };
        my @array    = @$arrayref;
        my $null     = qq{};
        my ( %faxstats, $section );
        foreach my $line ( @array ) {
            next if ( ( ! $line ) || ( $line =~ /----/ ) );
            my ( $key, $value ) = split( ':', $line );
            $section = $key if ( $value eq $null );
            $key     =~ s/\s+$//g;
            $value   =~ s/^\s+//g;
            if( $DEBUG ) {
                if( $value ne $null ) {
                    $value   = gen_rand_num( min => 1, max => 50 );
                };
            };
            $faxstats{ "$section" }{ "$key" } = $value if ( $value ne $null );
        };
        my $goodbye = $astman->disconnect( );
        sleep( 1 );

        return( \%faxstats );

    };


=item read_config( )

Read the config parameters from a standard .ini file. Return a hashref that points
to a hash of hashes. We strip out any trailing comments and trailing whitespace
after the value.

Example: $hashref = read_config( $ini )

=cut


    sub read_config {
        my $cfgfile = shift;
        my %config  = ( );
        open( my $cfgfh, '<', $cfgfile )
                || croak "Cannot open config file $cfgfile for reading: $!";
        my @lines = map { split /\n/ } <$cfgfh>;
        my $section;
        foreach my $line ( @lines ) {
            $line =~ s/\r|\n|^\s+|\s+$|;.*$|#.*$//g;
            $line =~ /^\[.+\]/ and do {
                $section = $line;
                $section =~ s/^\[|\]//g;
                next;
            };
            $line =~ /^.+=/ and do {
                my( $key, $value ) = split /\s*=\s*/, $line, 2;
                $value =~ s/^\s+|\s+$//; # Trim whitespace
                $config{ $section }{ $key } = $value;
                next;
            };
            $line =~ /^.+=.+$/ and do {
                my( $key, $value ) = split /\s*=\s*/, $line, 2;
                $value =~ s/^\s+|\s+$//; # Trim whitespace
                $config{ $section }{ $key } = $value;
                next;
            };
        };
        return( \%config );
    };


=item get_my_name( )

Takes $0 which contains the invocation of this script. It strips off everything
(the path) and returns just the filename. Does not require the module File::Basename.
$build[ 1 ] = the path that was stripped off, if any, and $build[ 0 ] = the file
name.

Example: my $basename = get_my_name( $0 )

where $0 = '/usr/local/nagios/libexec/check_sip_registry'
    or
      $0 = './check_sip_registry'
    or
      $0 = 'check_sip_registry'

and returns $basename = check_sip_registry

=cut

    sub get_my_name {
        my $pathfile = shift;
        my @split    = ( );
        my @build    = ( );
        @split       = split( /\//, $pathfile );
        push( @build, pop( @split ) );
        push( @build, join( '/', @split ) . '/' ) if ( @split );
        $build[ 1 ]  =~ s/\/$//g;
        return( $build[ 0 ] );
    };


=item split_path_and_file( $filepath )

Takes path with filename and returns either an array or arrayref of ( $filename, $path ). 
Requires the full pathname.

Example:

$fullpath = "/usr/local/bin/file.pl";
( $file, $path ) = split_path_and_file( $fullpath )

where $file = file.pl and $path = /usr/local/bin

=cut

    sub split_path_and_file {
        my $pathfile = shift;
        my @split    = split( /\//, $pathfile );
        my @build;
        push( @build, pop( @split ) );
        push( @build, join( '/', @split ) . '/' ) if ( @split );
        $build[ 1 ] =~ s/\/$//g;
        return( wantarray ? @build : \@build );
    };

=item get_epochtime( )

Return the current time as seconds since the epoch. Takes the format xxxxxxxxxx or
XXXXXXXXXX.xxxxxx. Requires the module Time::HiRes qw/gettimeofday/. Uses named
argument units => 'sec' or units => 'usec'. Defaults to 'usec'.

Example: $time = get_epochtime( units => 'usec' )
   where $time = 1438610015.828835


=cut



    sub get_epochtime {

        my %args  = @_;
        my $units = $args{ units };
        $units    = 'usec' if ! defined $units;

        my $time;
        my ( $sec, $usec ) = gettimeofday( );
        $time = $sec if $units eq 'sec';
        $time = $sec . $DOT . $usec if $units eq 'usec';

        return( $time );

    };


=item get_date_time( )

Returns the current date and time in the DateTime format of "YYYY-MM-DD HH:MM:SS".

Ex: $datetime = get_date_time( )
    returns  date_time = 2016-04-28 14:23:02

=cut


    sub get_date_time {

        my %args = @_;

        my $secs;
        $secs = $args{ Secs } if defined $args{ Secs };
        $secs = $args{ secs } if defined $args{ secs };
        $secs = '1' if $secs eq $NULL;

        my ( $sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst ) = localtime( time );

        $year += 1900;
        $mon  += 1;
        $mon  = "0" . $mon  if $mon  < 10;
        $mday = "0" . $mday if $mday < 10;
        $hour = "0" . $hour if $hour < 10;
        $min  = "0" . $min  if $min  < 10;
        $sec = "0"  . $sec  if $sec  < 10;

        my $datetime = "$year" . '-' . "$mon" . '-' . "$mday" . ' '
                     . "$hour" . ':' . "$min" . ':' . "$sec";

        return( $datetime );

    };


=item gen_rand_num( )

Given both a min and a max number, generate a random number inclusive.
Uses named parameters.

Example: $number = gen_rand_num( min => $min, max => $max );

=cut


    sub gen_rand_num {
        my %args = @_;
        my $min  = $args{ min };
        my $max  = $args{ max };
        my $rand = int( rand( $max - $min ) ) + $min;
    };



=item is_module_loaded( )

Given a PERL module, test to see if it is loaded. Returns a 0|1.

Example: my $loaded = is_module_loaded( 'Asterisk::AMI' );


=cut


    sub is_module_loaded {
        my $mod = shift;

        eval( "use $mod" );
        if( $@ ) {
            #print "\$@ = $@\n";
            return( 0 );
        } else {
            return( 1 );
        };
    };



    1;



