#!/usr/bin/perl


    use Carp;
    use strict;
    use Data::Dumper;
    use Getopt::Long;
    Getopt::Long::Configure( 'bundling', 'auto_abbrev', 'ignore_case', 'getopt_compat' );


    #
    # THESE TWO VALUES MUST CONTAIN THE FULL PATH TO THEIR RESPECTIVE FILES. FAILURE
    # TO DO THIS WILL LIKELY RESULT IN THE PLUGIN FAILING!!!
    # 
    my $faxlib   = "faxstats-lib.pl"; # This should be the full path to the file.
    my $ini_file = "faxstats.ini";    # This should be the full path to the .ini file.

    # Test to see if we can find the faxlib and ini_file files.
    if( ! -e $faxlib ) {
        print "\nCannot find the required faxlib $faxlib. Exiting...\n\n";
        exit( 1 );
    };
    if( ! -e $ini_file ) {
        print "\nCannot find the required ini_file $ini_file. Exiting...\n\n";
        exit( 1 );
    };

    require "$faxlib";

    # Did we remember to install the Asterisk::AMI PERL module.
    my $module = 'Asterisk::AMI';
    my $loaded = is_module_loaded( "$module" );
    if( ! $loaded ) {
        print "\nThe required PERL CPAN module $module is not installed. Exiting...\n\n";
        exit( 1 );
    };

    # Constants. 
    my $TIMEOUT    = 15;  
    my %ERRORS     = ( 'OK'        => 0, 'WARNING' => 1, 'CRITICAL' => 2, 'UNKNOWN' => 3, 
                       'DEPENDENT' => 4 );
    my $ETB        = "\027"; # This is required for NSCA v. 2.9.1. When submitting
                             # multiple simultaneous results, separate each result 
                             # with the ETB character (^W or 0x17).
    my $PROGNAME   = get_my_name( $0 );
    my $VERSION    = 'V1.0';
    my $start_time = get_epochtime( );

    my ( $usage_msg, $version_msg, $blurb_msg, $extra_msg, $help_msg, $system );
    my ( $usage, $verbose, $version, $blurb, $extra, $ini, $timeout, $help );
    $timeout   = $TIMEOUT;
    $usage_msg = "\nUsage: $PROGNAME -i|--ini <.ini file> [-t <timeout>] [ -V|--version = print version number]\n";
    $blurb_msg = "\nThis plugin will log into the Asterisk AMI and get the fax statistics.\n"
               . "Assumes that the Digium Fax for Asterisk module ( FFA ) is used. Can\n"
               . "optionally specify warning and critical levels in the .ini file.\n";
    $extra_msg = "\nExample:
        $PROGNAME --ini faxstats.ini
        will return the Asterisk fax statistics";
    $version_msg = "\n$PROGNAME $VERSION\n";
    $help_msg    = "$version_msg $usage_msg $blurb_msg $extra_msg\n";
    $timeout     = $TIMEOUT;

    GetOptions ("verbose|v+"  => \$verbose,
                "version|V"   => \$version,
                "timeout|t=i" => \$timeout,
                "help|h"      => \$help,
                "blurb"       => \$blurb,
                "extra"       => \$extra,
                "usage"       => \$usage,
                "ini|i=s"     => \$ini,
    );
    $ini = $ini_file if ! $ini;

    alarm( $timeout );

    # Parse arguments and process standard ones (e.g. usage, help, version)
    # and perform sanity checking on command line options
    if( $usage ) {
        print "$usage_msg\n";
        exit( 0 );
    };
    if( $blurb ) {
        print "$blurb_msg\n";
        exit( 0 );
    };
    if( $extra ) {
        print "$extra_msg\n";
        exit( 0 );
    };
    if( $help ) {
        print "$help_msg\n";
        exit( 0 );
    };
    if( $version ) {
        print "$version_msg\n";
        exit( 0 );
    };
    if( ! $ini && ! $version && ! $blurb && ! $extra && ! $usage ) {
        $PROGNAME   = uc( $PROGNAME );
        my $message = "The .ini file must be specified.\n $usage_msg\n";
        print "\n$PROGNAME UNKNOWN - $message";
        exit( 0 );
    };

    # Read the .ini file to get the warning and critical thresholds along with
    # the nsca or nsca-ng settings.
    my $hashref     = read_config( $ini );
    my %config      = %$hashref;
    my $use_nsca_ng = $config{ nsca }{ use_nsca_ng };
    my $nscaprog    = $config{ nsca }{ nscaprog    };
    my $nscacfg     = $config{ nsca }{ nscacfg     };
    my $nscahost    = $config{ nsca }{ nscahost    };
    my $nagioshost  = $config{ nsca }{ nagioshost  };

    my %services;
    $services{ 'FAX Statistics' }{ 'Current Sessions'   } = "Fax - Current Sessions";
    $services{ 'FAX Statistics' }{ 'Reserved Sessions'  } = "Fax - Reserved Sessions";
    $services{ 'FAX Statistics' }{ 'Transmit Attempts'  } = "Fax - Transmit Attempts";
    $services{ 'FAX Statistics' }{ 'Receive Attempts'   } = "Fax - Receive Attempts";
    $services{ 'FAX Statistics' }{ 'Completed FAXes'    } = "Fax - Completed FAXes";
    $services{ 'FAX Statistics' }{ 'Failed FAXes'       } = "Fax - Failed FAXes";

    $services{ 'Digium G.711' }{ 'Licensed Channels'  } = "Fax - G.711 Licensed Channels";
    $services{ 'Digium G.711' }{ 'Max Concurrent'     } = "Fax - G.711 Max Concurrent";
    $services{ 'Digium G.711' }{ 'Success'            } = "Fax - G.711 Success";
    $services{ 'Digium G.711' }{ 'Switched to T.38'   } = "Fax - G.711 Switched to T.38";
    $services{ 'Digium G.711' }{ 'Canceled'           } = "Fax - G.711 Canceled";
    $services{ 'Digium G.711' }{ 'No FAX'             } = "Fax - G.711 No FAX";
    $services{ 'Digium G.711' }{ 'Partial'            } = "Fax - G.711 Partial";
    $services{ 'Digium G.711' }{ 'Negotiation Failed' } = "Fax - G.711 Negotiation Failed";
    $services{ 'Digium G.711' }{ 'Train Failure'      } = "Fax - G.711 Train Failure";
    $services{ 'Digium G.711' }{ 'Protocol Error'     } = "Fax - G.711 Protocol Error";
    $services{ 'Digium G.711' }{ 'IO Partial'         } = "Fax - G.711 IO Partial";
    $services{ 'Digium G.711' }{ 'IO Fail'            } = "Fax - G.711 IO Fail";

    $services{ 'Digium T.38' }{ 'Licensed Channels'  } = "Fax - T.38 Licensed Channels";
    $services{ 'Digium T.38' }{ 'Max Concurrent'     } = "Fax - T.38 Max Concurrent";
    $services{ 'Digium T.38' }{ 'Success'            } = "Fax - T.38 Success";
    $services{ 'Digium T.38' }{ 'Canceled'           } = "Fax - T.38 Canceled";
    $services{ 'Digium T.38' }{ 'No FAX'             } = "Fax - T.38 No FAX";
    $services{ 'Digium T.38' }{ 'Partial'            } = "Fax - T.38 Partial";
    $services{ 'Digium T.38' }{ 'Negotiation Failed' } = "Fax - T.38 Negotiation Failed";
    $services{ 'Digium T.38' }{ 'Train Failure'      } = "Fax - T.38 Train Failure";
    $services{ 'Digium T.38' }{ 'Protocol Error'     } = "Fax - T.38 Protocol Error";
    $services{ 'Digium T.38' }{ 'IO Partial'         } = "Fax - T.38 IO Partial";
    $services{ 'Digium T.38' }{ 'IO Fail'            } = "Fax - T.38 IO Fail";

    my @params   = ( 'ami', 'statistics', 'g.711', 't.38' );
    my @sections = ( 'FAX Statistics', 'Digium G.711', 'Digium T.38' );
    my $hashref  = get_fax_stats( \%config );
    my %stats    = %$hashref;
    if( $verbose ) {
        print "\n";
        foreach my $section ( @sections ) {
            for my $key ( keys %{ $stats{ $section } } ) {
                print "stats{ $section }{ $key } = $stats{ $section }{ $key }\n";
            };
        };
        print "\n";
    };


    my ( $code, $nsca_cmd, $output, $service, $warn, $crit, $value );
    my $datetime = get_date_time( );
    $output      = "Last received passive results for the Asterisk fax stats at "
                 . "$datetime from plugin $PROGNAME\n";
    $code        = 'OK';
    $nsca_cmd    = "$nscahost\tAsterisk Fax Check\t$code\t$output$ETB";

    foreach my $section ( @sections ) {
        for my $key ( keys %{ $stats{ $section } } ) {
            $service        = $services{ $section }{ $key };
            my $section_str = 't.38'       if $section =~ /T.38/;
            $section_str    = 'g.711'      if $section =~ /G.711/;
            $section_str    = 'statistics' if $section =~ /Statistics/;

            my $warn_str = lc( $key );
            $warn_str    =~ s/ /_/g;
            $warn_str    = $warn_str . '_warn';

            my $crit_str = lc( $key );
            $crit_str    =~ s/ /_/g;
            $crit_str    = $crit_str . '_crit'; 

            $warn  = $config{ $section_str }{ $warn_str };
            $crit  = $config{ $section_str }{ $crit_str };
            $value = $stats{ $section }{ $key };

            if( $warn ) {
                if( $crit ) {
                    $code = 'OK'       if $value < $warn;
                    $code = 'WARNING'  if $value >= $warn && $value < $crit;
                    $code = 'CRITICAL' if $value >= $crit;
                };
                if( ! $crit ) {
                    $code = 'OK'      if $value < $warn;
                    $code = 'WARNING' if $value >= $warn
                };
            };
            if( ! $warn ) {
                if( $crit ) {
                    $code = 'OK'       if $value < $crit;
                    $code = 'CRITICAL' if $value >= $crit;
                };
                if( ! $crit ) {
                    $code = 'OK';
                };
            };
                
            $output = $service;
            $output =~ s/^Fax - //g;
            $output = $output . " = $value";

            $nsca_cmd .= "$nscahost\t$service\t$code\t$output\n$ETB";
        };
    };

    print "\n\n$nsca_cmd\n";

    my $retval    = `$system /bin/echo -e "$nsca_cmd" | $nscaprog -H $nagioshost -c $nscacfg`;
    my $cmd_bytes = length( $nsca_cmd );
    print "send_nsca results ( $cmd_bytes Bytes sent ): $retval\n" if $verbose;

    my $stop_time    = get_epochtime( );
    my $elapsed_time = $stop_time - $start_time;
    $elapsed_time    = abs( sprintf( "%.2f", $elapsed_time ) );
    print "\nPlugin finished in approx. $elapsed_time secs.\n\n" if $verbose;

    alarm( 0 );

    exit( 0 );

