#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <math.h>
#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>

#define IF_STATS_FILE_PATH "/tmp/check-bandwidth-%s-%s.db"
#define IF_STATS_SAMPLES 5

#define CRIT_VARIATION 0.35
#define WARN_VARIATION 0.25

#define CLOCK_NEXT(i, s) (i + 1) % s
#define CLOCK_PREV(i, s) (i + s - 1) % s 

#define COMPUTE_RATE(x1, x2, t1, t2) ((x2 - x1) / (t2 - t1))

#define STATUS_STRING "Bandwidth %s: TX/RX (Mb/s) %.2f/%.2f/%.1f%%  /  %.2f/%.2f/%.1f%% (%s)"
#define PERFORMANCE_STRING "|in_mbps=%.2f out_mbps=%.2f avg_in_mbps=%.2f avg_out_mbps=%.2f in_variation=%.2f%% out_variation=%.2f%%\n"

struct if_stats {
    uint64_t in_octets[IF_STATS_SAMPLES];
    uint64_t out_octets[IF_STATS_SAMPLES];
    time_t last_update[IF_STATS_SAMPLES];
    int index;
};

void help();
char *ifname_sanitize(char *fn, int len);
char *my_snmp_get(struct snmp_session *ss, char *target, struct snmp_pdu **response);
char *find_net_if_oid(struct snmp_session *ss, char *name, int *net_if_oid);
char *get_net_if_octets(struct snmp_session *ss, int net_if_oid, uint64_t *octets, int in);
void if_stats_create(struct if_stats *s);
void if_stats_insert(struct if_stats *s, uint64_t in, uint64_t out);
void if_stats_dump(struct if_stats *s);
void if_stats_compute_average(struct if_stats *s, float *in_avg, float *out_avg);
void if_stats_compute_current_rate(struct if_stats *s, uint64_t in, uint64_t out, float *in_bps, float *out_bps);
int if_stats_full(struct if_stats *s);


int main(int argc, char **argv) {
    struct snmp_session session, *ss;
    uint64_t in_octets, out_octets;
    float in_avg, out_avg, in_bps, out_bps;
    float in_variation = 0, out_variation = 0;
    int version;
    int fd;
    int net_if_oid = -1;
    int nbytes;
    char host[255];
    char community[64];
    char net_if[64], net_if2[64];
    char stats_file[255];
    char *resp;
    mode_t if_stat_file_mode;
    struct if_stats net_if_stats, net_if_stats2;

    ss = NULL;

    if (5 != argc) {
        help();
        return 1;
    }

    strncpy(host, argv[1], sizeof(host));
    strncpy(community, argv[2], sizeof(community));
    version = atol(argv[3]);
    strncpy(net_if, argv[4], sizeof(net_if));
    strncpy(net_if2, argv[4], sizeof(net_if2));
    ifname_sanitize(net_if2, sizeof(net_if2));

#ifdef DEBUG
    printf("host=%s, community=%s, version=%d, if=%s\n", host, community, version, net_if);
#endif

    switch (version) {
        case 1:
            version = SNMP_VERSION_1;
            break;
        case 2:
            version = SNMP_VERSION_2c;
            break;
        default:
            printf("SNMP Version %d is not supported.", version);
            return 3;
    }


    init_snmp("check_bandwidth");
    netsnmp_init_mib();

    if (!netsnmp_read_module("IF-MIB")) {
        puts("Could not load IF-MIB module!");
        return 3;
    }

    snmp_sess_init(&session);
    session.peername = host;
    session.community = community;
    session.community_len = strlen(community);
    session.version = version;

    ss = snmp_open(&session);

    if (!ss) {
        puts("SNMP Connect error");
        return 3;
    }

    sprintf(stats_file, IF_STATS_FILE_PATH, host, net_if2);

    if_stat_file_mode = S_IWUSR | S_IRUSR | S_IRGRP;
    fd = open(stats_file, O_CREAT | O_RDWR, if_stat_file_mode);

    if (fd == -1) {
        printf("Could not open db file: %s\n", stats_file);
        return 3;
    }

    nbytes = read(fd, &net_if_stats, sizeof(net_if_stats));

    if (nbytes == -1) {
        printf("Error while reading from db: %s\n", stats_file);
        close(fd);
        return 3;
    }

    if (nbytes == 0) {
        if_stats_create(&net_if_stats);
    }

    resp = find_net_if_oid(ss, net_if, &net_if_oid);

    if (resp) {
        printf("SNMP Failure: %s\n", resp);
        close(fd);
        return 3;
    }

    if (net_if_oid == -1) {
        printf("Interface not found: %s\n", net_if);
        close(fd);
        return 3;
    }

    resp = get_net_if_octets(ss, net_if_oid, &in_octets, 1);
    if (resp) {
        printf("SNMP Failure: %s\n", resp);
        close(fd);
        return 3;
    }

    resp = get_net_if_octets(ss, net_if_oid, &out_octets, 0);
    if (resp) {
        printf("SNMP Failure: %s\n", resp);
        close(fd);
        return 3;
    }

    if_stats_compute_average(&net_if_stats, &in_avg, &out_avg);
    if_stats_compute_current_rate(&net_if_stats, in_octets, out_octets, &in_bps, &out_bps);
    //if_stats_dump(&net_if_stats);
    if (if_stats_full(&net_if_stats)) {
        in_variation = fabsf(in_avg - in_bps);
        in_variation /= in_avg;
        out_variation = fabsf(out_avg - out_bps);
        out_variation /= out_avg;
    }
    //printf("IN AVG: %.2f\n", in_avg);
    //printf("OUT AVG: %.2f\n", out_avg);
    //printf("IN bps: %.2f\n", in_bps);
    //printf("OUT bps: %.2f\n", out_bps);
    //printf("IN variation: %.2f\n", in_variation);
    //printf("OUT variation: %.2f\n", out_variation);


    if_stats_insert(&net_if_stats, in_octets, out_octets);

    memset(&net_if_stats2, 0, sizeof(net_if_stats2));

    write(fd, &net_if_stats2, sizeof(net_if_stats2));

    lseek(fd, 0, SEEK_SET);

    nbytes = write(fd, &net_if_stats, sizeof(net_if_stats));

    if (nbytes != sizeof(net_if_stats)) {
        printf("Internal Error: Could not update database file.");
        close(fd);
        return 3;
    }

    close(fd);

    in_bps /= 1024 * 1024;
    out_bps /= 1024 * 1024;
    in_avg /= 1024 * 1024;
    out_avg /= 1024 * 1024;

    if (in_variation >= CRIT_VARIATION) {
        printf(STATUS_STRING, "CRITICAL", out_bps, out_avg, out_variation * 100,
                in_bps, in_avg, in_variation * 100, net_if);
        printf(PERFORMANCE_STRING, in_bps, out_bps, in_avg, out_avg,
                in_variation * 100, out_variation * 100);

        close(fd);
        return 2;
    } else if (in_variation >= WARN_VARIATION) {
        printf(STATUS_STRING, "WARNING", out_bps, out_avg, out_variation * 100,
                in_bps, in_avg, in_variation * 100, net_if);
        printf(PERFORMANCE_STRING, in_bps, out_bps, in_avg, out_avg,
                in_variation * 100, out_variation * 100);

        close(fd);
        return 1;
    }

    printf(STATUS_STRING, "OK", out_bps, out_avg, out_variation * 100,
            in_bps, in_avg, in_variation * 100, net_if);
    printf(PERFORMANCE_STRING, in_bps, out_bps, in_avg, out_avg,
            in_variation * 100, out_variation * 100);

    close(fd);
    return 0;
}


void help() {
    char *help = "check_bandwidth <host> <community> <snmp_version> <interface>";
    puts(help);
}

char *ifname_sanitize(char *ifname, int len) {
    int i;
    for (i = 0; ifname[i] != '\0' && i < len; i++) {
        switch (ifname[i]) {
            case ' ':
                ifname[i] = '_';
                break;
            case '/':
                ifname[i] = '_';
                break;
            case 0x5C:
                ifname[i] = '_';
                break;
            case '%': // backslash
                ifname[i] = '_';
                break;
            case '*':
                ifname[i] = '_';
                break;
            case ':':
                ifname[i] = '_';
                break;
            case '?':
                ifname[i] = '_';
                break;
            case '<':
                ifname[i] = '_';
                break;
            case '>':
                ifname[i] = '_';
                break;
            case '!':
                ifname[i] = '_';
                break;
            case '"': // double quote
                ifname[i] = '_';
                break;
            case 0x27: // single quote
                ifname[i] = '_';
                break;

        }
    }

    return ifname;
}

char *my_snmp_get(struct snmp_session *ss, char *target, struct snmp_pdu **response) {
    struct snmp_pdu *pdu;
    oid myOID[MAX_OID_LEN];
    size_t myOID_len = MAX_OID_LEN; 
    char *msg;
    int status;

    msg = (char*)malloc(sizeof(char));

    pdu = snmp_pdu_create(SNMP_MSG_GET);

    if (!read_objid(target, myOID, &myOID_len)) {
        sprintf(msg, "SNMP Failure: could not parse OID string");
        return msg;
    }     

    snmp_add_null_var(pdu, myOID, myOID_len);

    status = snmp_synch_response(ss, pdu, response); 

    if (status != STAT_SUCCESS) {
        sprintf(msg, "SNMP Failure: status = %d\n", status);
        return msg;
    }


    if ((*response)->errstat != SNMP_ERR_NOERROR) {
        sprintf(msg, "SNMP Error: %s\n", snmp_errstring((*response)->errstat));
        return msg;
    }

    return NULL;
}


char *find_net_if_oid(struct snmp_session *ss, char *name, int *net_if_oid) {
    int i;

    *net_if_oid = -1;
    for (i = 1; i <= 255 && *net_if_oid == -1; i++) {
        char target[255];
        char *resp;
        struct snmp_pdu *response;
        struct variable_list *vars;
        sprintf(target, "IF-MIB::ifDescr.%d", i);
        resp = NULL;
        resp = my_snmp_get(ss, target, &response); 
        if (resp) {
            return resp;
        }
        for (vars = response->variables; vars && vars->type == ASN_OCTET_STR; vars = vars->next_variable) {
            char *sp = (char*)malloc(sizeof(vars->val_len) + 1);
            memcpy(sp, vars->val.string, vars->val_len);
            sp[vars->val_len] = '\0';
            if (!strcmp(sp, name)) {
                *net_if_oid = i;
            }
            free(sp);
        }
        free(response);
    }
    return NULL;
}

char *get_net_if_octets(struct snmp_session *ss, int net_if_oid, uint64_t *octets, int in) {
    struct snmp_pdu *response;
    struct variable_list *vars;
    char target[255];
    char *resp;

    if (in) {
        sprintf(target, "IF-MIB::ifHCInOctets.%d", net_if_oid);
    } else {
        sprintf(target, "IF-MIB::ifHCOutOctets.%d", net_if_oid);
    }

    resp = my_snmp_get(ss, target, &response);

    if (resp) {
        return resp; 
    }

    vars = response->variables;
    *octets = vars->val.counter64->high;
    *octets <<= 32;
    *octets += vars->val.counter64->low;

    free(response);

    return NULL;
}

void if_stats_create(struct if_stats *s) {
    int i;

    for (i = 0; i < IF_STATS_SAMPLES; i++) {
        s->in_octets[i] = -1;
        s->out_octets[i] = -1;
        s->last_update[i] = 0;
    }
    s->index = 0;
    time(&s->last_update[s->index]);
}

void if_stats_insert(struct if_stats *s, uint64_t in, uint64_t out) {
    s->in_octets[s->index] = in;
    s->out_octets[s->index] = out;
    time(&s->last_update[s->index]);
    s->index = CLOCK_NEXT(s->index, IF_STATS_SAMPLES);
}

void if_stats_dump(struct if_stats *s) {
    int i;
    int j;

    printf("index: %d\n", s->index);
    printf("last_update: %ld\n", s->last_update[CLOCK_PREV(s->index, IF_STATS_SAMPLES)]);
    printf("in: ");
    for (i = s->index, j = 0; i != s->index || j == 0; i = CLOCK_NEXT(i, IF_STATS_SAMPLES), j = 1) {
        printf("[%d]%lu ", i, s->in_octets[i]);
    }
    printf("\n");

    printf("out: ");
    for (i = s->index, j = 0; i != s->index || j == 0; i = CLOCK_NEXT(i, IF_STATS_SAMPLES), j = 1) {
        printf("[%d]%lu ", i, s->out_octets[i]);
    }
    printf("\n");

}

void if_stats_compute_average(struct if_stats *s, float *in_avg, float *out_avg) {
    int i;
    int j;
    int samples = 0;
    //time_t now;
    *in_avg = 0;
    *out_avg = 0;

    i = s->index;
    j = CLOCK_NEXT(i, IF_STATS_SAMPLES);


    if (s->in_octets[i] == -1 || s->in_octets[j] == -1) {
        return;
    }

    //time(&now);

    while (j != s->index) {
        if (s->in_octets[j] == -1) {
            break;
        }
        *in_avg += COMPUTE_RATE(s->in_octets[i], s->in_octets[j], s->last_update[i], s->last_update[j]) * 8;
        *out_avg += COMPUTE_RATE(s->out_octets[i], s->out_octets[j], s->last_update[i], s->last_update[j]) * 8;
        samples++;
        i = CLOCK_NEXT(i, IF_STATS_SAMPLES);
        j = CLOCK_NEXT(j, IF_STATS_SAMPLES);
    }
    *in_avg /= samples;
    *out_avg /= samples;
}

void if_stats_compute_current_rate(struct if_stats *s, uint64_t in, uint64_t out, float *in_bps, float *out_bps) {
    int j;
    time_t now; 
    *in_bps = 0;
    *out_bps = 0;

    j = CLOCK_PREV(s->index, IF_STATS_SAMPLES); 

    if (s->in_octets[j] == -1) {
        return;
    }

    time(&now);

    *in_bps = COMPUTE_RATE(s->in_octets[j], in, s->last_update[j], now) * 8;
    *out_bps = COMPUTE_RATE(s->out_octets[j], out, s->last_update[j], now) * 8;
}


int if_stats_full(struct if_stats *s) {
    int i;

    for (i = 0; i < IF_STATS_SAMPLES; i++) {
        if (s->in_octets[i] == -1) {
            return 0;
        }
    }

    return 1;
}
