#!/usr/bin/env bash

# The MIT License (MIT)
#
# Copyright (C) 2016 Cloudbase Solutions SRL
#
# Author: Ionut Balutoiu <ibalutoiu@cloudbasesolutions.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

function help() {
    echo "Required parameters:"
    echo "    --notification-type: Occurred notification type"
    echo "    --service-description: Monitored service description"
    echo "    --service-state: Monitored service state"
    echo "    --service-output: Monitored service output"
    echo "    --host-alias: Monitored host alias"
    echo "    --host-address: Monitored host address"
    echo "    --long-date-time: Time and date when alert occurred"
    echo "    --contact-email: Contact email"
    echo ""
    echo "Optional parameters:"
    echo "    --notification-comment: Notification comment"
    echo "    --plugin-path: Full path on the disk for the plug-in executable"
    echo "    --config-file: AzureStack config file path"
    echo "    --action: Plug-in action. If this is different than 'close', the handler won't do anything if the notification comment is a close command"
    echo ""
}

while [ $# -gt 0 ]
do
    case $1 in
        --notification-type)
            NOTIFICATION_TYPE=$2
            shift;;
        --notification-comment)
            NOTIFICATION_COMMENT=$2
            shift;;
        --service-description)
            SERVICE_DESCRIPTION=$2
            shift;;
        --service-state)
            SERVICE_STATE=$2
            shift;;
        --service-output)
            SERVICE_OUTPUT=$2
            shift;;
        --host-alias)
            HOST_ALIAS=$2
            shift;;
        --host-address)
            HOST_ADDRESS=$2
            shift;;
        --long-date-time)
            LONG_DATE_TIME=$2
            shift;;
        --contact-email)
            CONTACT_EMAIL=$2
            shift;;
        --plugin-path)
            PLUGIN_PATH=$2
            shift;;
        --config-file)
            CONFIG_FILE=$2
            shift;;
        --help)
            help
            exit 0;;
        *)
            echo "No such option"
            help
            exit 1
    esac
    shift
done

if [ -z "$NOTIFICATION_TYPE" ]; then echo "'--notification-type' parameter must be provided"; exit 1; fi
if [ -z "$SERVICE_DESCRIPTION" ]; then echo "'--service-description' parameter must be provided"; exit 1; fi
if [ -z "$SERVICE_STATE" ]; then echo "'--service-state' parameter must be provided"; exit 1; fi
if [ -z "$SERVICE_OUTPUT" ]; then echo "'--service-output' parameter must be provided"; exit 1; fi
if [ -z "$HOST_ALIAS" ]; then echo "'--host-alias' parameter must be provided"; exit 1; fi
if [ -z "$HOST_ADDRESS" ]; then echo "'--host-address' parameter must be provided"; exit 1; fi
if [ -z "$LONG_DATE_TIME" ]; then echo "'--long-date-time' parameter must be provided"; exit 1; fi
if [ -z "$CONTACT_EMAIL" ]; then echo "'--contact-email' parameter must be provided"; exit 1; fi

COMMAND=$(echo $NOTIFICATION_COMMENT | awk '{print $1}')
if [[ "$COMMAND" = "/close-alert" ]]; then
    if [ -z "$PLUGIN_PATH" ]; then echo "'--plugin-path' parameter must be provided"; exit 1; fi
    if [ -z "$CONFIG_FILE" ]; then echo "'--config-file' parameter must be provided"; exit 1; fi

    UUID=$(echo $NOTIFICATION_COMMENT | awk '{print $2}')

    $PLUGIN_PATH --config-file $CONFIG_FILE \
                 --action Close \
                 --alert-id $UUID

    exit 0
fi


MAIL_SUBJECT="** $NOTIFICATION_TYPE Service Alert: $HOST_ALIAS/$SERVICE_DESCRIPTION is $SERVICE_STATE **"

read -r -d '' MAIL_BODY << END
"***** Nagios Monitor XI Alert *****

Notification Type: $NOTIFICATION_TYPE

Service: $SERVICE_DESCRIPTION
Host: $HOST_ALIAS
Address: $HOST_ADDRESS
State: $SERVICE_STATE

Date/Time: $LONG_DATE_TIME

Additional Info:

$SERVICE_OUTPUT

END

/usr/bin/printf "%b" "$MAIL_BODY" | /usr/bin/mail -s "$MAIL_SUBJECT" "$CONTACT_EMAIL"
