#!/usr/bin/env python

# The MIT License (MIT)
#
# Copyright (C) 2016 Cloudbase Solutions SRL
#
# Author: Ionut Balutoiu <ibalutoiu@cloudbasesolutions.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import argparse
import json
import requests
import sys
import uuid

import adal
from six.moves.urllib import parse
import yaml

# Exit codes as they are interpreted by Nagios
EXIT_CODES = {
    'OK': 0,
    'WARNING': 1,
    'CRITICAL': 2,
    'UNKNOWN': 3
}

reload(sys)
sys.setdefaultencoding('utf8')


def get_client_cert(client_cert_path):
    with open(client_cert_path, 'rb') as f:
        return f.read()


def get_authentication_token(client_id, client_secret, client_cert_path,
                             client_cert_thumbprint, tenant,
                             auth_uri, resource, verify=True,
                             validate_authority=True):
    context = adal.AuthenticationContext(
        auth_uri, validate_authority=validate_authority,
        api_version=None, verify_ssl=verify)

    if client_secret:
        token = context.acquire_token_with_client_credentials(
            resource, client_id, client_secret)
    else:
        client_cert = get_client_cert(client_cert_path)

        token = context.acquire_token_with_client_certificate(
            resource, client_id, client_cert, client_cert_thumbprint)

    return token


def rest_api_call(url, method, headers={}, params={}, data="", verify=True):
    if method not in ['get', 'post', 'put', 'patch', 'delete']:
        print("MAS CRITICAL: Invalid REST method: '%s'" % method)
        exit(EXIT_CODES['CRITICAL'])
    _method = getattr(requests, method)
    r = _method(url, headers=headers, params=params, data=data, verify=verify)
    if not r.ok:
        print("MAS CRITICAL: Response from url '%s' was not ok." % url)
        print("Response text: '%s'" % r.text)
        exit(EXIT_CODES['CRITICAL'])
    return r


def format_message(msg_items):
    msg = ""
    for item in msg_items:
        item_type = item.get('type')
        if item_type == 'Text':
            msg += item.get('text', "")
        elif item_type == 'NewLine':
            msg += ' '
    return msg


def print_alerts(alerts, alerts_type):
    if len(alerts) > 0:
        print("------------")
        print("%s ALERTS:" % alerts_type.upper())
        for alert in alerts:
            print("------------")
            print("Alert id: %s" % alert['properties']['alertId'])
            print("Title: %s" % alert['properties']['title'])
            print("Impacted resource: %s" %
                  alert['properties']['impactedResourceDisplayName'])
            print("Description: %s" %
                  format_message(alert['properties']['description']))
            print("Remediation: %s" %
                  format_message(alert['properties']['remediation']))


# Parse the parameters
parser = argparse.ArgumentParser(
    description='Nagios plugin for monitoring AzureStack alerts')

parser.add_argument('--config-file',
                    type=argparse.FileType('rb'),
                    help='Azure Stack config file path',
                    required=True)
parser.add_argument('--action',
                    type=str,
                    help=("Action the plug-in will execute. 'Monitor' is "
                          "used for monitoring active alerts. 'Close' is "
                          "used to manually close a particular alert (close "
                          "action also excepts the '--alert-id' parameter "
                          "to be set."),
                    required=True)
parser.add_argument('--alert-id',
                    type=str,
                    help=("Id of the alert which will be closed. This is used "
                          "only when '--action' parameter is set on "
                          "'Close', otherwise it will be ignored."))

try:
    args = parser.parse_args()
    config = yaml.load(args.config_file)

    mandatory_params = [
        'external_domain_fqdn',
        'region',
        'client_id',
    ]

    missing_params = []
    for p in mandatory_params:
        if not config.get(p):
            missing_params.append(p)

    if len(missing_params) > 0:
        missing = ', '.join(missing_params)
        print("MAAS UNKNOWN: Missing plug-in parameters: %s" % missing)
        exit(EXIT_CODES['UNKNOWN'])

    if not (bool(config.get("client_secret")) ^
            (bool(config.get("client_cert_path")) and
             bool(config.get("client_cert_thumbprint")))):
        print("MAAS UNKNOWN: Either client_secret or "
              "client_cert_path and client_cert_thumbprint must be provided")
        exit(EXIT_CODES['UNKNOWN'])

    allowed_actions = ['Monitor', 'Close']
    if args.action not in allowed_actions:
        allowed = ', '.join(allowed_actions)
        print("MAAS UNKNOWN: Action '%s' is not valid. "
              "The allowed actions are: %s" % (args.action, allowed))
        exit(EXIT_CODES['UNKNOWN'])

    if args.action == 'Close':
        if not args.alert_id:
            print("MAAS UNKNOWN: Action 'Close' requires "
                  "parameter '--alert-id'")
            exit(EXIT_CODES['UNKNOWN'])
        else:
            try:
                alert_uuid = uuid.UUID(args.alert_id, version=4)
            except ValueError:
                # If it's a value error, then the string is not a valid
                # hex code for a UUID.
                print("MAAS UNKNOWN: Alert id must be a valid UUID")
                exit(EXIT_CODES['UNKNOWN'])

    api_version = "2016-05-01"
    verify_https = not config.get("insecure", False)

    external_domain_fqdn = config.get("external_domain_fqdn")
    region = config.get("region")

    admin_api_url = "https://adminmanagement.%s.%s" % (
        region, external_domain_fqdn)

    endpoints_url = parse.urljoin(
        admin_api_url, 'metadata/endpoints?api-version=1.0')
    response = rest_api_call(endpoints_url, 'get', verify=verify_https)
    endpoints = json.loads(response.text)

    login_endpoint = endpoints['authentication']['loginEndpoint']
    is_adfs = parse.urlparse(login_endpoint).path in ['/adfs', '/adfs/']

    if is_adfs:
        auth_uri = login_endpoint
    else:
        if not config.get("tenant_id"):
            print("MAAS UNKNOWN: The \"tenant_id\" parameter is required")
            exit(EXIT_CODES['UNKNOWN'])
        auth_uri = parse.urljoin(login_endpoint, config.get("tenant_id"))

    resource = endpoints['authentication']['audiences'][0]
    token = get_authentication_token(client_id=config.get("client_id"),
                                     client_secret=config.get("client_secret"),
                                     client_cert_path=config.get(
                                         "client_cert_path"),
                                     client_cert_thumbprint=config.get(
                                         "client_cert_thumbprint"),
                                     tenant=config.get("tenant_id"),
                                     auth_uri=auth_uri,
                                     resource=resource,
                                     verify=verify_https,
                                     validate_authority=not is_adfs)

    headers = {
        'Content-Type': 'application/json',
        'Authorization': '%s %s' % (token['tokenType'],
                                    token['accessToken'])
    }

    subscriptions_url = "%s/subscriptions?api-version=%s" % (
        admin_api_url, api_version)
    r = rest_api_call(url=subscriptions_url,
                      method='get',
                      headers=headers,
                      verify=verify_https)
    subs = json.loads(r.text)
    enabled_subs = [s for s in subs['value'] if s['state'] == 'Enabled']

    if not enabled_subs:
        print("MAAS UNKNOWN: No enabled subscription found")
        exit(EXIT_CODES['UNKNOWN'])

    subscription_id = enabled_subs[0]['subscriptionId']

    base_url = ("%(admin_api)s/subscriptions/%(subscription_id)s/"
                "resourceGroups/System.%(region)s/providers/"
                "Microsoft.InfrastructureInsights.Admin/"
                "regionHealths/%(region)s" % {
                    'admin_api': admin_api_url,
                    'subscription_id': subscription_id,
                    'region': config.get("region")})

    alerts_url = ("%s/Alerts?api-version=%s" % (base_url, api_version))

    r = rest_api_call(url=alerts_url,
                      method='get',
                      headers=headers,
                      verify=verify_https)
    alerts = json.loads(r.text)

    if args.action == 'Close':
        alrt = None
        for a in alerts['value']:
            if a['properties']['alertId'] == args.alert_id:
                alrt = a
                break
        if not alrt:
            print("MAAS UNKNOWN: Alert with id '%s' was not found" %
                  args.alert_id)
            exit(EXIT_CODES['UNKNOWN'])

        alrt['properties']['state'] = "Closed"

        url = ("%s/Alerts/%s?api-version=%s" % (base_url,
                                                args.alert_id,
                                                api_version))
        r = rest_api_call(url=url,
                          method='put',
                          headers=headers,
                          data=json.dumps(alrt),
                          verify=verify_https)
        print("MAAS OK: Alert with id '%s' was successfully closed." %
              args.alert_id)
        exit(EXIT_CODES['OK'])

    # Find critical and warning alerts
    critical_alerts = [
        alert for alert in alerts['value']
        if (alert['properties']['severity'] == 'Critical' and
            alert['properties']['state'] == 'Active')]
    warning_alerts = [
        alert for alert in alerts['value']
        if (alert['properties']['severity'] == 'Warning' and
            alert['properties']['state'] == 'Active')]

    warning_alerts_count = len(warning_alerts)
    critical_alerts_count = len(critical_alerts)

    if critical_alerts_count == 0 and warning_alerts_count == 0:
        message = "MAS OK: No alerts reported"
        status_code = EXIT_CODES['OK']
    elif critical_alerts_count > 0 and warning_alerts_count > 0:
        message = ("MAS CRITICAL: %s critical alerts and %s warning alerts" %
                   (critical_alerts_count, warning_alerts_count))
        status_code = EXIT_CODES['CRITICAL']
    elif critical_alerts_count > 0 and warning_alerts_count == 0:
        message = "MAS CRITICAL: %s critical alerts" % critical_alerts_count
        status_code = EXIT_CODES['CRITICAL']
    elif critical_alerts_count == 0 and warning_alerts_count > 0:
        message = "MAS WARNING: %s warning alerts" % warning_alerts_count
        status_code = EXIT_CODES['WARNING']

    # Print relevant information
    print(message)
    print_alerts(critical_alerts, 'Critical')
    print_alerts(warning_alerts, 'Warning')

    # Exit with proper status code
    exit(status_code)
except adal.adal_error.AdalError as e:
    print("MAAS UNKNOWN: %s" % e)
    exit(EXIT_CODES['UNKNOWN'])
except Exception as e:
    print("MAAS UNKNOWN: %s, %s" % (type(e), e))
    exit(EXIT_CODES['UNKNOWN'])
