/* vim: set ts=2 sw=2: */
/****************************************************************************
 * This file is part of nscam.                                              *
 *                                                                          *
 * nscam is free software: you can redistribute it and/or modify            *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation, either version 3 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * nscam is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with nscam.  If not, see <http://www.gnu.org/licenses/>.           *
 *                                                                          *
 ****************************************************************************
 * filename : crypt.c                                                       *
 *                                                                          *
 * id: $Id$
 *                                                                          *
 * date:    : 2011-05-08                                                    *
 * purpose  : Mcrypt intarface and other crypt functions.                   *
 *                                                                          *
 * author   : Peter Meszaros                                                *
 *            e-mail: hauptadler@gmail.com                                  *
 *                                                                          *
 ****************************************************************************/

#include "crypt.h"

#define ALGO_LEN		32
#define MODE_LEN		32
#define KEY_LEN			256
#define PASSWD_LEN  64

struct {
  char algo[ALGO_LEN+1];
  char mode[MODE_LEN+1];
  unsigned char keylen;
  char key[KEY_LEN+1];
  char passwd[PASSWD_LEN+1];
} key;

MCRYPT td;

void printhex(FILE *f, unsigned char *b, int length)
{
	int i;

	fprintf(f, "%08x:", length);
	for(i=0; i<length; i++) {
	  fprintf(f, " %02x", b[i]);
	}
	fprintf(f, "\n");

	return;
}

void generateIVrandom(unsigned char *b, int length, int seed)
{
	int i;

	if (seed) srandom(time(NULL));

	for(i=0; i<length; i++) 
		b[i] = (unsigned char)random();

	return;
}

void generateIVdevrandom(unsigned char *b, int length)
{
	int f, fl;
	ssize_t i;

	if ((f = open("/dev/random", O_RDONLY)) < 0)
		err(EXIT_FAILURE, "open(/dev/random) failed");

   if ((fl = fcntl(f, F_GETFL, 0)) < 0)
		err(EXIT_FAILURE, "fcntl(GETFL) failed");

   if (fcntl(f, F_SETFL, fl | O_NONBLOCK) < 0)
		err(EXIT_FAILURE, "fcntl(SETFL) failed");

  if ((i = read(f, b, length)) < 0)
		err(EXIT_FAILURE, "read(/dev/random) reading %d bytes failed", length);
	else if (i < length)
		errx(EXIT_FAILURE, "read(/dev/random) only %d has read from entropy pool", i);

	close(f);

	return;
}

void loadkeyfile(char *fn)
{
	int f, i;
	struct stat st;
	unsigned char *b, *p;

	if ((f = open(fn, O_RDONLY)) < 0)
		err(EXIT_FAILURE, "open(%s) failed", fn);

  if (fstat(f, &st) < 0)
		err(EXIT_FAILURE, "stat(%s) failed", fn);

  if ((p = b = calloc(1, st.st_size+1)) == NULL)
		err(EXIT_FAILURE, "calloc(%lu) failed", st.st_size);

	if ((i = read(f, b, st.st_size)) < 0)
		err(EXIT_FAILURE, "read(%s) reading %lu bytes failed", fn, st.st_size);
	else if (i < st.st_size)
		errx(EXIT_FAILURE, "read(%s) only %d has read instead of %lu", fn, i, st.st_size);

	b[st.st_size] = '\0';

  /* algo */
  if ((p = strtok(b, "|")))
    strncpy(key.algo, p, ALGO_LEN);
  else
    errx(EXIT_FAILURE, "Invalid algo in keyfile '%s'", fn);

  /* mode */
  if ((p = strtok(NULL, "|")))
    strncpy(key.mode, p, MODE_LEN);
  else
    errx(EXIT_FAILURE, "Invalid mode in keyfile '%s'", fn);

  p += strlen(p) + 1; /* skip mode field */

  /* IV */
  key.keylen = p[0];
  memcpy(key.key, p+1, key.keylen);

  /* passwd, the key may contain '|'... */
  strncpy(key.passwd, p+1+key.keylen, PASSWD_LEN);

  free(b);
	close(f);
}

void printkeyfile(FILE *out)
{

  fprintf(out, "Algo: '%s'\n", key.algo);
  fprintf(out, "Mode: '%s'\n", key.mode);
  fprintf(out, "Key: ");
  printhex(out, key.key, key.keylen);
	fprintf(out, "Pswd: '%s'\n", key.passwd);
}

int McryptPrintInfo(int print)
{
  int i, lalg, lmod, iv, maxiv = 0;
  char **alg;
  char **mod;
	MCRYPT td;

  if (print) printf("=== Mcrypt info ===\n");

  alg = mcrypt_list_algorithms(NULL, &lalg);
  mod = mcrypt_list_modes(NULL, &lmod);

	for(i=0; i<lalg; i++) {
		if (print) printf("%3d '%24s' -- ", i, alg[i]);
		if ((td = mcrypt_module_open(alg[i], NULL, mod[0], NULL)) == MCRYPT_FAILED) {
			if (print) printf("mcrypt_module_open() - failed\n");
			continue;
		}
		iv = mcrypt_enc_get_iv_size(td);
		if (print) printf("IV size: %3d\n", iv);
		if (iv > maxiv) maxiv = iv;

		mcrypt_module_close(td);
	}
  if (print) printf("%d algorithms supported\n\n", lalg);

	if (print) {
		for(i=0; i<lmod; i++)
			printf("%3d '%s'\n", i, mod[i]);
		printf("%d modes supported\n", lmod);

		printf("\nMax. IV: %d bytes\n", maxiv);
	}

  mcrypt_free_p(alg, lalg);
  mcrypt_free_p(mod, lmod);

  return maxiv;
}

int McryptInit(char *fn)
{
  int i;

  loadkeyfile(fn);

  td = mcrypt_module_open(key.algo, NULL, key.mode, NULL);
  if (td == MCRYPT_FAILED) return 1;

  if ((i = mcrypt_generic_init(td, key.passwd, key.keylen, key.key)) < 0) {
     mcrypt_perror(i);
     return 1;
  }

  return 0;
}

int McryptDecrypt(char *buf, int len)
{
  int i;

  if ((i = mdecrypt_generic(td, buf, len))) {
    mcrypt_perror(i);
    return 1;
	}
  return 0;
}

int McryptEncrypt(char *buf, int len)
{
  int i;

  if ((i = mcrypt_generic(td, buf, len))) {
    mcrypt_perror(i);
    return 1;
	}
  return 0;
}

void McryptDone(void)
{
  mcrypt_generic_deinit(td);
  mcrypt_module_close(td);
}

