/* vim: set ts=2 sw=2: */
/****************************************************************************
 * This file is part of nscam.                                              *
 *                                                                          *
 * nscam is free software: you can redistribute it and/or modify            *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation, either version 3 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * nscam is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with nscam.  If not, see <http://www.gnu.org/licenses/>.           *
 *                                                                          *
 ****************************************************************************
 * filename : keygen.c                                                      *
 *                                                                          *
 * id: $Id$
 *                                                                          *
 * date:    : 2011-05-08                                                    *
 * purpose  : Generate IV and password file for enrypted communication and  *
 *            prints out mcrypt's possibilities.                            *
 *                                                                          *
 * author   : Peter Meszaros                                                *
 *            e-mail: hauptadler@gmail.com                                  *
 *                                                                          *
 ****************************************************************************/

#include "crypt.h"

char *version = PACKAGE_STRING ", written by " PACKAGE_BUGREPORT;

void genkeyfile(char *fn, enum genmode mode, unsigned char ivlen, int seed, char *pw, char *algo, char *cmode)
{
	int f, i, l = strlen(pw);
	unsigned char *b;

	if ((f = open(fn, O_CREAT|O_WRONLY, 0400)) < 0)
		err(EXIT_FAILURE, "open(%s) failed", fn);

  if (dprintf(f, "%s|%s|", algo, cmode) < 0)
		err(EXIT_FAILURE, "dprintf(%s, %s) failed", algo, cmode);

  if ((b = calloc(1, 1+ivlen+l)) == NULL)
		err(EXIT_FAILURE, "calloc(%d) failed", ivlen);

	b[0] = ivlen;

  if (mode == DEVRANDOM)
    generateIVrandom(b+1, ivlen, seed);
	else
    generateIVdevrandom(b+1, ivlen);
	
	memcpy(b+1+ivlen, pw, l);

	/* write IV length */
	if ((i = write(f, b, 1+ivlen+l)) < 0)
		err(EXIT_FAILURE, "write(%s) writing %d bytes failed", fn, 1+ivlen+l);
	else if (i < 1+ivlen+l)
		errx(EXIT_FAILURE, "write(%s) only %d has written instead of %d", fn, i, 1+ivlen+l);

  free(b);
	close(f);

	fprintf(stderr, "%s keyfile successfully generated.\n", fn);
}

void Usage(int argc, char *argv[])
{
	fprintf(stderr, "Usage: %s [VhdPs][-f filename][-p password] # %s\n", basename(argv[0]), version);
	fprintf(stderr, "  -V           prints version number\n");
	fprintf(stderr, "  -h           prints this message\n");
	fprintf(stderr, "  -d           use /dev/random at generating IV\n");
	fprintf(stderr, "  -f filename  key file name\n");
	fprintf(stderr, "  -p password  password string\n");
	fprintf(stderr, "  -s           use timestamp as seed\n");
	fprintf(stderr, "  -P           prints keyfile content\n");
	fprintf(stderr, "  -a algorithm used encryption algorithm (default: 'twofish')\n");
	fprintf(stderr, "  -m mode      used encryption mode (default: 'cfb')\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Examples:\n");
	fprintf(stderr, "  Mcrypt info (list algorithms & modes):\n");
	fprintf(stderr, "   ./keygen\n");
	fprintf(stderr, "  Generate keyfile:\n");
	fprintf(stderr, "   ./keygen -f nscam.key -p 12345 -a twofish -m cfb -s -d\n");
	fprintf(stderr, "  Print keyfile:\n");
	fprintf(stderr, "   ./keygen -f nscam.key -P\n");
}

int main(int argc, char *argv[])
{
	int opt;
  int maxiv = 0;
	unsigned char *b;
	char *fname = NULL, *passwd = "", *mode = "cfb", *algo = "twofish";
  int seed = 0, print = 0;
  enum genmode devrand = RANDOM;

	while ((opt = getopt(argc, argv, "Vhsdf:p:Pa:m:")) != -1) {
		switch (opt) {
			case 's':
			  seed = 1;
				break;
			case 'd':
			  devrand = DEVRANDOM;
				break;
			case 'f':
			  fname = optarg;
				break;
			case 'p':
			  passwd = optarg;
				break;
			case 'P':
			  print = 1;
				break;
			case 'a':
			  algo = optarg;
				break;
			case 'm':
			  mode = optarg;
				break;
			case 'V':
			case 'h':
				Usage(argc, argv);
				exit(EXIT_SUCCESS);
				break;
			default: /* '?' */
				Usage(argc, argv);
				exit(EXIT_FAILURE);
		}
	}

	if (print) {
		loadkeyfile(fname);
		printkeyfile(stdout);
	} else {
		maxiv = McryptPrintInfo((argc == 1));
		if (fname == NULL) exit(EXIT_SUCCESS);

		if ((b = calloc(1, maxiv)) == NULL)
			err(EXIT_FAILURE, "calloc(%d) failed", maxiv);

		genkeyfile(fname, devrand, maxiv, seed, passwd, algo, mode);

		free(b);
	}

  return EXIT_SUCCESS;
}
