/* vim: set ts=2 sw=2: */
/****************************************************************************
 * This file is part of nscam.                                              *
 *                                                                          *
 * nscam is free software: you can redistribute it and/or modify            *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation, either version 3 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * nscam is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with nscam.  If not, see <http://www.gnu.org/licenses/>.           *
 *                                                                          *
 ****************************************************************************
 * filename : nscamd.c                                                      *
 *                                                                          *
 * id: $Id$
 *                                                                          *
 * date:    : 2011-02-11                                                    *
 * purpose  : Multicast server for receiving nsca messages.                 *
 *                                                                          *
 * author   : Peter Meszaros <hauptadler@gmail.com>                         *
 *                                                                          *
 ****************************************************************************/

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <net/if.h>
#include <libgen.h>

#include "../config.h"

#define MAXLINE 4096

int verbose;
char *version = PACKAGE_STRING ", written by " PACKAGE_BUGREPORT;
char *progname;
char *outfile = "/usr/local/nagios/var/cmd/rw";
int daemonize;
char *network;
char *keyfile;
int ttl = 64;

struct in_addr ipaddr;
uint32_t ipmask;

void mylog(int priority, const char *fmt, ...)
{
	va_list		ap;

	va_start(ap, fmt);
	vsyslog(priority, fmt, ap);
	va_end(ap);

	if (priority == LOG_ERR) exit(EXIT_FAILURE);
}

void Usage(int argc, char *argv[])
{
	fprintf(stderr, "Usage: %s [-dhvV][-t #][-o outfile][-n1.2.3.0/24][-k keyfile] # version: %s\n", progname, version);
	fprintf(stderr, "  -h            Print out this message\n");
	fprintf(stderr, "  -v            Be verbose\n");
	fprintf(stderr, "  -V            Print out software version\n");
	fprintf(stderr, "  -d            Daemonize\n");
	fprintf(stderr, "  -t            Multicast TTL, default 64\n");
	fprintf(stderr, "  -o outfile    Write output to an alternate file (or pipe)\n");
	fprintf(stderr, "  -n 1.2.3.0/24 Accept packeges only from this network\n");
	fprintf(stderr, "  -l 224.0.3.1  Listen on this multicast address\n");
	fprintf(stderr, "  -k keyfille   Cryption keyfile\n");
}

int cidrparse(char *cidr)
{
	char *mask;
	unsigned int n;

	if ((mask = strchr(cidr, '/')))
		*mask++ = '\0';
	else
	  return EXIT_FAILURE;

  n = (unsigned int)atoi(mask);
	if (!n || (n > 32))
	  return EXIT_FAILURE;
   
	ipmask = htonl(0xffffffff << (32 - n));

	if ((ipaddr.s_addr = inet_addr(cidr)) == INADDR_NONE)
		return EXIT_FAILURE;

	ipaddr.s_addr &= ipmask;

	return EXIT_SUCCESS;
}

int main(int argc, char **argv)
{
  int opt;
	int fd;
	int	n, on;
	socklen_t	salen;
	struct sockaddr_in	sa, safrom;
	char line[MAXLINE+1];
  struct servent *se;
	struct ip_mreq mreq;
  char *addr = "224.0.2.1";
  short port = 5669;
  char *service = "nscamd";
	int out;

  openlog(progname, LOG_PID|(daemonize ? 0 : LOG_PERROR), LOG_DAEMON);

  progname = basename(argv[0]);

  while ((opt = getopt(argc, argv, "t:l:n:do:hVvk:")) != -1) {
 		switch (opt) {
      case 'h':
      case 'V':
				Usage(argc, argv);
				exit(EXIT_SUCCESS);
        break;
      case 'v':
        verbose = 1;
        break;
      case 't':
        ttl = atoi(optarg);
        break;
      case 'o':
        outfile = optarg;
        break;
      case 'd':
        daemonize = 1;
        break;
      case 'l':
        addr = optarg;
				if (memcmp(addr, "224.", 4))
          mylog(LOG_ERR, "Invalid multicast address: '%s'", optarg);
        break;
      case 'n':
        if (cidrparse(optarg))
          mylog(LOG_ERR, "Invalid CIDR network: '%s'", optarg);
        break;
      case 'k':
        keyfile = optarg;
        break;
 		  default: /* '?' */
				Usage(argc, argv);
				exit(EXIT_FAILURE);
    }
  }

  fprintf(stderr, "This progam listen on %s multicast address,\n", addr);
  fprintf(stderr, "service: %s or port %d if service cannot be found by name.\n", service, port); 
  fprintf(stderr, "  Verbose %s.\n", verbose ? "on" : "off");
  fprintf(stderr, "  Multicast TTL: %d.\n", ttl);
  fprintf(stderr, "  Network %s mask 0x%08x\n", inet_ntoa(ipaddr), ntohl(ipmask));
#ifdef HAVE_LIBMCRYPT
  fprintf(stderr, "  Keyfile: %s \n", keyfile ? keyfile : "");
#else
  fprintf(stderr, "  Decryption with libmcrypt is not included.\n");
#endif

  if (daemonize) {
		pid_t pid;

		if ((pid = fork()) < 0) /* fork error */
			mylog(LOG_ERR, "fork(): %d - '%s'", errno, strerror(errno));
		else if (pid == 0)      /* parent exits */
			return EXIT_SUCCESS;

	}

  port = (se = getservbyname(service, "udp")) ? se->s_port : htons(port);

  /* recv */
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    mylog(LOG_ERR, "socket(): %d - '%s'", errno, strerror(errno));

  on = 1;
	if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on))) < 0)
    mylog(LOG_ERR, "setsockopt() SO_REUSEADDR: %d - '%s'", errno, strerror(errno));

  on = ttl;
	if ((setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL, &on, sizeof(on))) < 0)
    mylog(LOG_ERR, "setsockopt() IP_MULTICAST_TTL: %d - '%s'", errno, strerror(errno));

  memset(&sa, 0, sizeof(struct sockaddr_in));
  sa.sin_family = AF_INET;
  sa.sin_addr.s_addr = inet_addr(addr);
  sa.sin_port = port;

  if ((bind(fd, (struct sockaddr *)&sa, sizeof(struct sockaddr)) < 0))
    mylog(LOG_ERR, "bind(): %d - '%s'", errno, strerror(errno));

	memcpy(&mreq.imr_multiaddr, &(sa.sin_addr), sizeof(struct in_addr));
	mreq.imr_interface.s_addr = htonl(INADDR_ANY);
	if ((setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq))) < 0)
    mylog(LOG_ERR, "setsockopt() IP_ADD_MEMBERSHIP: %d - '%s'", errno, strerror(errno));

#ifdef HAVE_LIBMCRYPT
	if (keyfile)
		if (McryptInit(keyfile) == 1)
			errx(EXIT_FAILURE, "Mcrypt initialization failed.");
#endif

  if ((out = open(outfile, O_WRONLY)) == -1) 
    mylog(LOG_ERR, "open() %s: %d - '%s'", outfile, errno, strerror(errno));

  for (;;) {
		int s;

    salen = sizeof(safrom);
		if ((n = recvfrom(fd, line, MAXLINE, 0, (struct sockaddr *)&safrom, &salen)) < 0)
      mylog(LOG_WARNING, "recvfrom(): %d - '%s'", errno, strerror(errno));

#ifdef HAVE_LIBMCRYPT
		if (keyfile)
			if (McryptDecrypt(line, n))
				errx(EXIT_FAILURE, "Mcrypt decryption failure.");
#endif

		if (verbose)
			mylog(LOG_INFO, "multicast received from %s:%d: %s",
					inet_ntoa(safrom.sin_addr), htons(safrom.sin_port), line);

    if (ipaddr.s_addr != (safrom.sin_addr.s_addr & ipmask)) {
			mylog(LOG_WARNING, "Invalid address %s mask 0x%08x", inet_ntoa(ipaddr), ntohl(ipmask));
		} else if ((s = write(out, line, n)) != n) {
			if (s < 0) {
				mylog(LOG_WARNING, "write(): %d - '%s'", errno, strerror(errno));
			} else {
				mylog(LOG_WARNING, "write() %d bytes written insted of %d", s, n);
			}
		}
  }
	close(out);
	closelog();

#ifdef HAVE_LIBMCRYPT
  if (keyfile)
    McryptDone();
#endif

  return 0;
}

