/******************************************************************************\
* NRPE_NT: Nagios Remote Plugin Executor for NT/W2K 
* Copyright (c) 2003 Michael Wirtgen (michael.wirtgen@miwi-dv.com)
*
* Updated:  Patrick Spizzo (ishamael@moridin.com) - 20.Jul.2004
*   Added include_dir functionality back in (using ported version of original code from nrpe).
*   Added better error codes during service startup.
*   Made the default config_file location work regardless of .exe filename.
*   Uses standard OpenSSL function to display OpenSSL version number in Usage.
* 
* based on:
* NRPE.C - Nagios Remote Plugin Executor
* Copyright (c) 1999-2003 Ethan Galstad (nagios@nagios.org)
*
* License: GPL
* 
* Nagios and the Nagios logo are registered trademarks of Ethan Galstad.
*
\******************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <process.h>
#include "../common/common.h"
#include "utils.h"

#ifdef HAVE_SSL
#include "dh.h"
#include "mt.h"
#endif

#define NASTY_METACHARS         "|`&><'\"\\[]{}"        /* This may need to be modified for windows directory seperator */
#define NASTY_METACHARS_WIN         "|`&><'[]{}"        /* Modified for windows directory seperator and " */
#define MAX_COMMANDNAME_LENGTH	32      /* maximum short name of a command */
#define MAX_COMMANDLINE_LENGTH	1024    /* maximum command line length */
#define DEFAULT_COMMAND_TIMEOUT	60      /* default timeout for execution of plugins */

typedef struct command_struct
{
    char command_name[MAX_COMMANDNAME_LENGTH];
    char command_line[MAX_COMMANDLINE_LENGTH];
    struct command_struct *next;
} command;

/* Globals */

unsigned short server_port = DEFAULT_SERVER_PORT;
char server_address[MAX_INPUT_BUFFER] = "\0";
HANDLE g_hWorkerThread;
HANDLE hStopEvent;
char config_file[MAX_INPUT_BUFFER];
char exe_path[MAX_INPUT_BUFFER];
char allowed_hosts[MAX_INPUT_BUFFER];
char *nrpe_user = NULL;
char *nrpe_group = NULL;
int allow_arguments = FALSE;
int command_timeout = DEFAULT_COMMAND_TIMEOUT;
int socket_timeout = DEFAULT_SOCKET_TIMEOUT;
int Install = 0;
int Uninstall = 0;
SERVICE_STATUS_HANDLE srvstatus;
command *command_list = NULL;
int delay_shutdown = FALSE;
int use_win_metachars = FALSE;

#ifdef HAVE_SSL
int use_ssl = TRUE;
SSL_CTX *ctx;
#else
int use_ssl = FALSE;
#endif


/* Functions */
void Usage(char *progname);
int process_arguments(int argc, char **argv);
void wait_for_connections(int argc, char **argv);
void handle_connection(void *param);
int validate_request(packet * pkt, char **command_name,
                     char *macro_argv[MAX_COMMAND_ARGUMENTS]);
int contains_nasty_metachars(char *str);
int read_config_dir(char *dirname);
int add_command(char *command_name, char *command_line);
command *find_command(char *command_name);
void free_memory(void);
int read_config_file(char *filename);
int my_system(char *command, int timeout, int *early_timeout, char *output,
              int output_length);
int process_macros(char *input_buffer, char *output_buffer, int buffer_length,
                   char *macro_argv[MAX_COMMAND_ARGUMENTS]);
int is_an_allowed_host(char *connecting_host);
void SetSrvStatus(DWORD dwCurrentState, DWORD dwWin32ExitCode,
                  DWORD dwCheckPoint, DWORD dwWaitHint);
void ControlHandler(DWORD request);
int InitAll();
int CloseAll();

#ifdef HAVE_SSL
void LogSSLErrors();
int my_ssl_accept(SSL * ssl);
int my_ssl_read(SSL * ssl, void *receive_packet, int bytes_to_recv);
#endif


int main(int argc, char **argv)
{

    int result;
    SERVICE_TABLE_ENTRY ServiceTable[2];

    result = process_arguments(argc, argv);

    if (result != OK) {
        Usage("NRPE_NT.exe");
        return NOT_OK;
    }

    // Install service and exit
    if (Install) {
        result = InstallService(exe_path, use_ssl);
        if (!result)
            printf("NRPE_NT Service sucessfully installed!\n");
        else
            printf("NRPE_NT Service not installed!\n");

        return result;
    }


    // Uninstall service and exit
    if (Uninstall) {
        result = UninstallService();
        if (!result)
            printf("NRPE_NT Service sucessfully removed!\n");
        else
            printf("NRPE_NT Service could not be removed!\n");

        return result;
    }

    // Do some init stuff
    InitAll();

    // Fail if we have errors in the config file
    if (read_config_file(config_file)) {
        syslog(LOG_ERR, "Failed to process main config file '%s'\nExiting!\n",
               config_file);
        return NOT_OK;
    }

    // Run as a service
    ServiceTable[0].lpServiceName = "NRPE_NT";
    ServiceTable[0].lpServiceProc =
        (LPSERVICE_MAIN_FUNCTION) wait_for_connections;
    ServiceTable[1].lpServiceName = NULL;
    ServiceTable[1].lpServiceProc = NULL;

    syslog(LOG_DEBUG, "Starting server thread...\n");

    result = StartServiceCtrlDispatcher(ServiceTable);

    if (!result) {
        result = GetLastError();
        switch (result) {
        case ERROR_FAILED_SERVICE_CONTROLLER_CONNECT:
            Usage("NRPE_NT.EXE");
            syslog(LOG_ALERT,
                   "Unable to start service: Failed to connect to service controller.\n");
            break;
        case ERROR_INVALID_DATA:
            printf
                ("Unable to start service: Invalid data in function call.\n");
            syslog(LOG_ALERT,
                   "Unable to start service: Invalid data in function call.\n");
            break;
        case ERROR_SERVICE_ALREADY_RUNNING:
            printf("Unable to start service: Service already running.\n");
            syslog(LOG_ALERT,
                   "Unable to start service: Service already running.\n");
            break;
        default:
            printf("Unable to start service:  Unknown error: $d\n", result);
            syslog(LOG_ALERT, "Unable to start service:  Unknown error: $d\n",
                   result);
            break;
        }
    }

    WSACleanup();
    CloseAll();

    syslog(LOG_INFO, "NRPE_NT shutting down...\n");
    CloseLogFile();
    return result;

}


int process_arguments(int argc, char **argv)
{
    int i, res = OK;
    char *ptr;

    GetModuleFileName(NULL, exe_path, sizeof(exe_path));

    /* Set the default config file location */
    strcpy(config_file, exe_path);
    ptr = strrchr(config_file, '\\');
    ptr[0] = 0;
    strcat(config_file, "\\nrpe.cfg");

    if (argc > 1) {
        for (i = 1; i < argc; i++) {
            if ((argv[i][0] == '-') || (argv[i][0] == '/')) {
                switch (tolower(argv[i][1])) {
                case 'u':
                    Uninstall = -1;
                    break;
                case 'i':
                    Install = -1;
                    break;
                case 'c':
                    strcpy(config_file, argv[++i]);
                    break;
                case 'n':
                    use_ssl = FALSE;
                    break;
                default:
                    res = NOT_OK;
                    break;
                }
            }
            else {
                res = NOT_OK;
            }
        }
    }

    return res;
}

void wait_for_connections(int argc, char **argv)
{
    int result, maxdelay, checkp = 0;
    //unsigned dw;
    int fromlen;
    struct sockaddr_in local, from;
    WSADATA wsaData;
    SOCKET listen_socket, msgsock;
    char connecting_host[16];
    unsigned long NoBlock = 1;
    unsigned long inServerAddress;
    int flag = 1;
    __try {

        hStopEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
        if (!hStopEvent) {
            syslog(LOG_ERR, "Create StopEvent failed!\n");
            return;
        }

        srvstatus =
            RegisterServiceCtrlHandler("NRPE_NT",
                                       (LPHANDLER_FUNCTION) ControlHandler);
        if (!srvstatus) {
            syslog(LOG_ERR, "RegisterServiceCtrlHandler failed!\n");
            return;
        }

        SetSrvStatus(SERVICE_START_PENDING, NO_ERROR, checkp, 100);
        // WSAStartup
        if ((result = WSAStartup(0x202, &wsaData)) != 0) {
            syslog(LOG_ERR, "WSAStartup failed with error %d\n", result);
            WSACleanup();
            SetSrvStatus(SERVICE_STOPPED, result, 0, 0);
            return;
        }

        local.sin_family = AF_INET;

        if (strlen(server_address) == 0) {
            inServerAddress = INADDR_ANY;
        }
        else {
            inServerAddress = inet_addr(server_address);
        }

        if (inServerAddress == INADDR_NONE) {
            syslog(LOG_ERR,
                   "Invalid Server address while trying to bind to: %s\n Exiting\n",
                   server_address);
            WSACleanup();
            SetSrvStatus(SERVICE_STOPPED, result, 0, 0);
            return;
        }
        else {
            if (inServerAddress == INADDR_ANY) {
                syslog(LOG_DEBUG, "Binding to all interfaces\n");

            }
            else {
                syslog(LOG_DEBUG, "Binding to %s\n", server_address);

            }
            local.sin_addr.s_addr = inServerAddress;
        }

        SetSrvStatus(SERVICE_START_PENDING, NO_ERROR, ++checkp, 100);

        // socket 
        /* Port MUST be in Network Byte Order */
        local.sin_port = htons(server_port);
        listen_socket = socket(AF_INET, SOCK_STREAM, 0);

        setsockopt(listen_socket, SOL_SOCKET, SO_REUSEADDR, (char *) &flag,
                   sizeof(flag));

        if (listen_socket == INVALID_SOCKET) {
            result = WSAGetLastError();
            syslog(LOG_ERR, "socket() failed with error %d\n", result);
            WSACleanup();
            SetSrvStatus(SERVICE_STOPPED, result, 0, 0);
            return;
        }
        SetSrvStatus(SERVICE_START_PENDING, NO_ERROR, ++checkp, 100);

        // bind 
        if (bind(listen_socket, (struct sockaddr *) &local, sizeof(local))
            == SOCKET_ERROR) {
            result = WSAGetLastError();
            syslog(LOG_ERR, "bind() failed with error %d\n", result);
            WSACleanup();
            SetSrvStatus(SERVICE_STOPPED, result, 0, 0);
            return;
        }
        SetSrvStatus(SERVICE_START_PENDING, NO_ERROR, ++checkp, 100);

        //listen

        if (listen(listen_socket, SOMAXCONN) == SOCKET_ERROR) {
            result = WSAGetLastError();
            syslog(LOG_ERR, "listen() failed with error %d\n", result);
            WSACleanup();
            SetSrvStatus(SERVICE_STOPPED, result, 0, 0);
            return;
        }

        ioctlsocket(listen_socket, FIONBIO, &NoBlock);
        syslog(LOG_INFO, "Listening on port %d\n", server_port);
        SetSrvStatus(SERVICE_RUNNING, NO_ERROR, 0, 0);

        //Main listener loop, exit on service stop received....
        while (!(WaitForSingleObject(hStopEvent, 100) == WAIT_OBJECT_0)) {

            fromlen = sizeof(from);

            msgsock =
                accept(listen_socket, (struct sockaddr *) &from, &fromlen);
            if (msgsock == INVALID_SOCKET) {
                result = WSAGetLastError();
                if (!(result == WSAEWOULDBLOCK)) {
                    syslog(LOG_ERR, "accept() error %d\n", result);
                    WSACleanup();
                    SetSrvStatus(SERVICE_STOPPED, result, 0, 0);
                    return;
                }
                else {
                    continue;
                }
            }

            //syslog(LOG_DEBUG, "connection attempt from %s, port %d\n",inet_ntoa(from.sin_addr), htons(from.sin_port));

            /* is this is a blessed machine? */
            _snprintf(connecting_host, sizeof(connecting_host), "%s",
                      inet_ntoa(from.sin_addr));
            connecting_host[sizeof(connecting_host) - 1] = '\x0';

            if (!is_an_allowed_host(connecting_host)) {
                // log error to syslog facility 
                syslog(LOG_ERR, "Host %s is not allowed to talk to us!\n",
                       connecting_host);
            }
            else {

                // log info to syslog facility 
                //syslog(LOG_DEBUG, "Host address checks out ok\n");

                // handle the client connection 
                _beginthread(handle_connection, 0, (void *) msgsock);
                continue;
            }
        }



        /* Idle for 110% command_timeout seconds, to allow child threads to exit... */
        if (delay_shutdown) {   //SERVICE_STOP_PENDING
            //Avoid insane values for maxdelay
            maxdelay = __min(command_timeout, 300);
            maxdelay = maxdelay * 11;
            syslog(LOG_DEBUG,
                   "Waiting for child threads to exit (Delay: %d)\n",
                   maxdelay);
            for (checkp = 1; checkp < maxdelay; checkp++) {
                SetSrvStatus(SERVICE_STOP_PENDING, 0, checkp, maxdelay * 110);
                Sleep(100);
            }

        }
        shutdown(listen_socket, SD_BOTH);
        closesocket(listen_socket);

        syslog(LOG_DEBUG, "Exited listener loop\n");
        SetSrvStatus(SERVICE_STOPPED, 0, 0, 0);
        return;
    }
    __except(1) {
        syslog(LOG_ERR,
               "Error: Exception caught in wait_for_connections()\n");
    }

}


void handle_connection(void *param)
{
    //short result;
    int rc;
    SOCKET consock;
    packet receive_packet, send_packet;
    u_int32_t calculated_crc32;
    int bytes_to_recv;
    int bytes_to_send;
    char buffer[MAX_INPUT_BUFFER];
    command *temp_command;
    char raw_command[MAX_INPUT_BUFFER];
    char processed_command[MAX_INPUT_BUFFER];
    int early_timeout = FALSE;
    unsigned long NoBlock = 1;
    int x;
    short result = STATE_OK;
    int i = 0;
    char *command_name = NULL;
    char *macro_argv[MAX_COMMAND_ARGUMENTS];

#ifdef HAVE_SSL
    SSL *ssl = NULL;
#endif

    __try {

        consock = (SOCKET) param;
        ioctlsocket(consock, FIONBIO, &NoBlock);

#ifdef HAVE_SSL
        __try {
            /* do SSL handshake */
            if (use_ssl == TRUE) {
                ssl = SSL_new(ctx);
                if (ssl != NULL) {
                    rc = SSL_set_fd(ssl, consock);
                    if (rc == 0) {
                        syslog(LOG_ERR,
                               "Error: Could not do SSL_set_fd. %d:%d\n",
                               SSL_get_error(ssl, rc), rc);
                        goto AbortCon;
                    }
                    if (my_ssl_accept(ssl) != OK) {
                        goto AbortCon;
                    }
                }
                else {
                    syslog(LOG_ERR,
                           "Error: Could not create SSL connection structure.\n");
                    goto AbortCon;
                }
            }
        }
        __except(1) {
            syslog(LOG_ERR,
                   "Error: Exception caught in handle_connection:SSL_Handshake:%d:%d\n",
                   rc, i);
            goto AbortCon;
        }

#endif
        __try {
            bytes_to_recv = sizeof(receive_packet);
            if (use_ssl == FALSE)
                rc = recvall(consock, (char *) &receive_packet,
                             &bytes_to_recv, socket_timeout);
#ifdef HAVE_SSL
            if (use_ssl == TRUE) {
                if ((rc =
                     my_ssl_read(ssl, &receive_packet,
                                 bytes_to_recv)) == NOT_OK) {
                    goto AbortCon;
                }
            }
#endif
        }
        __except(1) {
            syslog(LOG_ERR,
                   "Error: Exception caught in handle_connection:receive_packet\n");
        }

        __try {

            if (rc <= 0) {
                if (use_ssl == TRUE)
                    syslog(LOG_ERR, "Error: Socket read failed (SSL)...\n");
                else
                    syslog(LOG_ERR,
                           "Error: Socket read failed (NonSSL)...\n");
                goto AbortCon;
            }

            if (rc != sizeof(receive_packet)) {
                syslog(LOG_ERR, "Error: Invalid packet size on receive!\n");
                goto AbortCon;
            }

            /* Init vars */
            command_name = NULL;
            for (x = 0; x < MAX_COMMAND_ARGUMENTS; x++) {
                macro_argv[x] = NULL;
            }

            if (validate_request(&receive_packet, &command_name, macro_argv)
                == -1) {

                /* log an error */
                syslog(LOG_ERR,
                       "Client request was invalid, bailing out...\n");

                /* free memory */
                free(command_name);
                command_name = NULL;
                for (x = 0; x < MAX_COMMAND_ARGUMENTS; x++) {
                    free(macro_argv[x]);
                    macro_argv[x] = NULL;
                }
                goto AbortCon;
            }

        }
        __except(1) {
            syslog(LOG_ERR,
                   "Error: Exception caught in handle_connection:Check_request\n");
        }

        __try {

            syslog(LOG_DEBUG,
                   "Host is asking for command '%s' to be run...\n",
                   receive_packet.buffer);

            if (!strcmp(command_name, NRPE_HELLO_COMMAND)) {

                syslog(LOG_DEBUG, "Echoing Server Version back to client\n");
                _snprintf(buffer, sizeof(buffer), "NRPE_NT v%s",
                          PROGRAM_VERSION);
                buffer[sizeof(buffer) - 1] = '\x0';

                /* log info to syslog facility */
                syslog(LOG_DEBUG, "Response: %s\n", buffer);

                result = STATE_OK;
            }
            else {
                temp_command = find_command(command_name);
                if (temp_command == NULL) {
                    _snprintf(buffer, sizeof(buffer),
                              "NRPE: Command '%s' not defined", command_name);
                    buffer[sizeof(buffer) - 1] = '\x0';

                    /* log error to syslog facility */
                    syslog(LOG_DEBUG, "%s\n", buffer);
                    result = STATE_CRITICAL;
                }

                else {

                    __try {

                        /* process command line */
                        strncpy(raw_command, temp_command->command_line,
                                sizeof(raw_command) - 1);
                        raw_command[sizeof(raw_command) - 1] = '\x0';
                        process_macros(raw_command, processed_command,
                                       sizeof(processed_command), macro_argv);

                        /* log info to syslog facility */
                        syslog(LOG_DEBUG, "Running command: %s\n",
                               processed_command);

                        /* run the command */
                        strcpy(buffer, "");
                        result =
                            my_system(processed_command, command_timeout,
                                      &early_timeout, buffer, sizeof(buffer));

                        /* log debug info */
                        syslog(LOG_DEBUG,
                               "Command completed with return code %d\n",
                               result);

                        /* see if the command timed out */
                        if (early_timeout == TRUE)
                            _snprintf(buffer, sizeof(buffer) - 1,
                                      "NRPE: Command timed out after %d seconds\n",
                                      command_timeout);
                        else if (!strcmp(buffer, ""))
                            _snprintf(buffer, sizeof(buffer) - 1,
                                      "NRPE: Unable to read output\n");

                        buffer[sizeof(buffer) - 1] = '\x0';

                        /* check return code bounds */
                        if ((result < 0) || (result > 3)) {

                            /* log error to syslog facility */
                            syslog(LOG_ERR,
                                   "Bad return code for [%s]: %d\n",
                                   buffer, result);

                            result = STATE_UNKNOWN;
                        }
                    }
                    __except(1) {
                        syslog(LOG_ERR,
                               "Error: Exception caught in my_system\n");
                    }

                }

            }


        }
        __except(1) {
            syslog(LOG_ERR,
                   "Error: Exception caught in handle_connection:Run command\n");
        }
        /* free memory */
        __try {
            free(command_name);
            command_name = NULL;
            for (x = 0; x < MAX_COMMAND_ARGUMENTS; x++) {
                free(macro_argv[x]);
                macro_argv[x] = NULL;
            }

            /* strip newline character from end of output buffer */
            if (buffer[strlen(buffer) - 1] == '\n')
                buffer[strlen(buffer) - 1] = '\x0';

            /* clear the response packet buffer */
            bzero((char *) &send_packet, sizeof(send_packet));

            /* fill the packet with semi-random data */
            randomize_buffer((char *) &send_packet, sizeof(send_packet));

            /* initialize response packet data */
            send_packet.packet_version =
                (int16_t) htons(NRPE_PACKET_VERSION_2);
            send_packet.packet_type = (int16_t) htons(RESPONSE_PACKET);
            send_packet.result_code = (int16_t) htons(result);
            strncpy(&send_packet.buffer[0], buffer, MAX_PACKETBUFFER_LENGTH);
            send_packet.buffer[MAX_PACKETBUFFER_LENGTH - 1] = '\x0';

            /* calculate the crc 32 value of the packet */
            send_packet.crc32_value = (u_int32_t) 0L;
            calculated_crc32 =
                calculate_crc32((char *) &send_packet, sizeof(send_packet));
            send_packet.crc32_value = (u_int32_t) htonl(calculated_crc32);


        /***** ENCRYPT RESPONSE *****/


            /* send the response back to the client */
            bytes_to_send = sizeof(send_packet);

            if (use_ssl == FALSE)
                result =
                    sendall(consock, (char *) &send_packet, &bytes_to_send);
#ifdef HAVE_SSL
            if (use_ssl == TRUE)
                SSL_write(ssl, &send_packet, bytes_to_send);
#endif

        }
        __except(1) {
            syslog(LOG_ERR,
                   "Error: Exception in handle_connection-packet_Send: %d\n",
                   _exception_code());
        }
    }
    __except(1) {
        syslog(LOG_ERR, "Error: Exception in handle_connection: %d\n",
               _exception_code());
    }


#ifdef HAVE_SSL
    if (use_ssl == TRUE) {
        SSL_shutdown(ssl);
        SSL_free(ssl);
        LogSSLErrors();
        ERR_remove_state(GetCurrentThreadId());
        ssl = NULL;
    }
#endif

    shutdown(consock, SD_BOTH);
    closesocket(consock);
    _endthread();
    return;

  AbortCon:
#ifdef HAVE_SSL
    if (use_ssl == TRUE) {
        SSL_clear(ssl);
        SSL_free(ssl);
        LogSSLErrors();
        ERR_remove_state(GetCurrentThreadId());
        ssl = NULL;
    }
#endif
    shutdown(consock, SD_BOTH);
    closesocket(consock);
    _endthread();
}

int validate_request(packet * pkt, char **command_name,
                     char *macro_argv[MAX_COMMAND_ARGUMENTS])
{
    u_int32_t packet_crc32;
    u_int32_t calculated_crc32;
    char *ptr;
#ifdef ENABLE_COMMAND_ARGUMENTS
    int x;
#endif

        /***** DECRYPT REQUEST ******/


    /* check the crc 32 value */
    packet_crc32 = ntohl(pkt->crc32_value);
    pkt->crc32_value = 0L;
    calculated_crc32 = calculate_crc32((char *) pkt, sizeof(packet));
    if (packet_crc32 != calculated_crc32) {
        syslog(LOG_ERR, "Error: Request packet had invalid CRC32.\n");
        return NOT_OK;
    }

    /* make sure this is the right type of packet */
    if (ntohs(pkt->packet_type) != QUERY_PACKET
        || ntohs(pkt->packet_version) != NRPE_PACKET_VERSION_2) {
        syslog(LOG_ERR, "Error: Request packet type/version was invalid!\n");
        return NOT_OK;
    }

    /* make sure buffer is terminated */
    pkt->buffer[MAX_PACKETBUFFER_LENGTH - 1] = '\x0';

    /* client must send some kind of request */
    if (!strcmp(pkt->buffer, "")) {
        syslog(LOG_ERR, "Error: Request contained no query!\n");
        return NOT_OK;
    }

    /* make sure request doesn't contain nasties */
    if (contains_nasty_metachars(pkt->buffer) == TRUE) {
        syslog(LOG_ERR, "Error: Request contained illegal metachars!\n");
        return NOT_OK;
    }

    /* make sure the request doesn't contain arguments */
    if (strchr(pkt->buffer, '!')) {
#ifdef ENABLE_COMMAND_ARGUMENTS
        if (allow_arguments == FALSE) {
            syslog(LOG_ERR,
                   "Error: Request contained command arguments, but argument option is not enabled!\n");
            return NOT_OK;
        }
#else
        syslog(LOG_ERR, "Error: Request contained command arguments!\n");
        return NOT_OK;
#endif
    }

    /* get command name */
#ifdef ENABLE_COMMAND_ARGUMENTS
    ptr = strtok(pkt->buffer, "!");
#else
    ptr = pkt->buffer;
#endif
    *command_name = strdup(ptr);
    if (*command_name == NULL) {
        syslog(LOG_ERR, "Error: Memory allocation failed\n");
        return NOT_OK;
    }

#ifdef ENABLE_COMMAND_ARGUMENTS
    /* get command arguments */
    if (allow_arguments == TRUE) {

        for (x = 0; x < MAX_COMMAND_ARGUMENTS; x++) {
            ptr = strtok(NULL, "!");
            if (ptr == NULL)
                break;
            macro_argv[x] = strdup(ptr);
            if (macro_argv[x] == NULL) {
                syslog(LOG_ERR, "Error: Memory allocation failed\n");
                return NOT_OK;
            }
            if (!strcmp(macro_argv[x], "")) {
                syslog(LOG_ERR,
                       "Error: Request contained an empty command argument\n");
                return NOT_OK;
            }
        }
    }
#endif

    return 0;
}


int contains_nasty_metachars(char *str)
{
    unsigned int result;

    if (str == NULL)
        return FALSE;

    if (use_win_metachars) {
        result = strcspn(str, NASTY_METACHARS_WIN);
    }
    else {
        result = strcspn(str, NASTY_METACHARS);
    }

    if (result != strlen(str))
        return TRUE;

    return FALSE;
}



int my_system(char *command, int timeout, int *early_timeout,
              char *output, int output_length)
{
    PROCESS_INFORMATION pi;
    STARTUPINFO si;
    HANDLE hChildOutR, hChildOutW, hChildInR, hChildInW;
    SECURITY_ATTRIBUTES sec;
    DWORD dwread, dwstate, dwtimeout, dwexitcode;
    char *temp;
    int retval, result;
    BOOL processOK;
    LPVOID lpMsgBuf;


    __try {
        // Set up members of SECURITY_ATTRIBUTES structure. 

        sec.nLength = sizeof(SECURITY_ATTRIBUTES);
        sec.bInheritHandle = TRUE;
        sec.lpSecurityDescriptor = NULL;

// Create Pipes
        CreatePipe(&hChildInR, &hChildInW, &sec, 0);
        CreatePipe(&hChildOutR, &hChildOutW, &sec, 0);

// Set up members of STARTUPINFO structure. 

        ZeroMemory(&si, sizeof(STARTUPINFO));
        si.cb = sizeof(STARTUPINFO);
        si.dwFlags = STARTF_USESTDHANDLES;
        si.hStdInput = hChildInR;
        si.hStdOutput = hChildOutW;
        si.hStdError = hChildOutW;


// Create the child process. 

        processOK = CreateProcess(NULL, command,        // command line 
                                  NULL, // process security attributes 
                                  NULL, // primary thread security attributes 
                                  TRUE, // handles are inherited 
                                  0,    // creation flags 
                                  NULL, // use parent's environment 
                                  NULL, // use parent's current directory 
                                  &si,  // STARTUPINFO pointer 
                                  &pi); // receives PROCESS_INFORMATION 

        if (processOK) {
            //Timeout is ms, not seconds....
            dwtimeout = 1000 * timeout;
            dwstate = WaitForSingleObject(pi.hProcess, dwtimeout);
            CloseHandle(hChildInR);
            CloseHandle(hChildInW);
            CloseHandle(hChildOutW);

            if (dwstate == WAIT_TIMEOUT) {
                TerminateProcess(pi.hProcess, 5);
                strcpy(output,
                       "The check didn't respond within the timeout period!");
                result = 5;
            }
            else {

                retval =
                    ReadFile(hChildOutR, output, output_length - 1,
                             &dwread, NULL);
                if (!retval || dwread == 0) {
                    strcpy(output, "No output available from command...");
                    result = GetExitCodeProcess(pi.hProcess, &dwexitcode);
                    if (!result)
                        result = STATE_UNKNOWN;
                    else
                        result = dwexitcode;
                }
                else {
                    dwread = __min(dwread, MAX_INPUT_BUFFER - 1);
                    output[dwread] = '\x0';
                    temp = strchr(output, '\n');
                    if (temp) {
                        retval = temp - output + 1;
                        output[retval] = '\x0';
                    }
                    result = GetExitCodeProcess(pi.hProcess, &dwexitcode);
                    if (!result)
                        result = STATE_UNKNOWN;
                    else
                        result = dwexitcode;
                }
            }
            CloseHandle(pi.hThread);
            CloseHandle(pi.hProcess);
            CloseHandle(hChildOutR);
        }
        else {

            FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                          (LPTSTR) & lpMsgBuf, 0, NULL);

            strcpy(output, "NRPE_NT failed to create process, exiting...");
            syslog(LOG_ERR,
                   "Error: Failed to create process for command %s \nSystemerror:%s\n",
                   command, lpMsgBuf);
            LocalFree(lpMsgBuf);
            result = STATE_UNKNOWN;
            CloseHandle(hChildInR);
            CloseHandle(hChildInW);
            CloseHandle(hChildOutW);
            CloseHandle(pi.hThread);
            CloseHandle(pi.hProcess);
            CloseHandle(hChildOutR);
        }


    }

    __except(1) {
        syslog(LOG_ERR, "Error: Exception caused by command %s\n", command);
        strcpy(output, "Command Exception occured!");
        result = STATE_UNKNOWN;
    }

    return result;
}


int read_config_dir(char *dirname)
{
    char config_file[MAX_PATH + 1];
    char szDir[MAX_PATH + 1];
    HANDLE dirp;
    WIN32_FIND_DATA FileData;
    DWORD e;

    int result = OK;
    int x;
    int finished = 0;

    /* create the directory path */
    _snprintf(szDir, sizeof(szDir) - 1, "%s\\*", dirname);
    szDir[sizeof(szDir) - 1] = 0;       /* _snprintf doesn't always zero terminate */

    syslog(LOG_DEBUG, "Processing config directory %s\n", dirname);

    if ((dirp = FindFirstFile(szDir, &FileData)) == INVALID_HANDLE_VALUE) {
        syslog(LOG_ERR,
               "Could not open config directory '%s' for reading.\n",
               dirname);
        return NOT_OK;
    }

    /* process the files */
    while (!finished) {

        /* process this if it's a config file... */
        x = strlen(FileData.cFileName);
        if (x > 4 && !strcmp(FileData.cFileName + (x - 4), ".cfg")) {

            /* only process normal files */
            if (FileData.dwFileAttributes == FILE_ATTRIBUTE_NORMAL ||
                (FileData.dwFileAttributes &
                 (FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_ARCHIVE |
                  FILE_ATTRIBUTE_HIDDEN))) {

                /* create the full path to the config file */
                _snprintf(config_file, sizeof(config_file) - 1, "%s\\%s",
                          dirname, FileData.cFileName);
                config_file[sizeof(config_file) - 1] = '\x0';

                /* process the config file */
                result = read_config_file(config_file);

                /* break out if we encountered an error */
                if (result != OK) {
                    syslog(LOG_INFO,
                           "Error reading config file while processing include_dir directive\n");
                    break;
                }
            }
        }

        /* recurse into subdirectories... */
        if (FileData.dwFileAttributes == FILE_ATTRIBUTE_DIRECTORY) {

            /* ignore current, parent and hidden directory entries */
            if (strcmp(FileData.cFileName, ".") &&
                strcmp(FileData.cFileName, "..")) {

                /* create the full path to the config directory */
                if (FileData.cFileName[1] == ':' &&
                    FileData.cFileName[2] == '\\') {
                    _snprintf(config_file, sizeof(config_file) - 1,
                              FileData.cFileName);
                }
                else {
                    _snprintf(config_file, sizeof(config_file) - 1,
                              "%s\\%s", dirname, FileData.cFileName);
                }
                config_file[sizeof(config_file) - 1] = '\x0';

                syslog(LOG_DEBUG, "Recursive config directory: %s\n",
                       config_file);

                /* process the config directory */
                result = read_config_dir(config_file);

                /* break out if we encountered an error */
                if (result != OK) {
                    break;
                }
            }
        }

        if (!FindNextFile(dirp, &FileData)) {
            if ((e = GetLastError()) != ERROR_NO_MORE_FILES) {
                syslog(LOG_ERR,
                       "Error getting next file from directory '%s' for reading.\n",
                       dirname);
            }
            finished = 1;
        }
    }

    FindClose(dirp);

    return result;
}

/* adds a new command definition from the config file to the list in memory */
int add_command(char *command_name, char *command_line)
{
    command *new_command;

    /* allocate memory for the new command */
    new_command = (command *) malloc(sizeof(command));
    if (new_command == NULL)
        return NOT_OK;

    strcpy(new_command->command_name, command_name);
    strcpy(new_command->command_line, command_line);

    /* add new command to head of list in memory */
    new_command->next = command_list;
    command_list = new_command;

    syslog(LOG_DEBUG, "Added command[%s]=%s\n", command_name, command_line);

    return OK;
}

/* given a command name, find the structure in memory */
command *find_command(char *command_name)
{
    command *temp_command;

    for (temp_command = command_list; temp_command != NULL;
         temp_command = temp_command->next)
        if (!strcmp(command_name, temp_command->command_name))
            return temp_command;

    return NULL;
}


void free_memory(void)
{
    command *this_command;
    command *next_command;

    /* free memory for the command list */
    this_command = command_list;
    while (this_command != NULL) {
        next_command = this_command->next;
        free(this_command);
        this_command = next_command;
    }

    return;
}

/* read in the configuration file */
int read_config_file(char *filename)
{
    FILE *fp;
    char input_buffer[MAX_INPUT_BUFFER];
    char *temp_buffer;
    char *varname;
    char *varvalue;
    int line;

    syslog(LOG_DEBUG, "Trying to open config file '%s' for reading\n",
           filename);

    /* open the config file for reading */
    fp = fopen(filename, "r");

    /* exit if we couldn't open the config file */
    if (fp == NULL) {
        syslog(LOG_ERR, "Unable to open config file '%s' for reading\n",
               filename);
        return NOT_OK;
    }

    syslog(LOG_DEBUG, "Opened config file '%s' for reading\n", filename);

    line = 0;
    while (fgets(input_buffer, MAX_INPUT_BUFFER - 1, fp)) {

        line++;

        /* skip comments and blank lines */
        if (input_buffer[0] == '#')
            continue;
        if (input_buffer[0] == '\x0')
            continue;
        if (input_buffer[0] == '\n')
            continue;

        /* get the variable name */
        varname = strtok(input_buffer, "=");
        if (varname == NULL) {
            syslog(LOG_ERR,
                   "No variable name specified in config file '%s' - Line %d\n",
                   filename, line);
            fclose(fp);
            return NOT_OK;
        }

        /* get the variable value */
        varvalue = strtok(NULL, "\n");
        if (varvalue == NULL) {
            syslog(LOG_ERR,
                   "No variable value specified in config file '%s' - Line %d\n",
                   filename, line);
            fclose(fp);
            return NOT_OK;
        }

        /* allow users to specify directories to recurse into for config files */
        else if (!strcmp(varname, "include_dir")) {

            /* strip trailing / if necessary */
            while (varvalue[strlen(varvalue) - 1] == '\\') {
                varvalue[strlen(varvalue) - 1] = '\x0';
            }

            /* process the config directory... */
            if (read_config_dir(varvalue) != OK) {
                syslog(LOG_ERR,
                       "Reading config directory '%s' returned an error!\n",
                       varvalue);
                return NOT_OK;
            }
        }

        /* allow users to specify individual config files to include */
        else if (!strcmp(varname, "include")
                 || !strcmp(varname, "include_file")) {

            /* process the config file... */
            if (read_config_file(varvalue) == -1) {
                syslog(LOG_ERR,
                       "Reading config file '%s' returned an error!\n",
                       varvalue);
                return NOT_OK;
            }
        }

        else if (!strcmp(varname, "server_port")) {
            server_port = atoi(varvalue);
            if (server_port < 1024) {
                syslog(LOG_ERR,
                       "Invalid port number specified in config file '%s' - Line %d\n",
                       filename, line);
                fclose(fp);
                return NOT_OK;
            }
        }

        else if (!strcmp(varname, "server_address")) {
            syslog(LOG_DEBUG, "reading server_address\n");
            strncpy(server_address, varvalue, sizeof(server_address) - 1);
            server_address[sizeof(server_address) - 1] = '\0';
            syslog(LOG_DEBUG, "Done reading server_address\n");

        }

        else if (!strcmp(varname, "allowed_hosts")) {
            if (strlen(input_buffer) > sizeof(allowed_hosts)) {
                syslog(LOG_ERR,
                       "Allowed hosts list too long in config file '%s' - Line %d\n",
                       filename, line);
                fclose(fp);
                return NOT_OK;
            }
            strncpy(allowed_hosts, varvalue, sizeof(allowed_hosts));
            allowed_hosts[sizeof(allowed_hosts) - 1] = '\x0';
        }

        else if (strstr(input_buffer, "command[")) {
            temp_buffer = strtok(varname, "[");
            temp_buffer = strtok(NULL, "]");
            if (temp_buffer == NULL) {
                syslog(LOG_ERR,
                       "Invalid command specified in config file '%s' - Line %d\n",
                       filename, line);
                fclose(fp);
                return NOT_OK;
            }
            add_command(temp_buffer, varvalue);
        }

        else if (strstr(input_buffer, "debug")) {
            debug = atoi(varvalue);
            if (debug > 0)
                debug = TRUE;
            else
                debug = FALSE;
        }

        else if (!strcmp(varname, "nrpe_user"))
            nrpe_user = strdup(varvalue);

        else if (!strcmp(varname, "nrpe_group"))
            nrpe_group = strdup(varvalue);

        else if (!strcmp(varname, "dont_blame_nrpe"))
            allow_arguments = (atoi(varvalue) == 1) ? TRUE : FALSE;

        else if (!strcmp(varname, "command_timeout")) {
            command_timeout = atoi(varvalue);
            if (command_timeout < 1) {
                syslog(LOG_ERR,
                       "Invalid command_timeout specified in config file '%s' - Line %d\n",
                       filename, line);
                fclose(fp);
                return NOT_OK;
            }
        }

        else if (!strcmp(varname, "loglevel")) {
            loglevel = atoi(varvalue);
            if (!loglevel)
                loglevel = 4;
        }

        else if (strstr(input_buffer, "delay_shutdown")) {
            delay_shutdown = atoi(varvalue);
            if (delay_shutdown > 0)
                delay_shutdown = TRUE;
            else
                delay_shutdown = FALSE;
        }

        else if (strstr(input_buffer, "use_win_metachars")) {
            use_win_metachars = atoi(varvalue);
            if (use_win_metachars > 0)
                use_win_metachars = TRUE;
            else
                use_win_metachars = FALSE;
        }

        else {
            syslog(LOG_ERR,
                   "Unknown option specified in config file '%s' - Line %d\n",
                   filename, line);
            fclose(fp);
            return NOT_OK;
        }

    }


    /* close the config file */
    fclose(fp);
    syslog(LOG_DEBUG, "Loaded config from: %s\n", filename);
    return OK;
}


int process_macros(char *input_buffer, char *output_buffer,
                   int buffer_length, char *macro_argv[MAX_COMMAND_ARGUMENTS])
{
    char *temp_buffer;
    int in_macro;
    int arg_index = 0;
    char *selected_macro = NULL;

    strcpy(output_buffer, "");

    in_macro = FALSE;

    for (temp_buffer = my_strsep(&input_buffer, "$"); temp_buffer != NULL;
         temp_buffer = my_strsep(&input_buffer, "$")) {

        selected_macro = NULL;

        if (in_macro == FALSE) {
            if (strlen(output_buffer) + strlen(temp_buffer) <
                buffer_length - 1) {
                strncat(output_buffer, temp_buffer,
                        buffer_length - strlen(output_buffer) - 1);
                output_buffer[buffer_length - 1] = '\x0';
            }
            in_macro = TRUE;
        }
        else {

            if (strlen(output_buffer) + strlen(temp_buffer) <
                buffer_length - 1) {

                /* argument macro */
                if (strstr(temp_buffer, "ARG") == temp_buffer) {
                    arg_index = atoi(temp_buffer + 3);
                    if (arg_index >= 1 && arg_index <= MAX_COMMAND_ARGUMENTS)
                        selected_macro = macro_argv[arg_index - 1];
                }

                /* an escaped $ is done by specifying two $$ next to each other */
                else if (!strcmp(temp_buffer, "")) {
                    strncat(output_buffer, "$",
                            buffer_length - strlen(output_buffer) - 1);
                }

                /* a non-macro, just some user-defined string between two $s */
                else {
                    strncat(output_buffer, "$",
                            buffer_length - strlen(output_buffer) - 1);
                    output_buffer[buffer_length - 1] = '\x0';
                    strncat(output_buffer, temp_buffer,
                            buffer_length - strlen(output_buffer) - 1);
                    output_buffer[buffer_length - 1] = '\x0';
                    strncat(output_buffer, "$",
                            buffer_length - strlen(output_buffer) - 1);
                }


                /* insert macro */
                if (selected_macro != NULL)
                    strncat(output_buffer,
                            (selected_macro ==
                             NULL) ? "" : selected_macro,
                            buffer_length - strlen(output_buffer) - 1);

                output_buffer[buffer_length - 1] = '\x0';
            }

            in_macro = FALSE;
        }
    }

    return OK;
}

/* checks to see if a given host is allowed to talk to us */
int is_an_allowed_host(char *connecting_host)
{
    char temp_buffer[MAX_INPUT_BUFFER];
    char *temp_ptr;

    strncpy(temp_buffer, allowed_hosts, sizeof(temp_buffer));
    temp_buffer[sizeof(temp_buffer) - 1] = '\x0';

    for (temp_ptr = strtok(temp_buffer, ","); temp_ptr != NULL;
         temp_ptr = strtok(NULL, ",")) {
        if (!strcmp(connecting_host, temp_ptr))
            return 1;
    }

    return 0;
}


void SetSrvStatus(DWORD dwCurrentState, DWORD dwWin32ExitCode,
                  DWORD dwCheckPoint, DWORD dwWaitHint)
{
    SERVICE_STATUS st;

    if ((dwCurrentState == SERVICE_START_PENDING)
        || (dwCurrentState == SERVICE_STOP_PENDING)) {
        st.dwControlsAccepted = 0;
    }
    else {
        st.dwControlsAccepted = SERVICE_ACCEPT_STOP | SERVICE_ACCEPT_SHUTDOWN;
    }

    st.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
    st.dwServiceSpecificExitCode = NO_ERROR;
    st.dwCurrentState = dwCurrentState;
    st.dwWin32ExitCode = dwWin32ExitCode;
    st.dwCheckPoint = dwCheckPoint;
    st.dwWaitHint = dwWaitHint;

    if (!SetServiceStatus(srvstatus, &st)) {
        syslog(LOG_ERR,
               "SetServiceStatus() failed with error %d\nShutting down!\n",
               GetLastError());
    }


}

void ControlHandler(DWORD request)
{
    DWORD dwState = SERVICE_RUNNING;

    switch (request) {
    case SERVICE_CONTROL_STOP:
        dwState = SERVICE_STOP_PENDING;
        break;

    case SERVICE_CONTROL_SHUTDOWN:
        dwState = SERVICE_STOP_PENDING;
        break;
    default:
        break;
    }
    SetSrvStatus(dwState, NO_ERROR, 0, 100);
    if (dwState == SERVICE_STOP_PENDING) {

        if (!SetEvent(hStopEvent)) {
            syslog(LOG_ERR, "SetStopEvent failed\n");
        }
    }
    return;
}

void Usage(char *progname)
{

    printf("\n");
    printf("NRPE_NT - Nagios Remote Plugin Executor for Windows\n");
    printf("Copyright (c) 2003-2005 Michael Wirtgen (nrpe_nt@miwi-dv.com)\n");
    printf("based on: NRPE - Nagios Remote Plugin Executor\n");
    printf("Copyright (c) 1999-2003 Ethan Galstad (nagios@nagios.org)\n");
    printf("Version: %s\n", PROGRAM_VERSION);
    printf("Last Modified: %s\n", MODIFICATION_DATE);
    printf("License: GPL\n");
    printf
        ("Nagios and the Nagios logo are registered trademarks of Ethan Galstad.\n");
#ifdef HAVE_SSL
    printf("\nSSL/TLS Available: Anonymous DH Mode, %s\n",
           SSLeay_version(SSLEAY_VERSION));
#endif
#ifndef HAVE_SSL
    printf("\nSSL/TLS Not Available!");
#endif
    printf("\n");
#ifdef ENABLE_COMMAND_ARGUMENTS
    printf
        ("***************************************************************\n");
    printf
        ("** POSSIBLE SECURITY RISK - COMMAND ARGUMENTS ARE SUPPORTED! **\n");
    printf
        ("**      Read the NRPE SECURITY file for more information     **\n");
    printf
        ("***************************************************************\n");
    printf("\n");
#endif

#ifdef HAVE_SSL
    printf("Usage:\t%s [-c <configfile>]|[-i]|[-i -n]|[-u] \n", progname);
#endif
#ifndef HAVE_SSL
    printf("Usage:\t%s [-c <configfile>]|[-i]|[-i -n]|[-u] \n", progname);
#endif
    printf
        ("\t-c reads <configfile> as main configfile (Default: .\\nrpe.cfg) \n");
    printf("\t-i installs nrpe_nt as a service\n");
#ifdef HAVE_SSL
    printf("\t-i -n installs nrpe_nt as a service, with SSL disabled\n");
#endif
    printf("\t-u uninstalls nrpe_nt as a service\n\n");

    display_license();
}

int InitAll()
{
#ifdef HAVE_SSL
    /* initialize SSL */
    DH *dh;
    SSL_METHOD *meth;
#endif

    debug = FALSE;
#ifdef _DEBUG
    debug = TRUE;
#endif
    /* Have a higher loglevel during startup ... */
    loglevel = 6;
    SetLogFile();
    generate_crc32_table();

#ifdef HAVE_SSL
    if (use_ssl == TRUE) {
        SSL_library_init();
        SSLeay_add_ssl_algorithms();
        SSL_load_error_strings();

        meth = SSLv23_server_method();
        if ((ctx = SSL_CTX_new(meth)) == NULL) {
            syslog(LOG_ERR, "Error: could not create SSL context.\n");
            return NOT_OK;
        }

        SSL_CTX_set_cipher_list(ctx, "ADH");
        dh = get_dh512();
        SSL_CTX_set_tmp_dh(ctx, dh);
        DH_free(dh);

        mt_setup();
        syslog(LOG_INFO,
               "SSL/TLS initialized. All network traffic will be encrypted.\n");
    }
    else {
        syslog(LOG_INFO,
               "SSL/TLS NOT initialized. Network encryption DISABLED.\n");
    }
#endif

    return OK;
}

int CloseAll()
{
#ifdef HAVE_SSL
    if (use_ssl == TRUE) {

        mt_cleanup();
        SSL_CTX_free(ctx);
        ERR_free_strings();
    }
#endif
    return OK;
}

#ifdef HAVE_SSL
void LogSSLErrors()
{
    int flags, line;
    char *data, *file;
    unsigned long code;
    char temp_buffer[MAX_INPUT_BUFFER];

    code = ERR_get_error_line_data(&file, &line, &data, &flags);
    while (code) {
        ERR_error_string_n(code, temp_buffer, sizeof(temp_buffer));
        temp_buffer[sizeof(temp_buffer) - 1] = '\x0';

        if (data && (flags & ERR_TXT_STRING)) {
            syslog(LOG_DEBUG, "SSL %s:%s\n", temp_buffer, data);
        }
        else {
            syslog(LOG_DEBUG, "SSL %s:\n", temp_buffer);
        }
        code = ERR_get_error_line_data(&file, &line, &data, &flags);
    }
}


int my_ssl_accept(SSL * ssl)
{
    int rc = 0;
    int i = 0;

    __try {
        while ((rc = SSL_accept(ssl)) != 1) {
            i++;
            if (i >= 100) {
                syslog(LOG_ERR,
                       "Error: reached max retry on SSL_accept, aborting. %d:%d\n",
                       SSL_get_error(ssl, rc), rc);
                return NOT_OK;
            }
            rc = SSL_get_error(ssl, rc);
            if ((rc == SSL_ERROR_WANT_READ)
                || (rc == SSL_ERROR_WANT_WRITE)) {
                //syslog(LOG_DEBUG, "Retrying SSL handshake. %d:%d\n", i, rc);

                Sleep(100);
                continue;
            }
            else {
                syslog(LOG_ERR,
                       "Error: Could not complete SSL handshake. %d:%d:%d\n",
                       i, rc, SSL_get_error(ssl, rc));
                return NOT_OK;
            }
        }
        return OK;
    }
    __except(1) {
        syslog(LOG_ERR,
               "Error: Exception caught in my_ssl_accept:%d:%d\n", i, rc);
        return NOT_OK;
    }

}

int my_ssl_read(SSL * ssl, void *receive_packet, int bytes_to_recv)
{
    int rc = 0;
    int i = 0;

    __try {
        while (((rc = SSL_read(ssl, receive_packet, bytes_to_recv)) < 1)
            ) {
            i++;
            if (i >= 100) {
                syslog(LOG_ERR,
                       "Error: reached max retry on SSL_read, aborting. %d:%d\n",
                       SSL_get_error(ssl, rc), rc);
                return NOT_OK;
            }
            rc = SSL_get_error(ssl, rc);
            if ((rc == SSL_ERROR_WANT_READ)
                || (rc == SSL_ERROR_WANT_WRITE)) {
                //syslog(LOG_DEBUG, "Retrying SSL read... %d:%d\n", i, rc);

                Sleep(100);
                continue;
            }
            else {
                syslog(LOG_DEBUG, "Error During  SSL read... %d:%d\n", i, rc);
                return NOT_OK;
            }

        }
        return rc;

    }
    __except(1) {
        syslog(LOG_ERR,
               "Error: Exception caught in my_ssl_read:%d:%d\n", i, rc);
        return NOT_OK;
    }

}

#endif
