/******************************************************************************\
* wincompat.c
*
* Some compabillity routines to help VC compile the unix sources as much as possible..
* These are only tested with the nrpe sources, and may break on anything else...
*
* NRPE_NT: Nagios Remote Plugin Executor for NT/W2K 
* Copyright (c) 2003 Michael Wirtgen (michael.wirtgen@miwi-dv.com)
*
* Updated:  Patrick Spizzo (ishamael@moridin.com) - 20.Jul.2004
*   Added code to handle error conditions and provide useful error messages.
*   Fixed syslog function to avoid buffer overflows.
*   Fixed the sleep conversion from Unix to Windows from 100 to 1000.
*   Added additional code to UninstallService to check that the service is stopped 
*       and to use the correct permissions for the SCManager.
* 
* based on:
* NRPE.C - Nagios Remote Plugin Executor
* Copyright (c) 1999-2003 Ethan Galstad (nagios@nagios.org)
*
* License: GPL
* 
* Nagios and the Nagios logo are registered trademarks of Ethan Galstad.
*
* strsep code taken from libgw32c-0.3 (http://gnuwin32.sourceforge.net)
* License: GPL
*
* syslog code modified from wsyslog.c from the php sources with the following
* copyright notice:
*
* Copyright 1996 by the University of Washington
*
*  Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose and without fee is hereby granted, provided
* that the above copyright notice appears in all copies and that both the
* above copyright notice and this permission notice appear in supporting
* documentation, and that the name of the University of Washington not be
* used in advertising or publicity pertaining to distribution of the software
* without specific, written prior permission.  This software is made available
* "as is", and
* THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
* WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
* NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
* INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
* LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
* (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
* WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*
*
\******************************************************************************/

#include <time.h>
#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <windows.h>
#include "wincompat.h"
#include "..\evlog.h"

#define MAX_INPUT_BUFFER	2048
#define SYSLOG_BUFSIZE 4096

const char *EVENTLOG_KEY =
    "SYSTEM\\CurrentControlSet\\Services\\EventLog\\Application\\NRPE_NT";
char logFile[MAX_INPUT_BUFFER] = "nrpe_nt.log\0";

FILE *flog;

void ERROR_printText(DWORD e)
{
    switch (e) {
    case ERROR_ACCESS_DENIED:
        printf("Access denied.\n");
        break;
    case ERROR_DATABASE_DOES_NOT_EXIST:
        printf("Non-existent database.\n");
        break;
    case ERROR_INVALID_PARAMETER:
        printf("Invalid parameter.\n");
        break;
    case ERROR_DUPLICATE_SERVICE_NAME:
        printf("Duplicate service name.\n");
        break;
    case ERROR_SERVICE_EXISTS:
        printf("Service exists.\n");
        break;
    case ERROR_INVALID_HANDLE:
        printf("Invalid handle.\n");
        break;
    case ERROR_INVALID_NAME:
        printf("Invalid service name.\n");
        break;
    case ERROR_SERVICE_DOES_NOT_EXIST:
        printf("Service does not exist.\n");
        break;
    case ERROR_SERVICE_MARKED_FOR_DELETE:
        printf("Service already marked for deletion.\n");
        break;
    default:
        printf("Unknown error: %d\n", e);
        break;
    }
}

/* Modified from msdn.com's example of using FormatMessage and GetLastError */
void genericErrorText(DWORD dw)
{
    char szBuf[80];
    LPVOID lpMsgBuf;

    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
                  FORMAT_MESSAGE_FROM_SYSTEM,
                  NULL,
                  dw,
                  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                  (LPTSTR) & lpMsgBuf, 0, NULL);

    _snprintf(szBuf, sizeof(szBuf) - 1, "Error %d: %s", dw, lpMsgBuf);
    szBuf[sizeof(szBuf) - 1] = 0;


    LocalFree(lpMsgBuf);
    ExitProcess(dw);
}


void syslog(int priority, const char *message, ...)
{
    va_list args;
    LPTSTR strs[2];
    char szEvent[1024];
    char tmp[SYSLOG_BUFSIZE];
    unsigned short etype;
    DWORD eID;
    HANDLE loghdl = NULL;
    time_t aclock;
    struct tm *nt;

    /* default event source */

    switch (priority) {         /* translate UNIX type into NT type */

    case LOG_ERR:
    case LOG_ALERT:
        etype = EVENTLOG_ERROR_TYPE;
        eID = EVMSGERROR;
        break;

    case LOG_INFO:
        etype = EVENTLOG_INFORMATION_TYPE;
        eID = EVMSGINFO;
        break;

    case LOG_DEBUG:
        etype = EVENTLOG_INFORMATION_TYPE;
        eID = EVMSGINFO;
        break;

    default:
        etype = EVENTLOG_WARNING_TYPE;
        eID = EVMSGWARNING;
    }

    // Build message 
    va_start(args, message);    /* initialize vararg mechanism */
    _vsnprintf(tmp, sizeof(tmp) - 1, message, args);    /* build message */
    tmp[sizeof(tmp) - 1] = 0;


    if (debug == TRUE) {        //Log all messages to nrpe.log
        if (priority <= loglevel) {
            time(&aclock);
            nt = localtime(&aclock);

            EnterCriticalSection(&g_cs_log);
            if (flog) {
                fprintf(flog, "%04d-%02d-%02d %02d:%02d:%02d; %d; %d;%s",
                        nt->tm_year + 1900, nt->tm_mon + 1, nt->tm_mday,
                        nt->tm_hour, nt->tm_min, nt->tm_sec,
                        GetCurrentThreadId(), priority, tmp);
                fflush(flog);
            }

            LeaveCriticalSection(&g_cs_log);
        }
    }

    if (!(priority == LOG_DEBUG)) {     // Log all but debug message to eventlog

        // Open handle to eventlog
        loghdl = RegisterEventSource(NULL, "NRPE_NT");
        wsprintf(szEvent, "%s: ", "NRPE_NT");
        strs[0] = szEvent;
        strs[1] = tmp;
        /* report the event */
        ReportEvent(loghdl, etype, (unsigned short) priority, eID, NULL, 2, 0,
                    strs, NULL);
        va_end(args);
        // Close evntlog handle
        DeregisterEventSource(loghdl);
    }

    return;
}


// stub for bzero unix call
void bzero(char *dest, int length)
{
    ZeroMemory(dest, length);
}

// Sleep takes ms instead of seconds on win
void sleep(int seconds)
{
    Sleep(1000 * seconds);
}

int InstallService(char *ExePath, int UseSSL)
{
    SC_HANDLE srv, scm;
    HKEY hKey;
    DWORD dwtemp;
    char ServicePath[MAX_INPUT_BUFFER];
// The buffer below ends with 2 zero characters.
// Value needs to be \0 seperated, \0\0 terminated...
    LPCTSTR lpDependencies = __TEXT("Tcpip\0");


    //check if wee need to add a trailing '-n' to disable SSL
    strcpy(ServicePath, ExePath);
#ifdef HAVE_SSL
    if (!UseSSL)
        strcat(ServicePath, " -n");
#endif


    scm = OpenSCManager(NULL, NULL, SC_MANAGER_CREATE_SERVICE);
    if (!scm) {
        printf("Unable to open service control manager: ");
        ERROR_printText(GetLastError());
        return 1;
    }
    srv = CreateService(scm, "NRPE_NT",
                        "Nagios Remote Plugin Executor for NT/W2K",
                        SERVICE_ALL_ACCESS, SERVICE_WIN32_OWN_PROCESS,
                        SERVICE_AUTO_START, SERVICE_ERROR_NORMAL, ServicePath,
                        NULL, NULL, /*NULL*/ lpDependencies, NULL, NULL);

    if (srv == NULL) {
        printf("Unable to create new service: ");
        ERROR_printText(GetLastError());
        CloseServiceHandle(scm);
        return 1;
    }
    CloseServiceHandle(srv);
    CloseServiceHandle(scm);

    if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, EVENTLOG_KEY, 0, NULL,
                       REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey,
                       &dwtemp)
        == ERROR_SUCCESS) {

        RegSetValueEx(hKey, "EventMessageFile", 0, REG_EXPAND_SZ,
                      (LPBYTE) ExePath, strlen(ExePath) + 1);
        dwtemp =
            EVENTLOG_ERROR_TYPE | EVENTLOG_WARNING_TYPE |
            EVENTLOG_INFORMATION_TYPE;
        RegSetValueEx(hKey, "TypesSupported", 0, REG_DWORD, (LPBYTE) & dwtemp,
                      sizeof(DWORD));
        RegCloseKey(hKey);
    }
    else {
        printf("Could not register EventSource: ");
        genericErrorText(GetLastError());
    }

    return 0;
}

int UninstallService()
{
    SC_HANDLE srv, scm;
    int res = 0;
    SERVICE_STATUS ss;

    if ((scm = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)) == NULL) {
        printf("Unable to open service control manager: ");
        ERROR_printText(GetLastError());
        return 1;
    }

    if ((srv = OpenService(scm, "NRPE_NT", SERVICE_ALL_ACCESS)) == NULL) {
        printf("Unable to open service: ");
        ERROR_printText(GetLastError());
        CloseServiceHandle(scm);
        return 1;
    }

    /* Stop the service if it is running */
    if (!QueryServiceStatus(srv, &ss)) {
        printf("Unable to query service: ");
        ERROR_printText(GetLastError());
        return 1;
    }

    if (ss.dwCurrentState != SERVICE_STOPPED) {
        printf("You must stop the service before you delete it.\n");
        printf("Run the command 'net stop nrpe_nt'.\n");
        return 1;
    }

    if (!DeleteService(srv)) {
        printf("Unable to delete service: ");
        ERROR_printText(GetLastError());
        res = 1;
    }

    if (RegDeleteKey(HKEY_LOCAL_MACHINE, EVENTLOG_KEY) != ERROR_SUCCESS) {
        printf("Could not unregister EventSource: ");
        genericErrorText(GetLastError());
    }

    CloseServiceHandle(srv);
    CloseServiceHandle(scm);
    return res;
}

void SetLogFile()
{
    char *ptr;

    GetModuleFileName(NULL, logFile, sizeof(logFile));

    /* Set the default config file location */
    ptr = strrchr(logFile, '\\');
    ptr[0] = 0;
    strncat(logFile, "\\nrpe_nt.log", MAX_INPUT_BUFFER - strlen(logFile) - 1);
    logFile[MAX_INPUT_BUFFER - 1] = '\x0';

    InitializeCriticalSection(&g_cs_log);

    EnterCriticalSection(&g_cs_log);
    flog = fopen(logFile, "a+");
    if (!flog) {
        syslog(LOG_ERR, "Unable to open logfile:%s\n", logFile);
    }

    LeaveCriticalSection(&g_cs_log);

}

void CloseLogFile()
{
    EnterCriticalSection(&g_cs_log);
    if (flog) {
        fclose(flog);
    }
    LeaveCriticalSection(&g_cs_log);
    DeleteCriticalSection(&g_cs_log);
}
