#!/usr/bin/perl
#
#    Copyright (C) 2003 by Martin Schmitz net&works GmbH, Germany
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#------------------------------------------------------------------------------
#
# This Program is an agent for Nagios, which monitors the Windows EventLog
# and sends the State back to the check_win_eventlog plugin.
# The agent waits for requests on a portnumber given on the commandline
# (Default is 1903).
# All other Parameter will be send by the correspondending nagios plugin.
#
use Win32::EventLog;
use IO::Socket::INET;
use Getopt::Std;

# This method reads the eventlog protocol specified by the first parameter
# and stores the results in a hash Reference given as second parameter.
# The rest of the parameter are used to select specific messages.
sub processLog
{
        my $list = shift;
        my $stat = shift;
        my $tmptype = shift;
        my $tmpevt = shift;
        my $tmpsource = shift;
        my $tmpmessage = shift;
        my @types = split(/:/,$tmptype);
        my @eventids = split(/:/,$tmpevt);
        my @sources = split(/:/,$tmpsource);
        my @messages = split(/:/,$tmpmessage);
        my %event=(
                'Length',NULL,
                'RecordNumber',NULL,
                'TimeGenerated',NULL,
                'TimeWritten',NULL,
                'EventID',NULL,
                'EventType',NULL,
                'Category',NULL,
                'ClosingRecordNumber',NULL,
                'Source',NULL,
                'Computer',NULL,
                'Strings',NULL,
                'Data',NULL,
        );
        #Read the last processed event number.
        my $lastEvent = $$stat{"LAST_PROCESSED_EVENT"};
        #Open the Eventlog.
        my $eventLog=new Win32::EventLog( $list ) || die "Could not open EventLog Protocol: $list";
        $eventLog->GetOldest($oldest) ||die "Could not read First Event Number";
        $eventLog->GetNumber($number) ||die "Could not read number of Events";
        # calculate the last eventnumber to be processed
        $stopEvent = $oldest + $number;
        #eventlog has been reset for some reason
        if ($lastEvent > $stopEvent)
        {
                $lastEvent = 0;
        }
        #we cannot process events, that are no longer stored.
        #so we just move to the oldest event
        if ($lastEvent < $oldest)
        {
                $lastEvent = $oldest;
        }
        $matchcount = 0;
        #reset the match count
        $$stat{"MATCH_COUNT"} = $matchcount;
        #for each event from the last processed one
        while ($lastEvent < $stopEvent)
        {
                #read the event
                $eventLog->Read((EVENTLOG_SEEK_READ|EVENTLOG_FORWARDS_READ),$lastEvent,$event) || die "Could not read Entry #$lastEvent";
                #GetMessageText MUST be called HERE!!! Otherwise it will not return a value
                $text = Win32::EventLog::GetMessageText($event);
                #to get a readable EventId
                $event->{'EventID'} = $event->{'EventID'} & 0xffff;
                #to get a readable Time
                $time = localtime($event->{'TimeGenerated'});

                #match the regular expressions against the event fields
                if( (not match(\@types,$event->{'EventType'}))
                         and (not match(\@eventids,$event->{'EventID'}))
                         and (not match(\@messages,$text))
                         and (not match(\@sources,$event->{'Source'}))
                         )
                {
                        #This should be notified, so increase the matchcount
                        $matchcount++;
                        #Store the State in the status hash.
                        $$stat{"LAST_MESSAGE"} = $text;
                        $$stat{"LAST_TIME"} = $time;
                        $$stat{"LAST_TYPE"} = $event->{'EventType'};
                        $$stat{"LAST_EVENTID"} = $event->{'EventID'};
                        $$stat{"LAST_SOURCE"} = $event->{'Source'};
                }
                $$stat{"MATCH_COUNT"} = $matchcount;
                $lastEvent++;
                $$stat{"LAST_PROCESSED_EVENT"} = $lastEvent;
        }
        $eventLog->Close();
        return 0;
}

# Checks if a skalar is contained in an array.
# This method allows the array to contain regular expressions.
# if an array element starts with '+', then the match is reset.
# this will allow for match any but xy depending on the order
# of the regular expressions
sub match
{
        my ($array,$value) = @_;
        $ret = 0;
        foreach $i (@$array)
        {
                $x = $i; # We need to make an copy here, because otherwise perl will modify the reference in the array
                if( $x =~ /^\+/ ) #starts with '+' ?
                {
                        $x =~ s/^\+//; #remove '+'
                        if( $value =~ /$x/ ) #if match, then reset return value to 0
                        {
                                $ret = 0;
                        }
                }
                else
                {
                        if( $value =~ /$i/ ) #if match, then set return value to true
                        {
                                $ret = 1;
                        }
                }
        }
        return $ret;
}

sub trim
{
    $_ = shift;
    chomp $_; # Remove Linefeeds
    s/#.*//; # Remove Comments
    s/^\s+//; # Remove Spaces from Beginning
    s/\s+$//; # Remove Space from End
    return $_;
}

getopts("p:");
if(not defined($opt_p))
{
        $opt_p = 1903; #default port
}

%STATI = (); #Will contain the status of the different requests
#open the socket.
$socket = new IO::Socket::INET(Listen => 5,
                                                                LocalAddr => '0.0.0.0',
                                                                LocalPort => $opt_p,
                                                                Proto => 'tcp') || die "Could not open Socket $opt_p";
$|=1; #Flush Data immediatly
while(1)
{
        %parameters = (); #Will hold the Command Parameter from the client
        eval #used to catch any exceptions, because the agent should not
                 #die if a client terminates unexpectet
        {
                $current = $socket->accept(); #block until a client wants to connect
                $line = "";
                #read the command parameter
                while($line ne "COMMAND_END")
                {
                        $line = <$current> || die "Could not read Command";
                        $line = trim( $line ); #remove newlines
                        @tmp = split(/=/,$line);
                        $parameters{$tmp[0]} = $tmp[1];
                }
                #get the last state or create a new one
                $status = $STATI{$parameters{"status_id"}};
                if ( not defined ($status))
                {
                        my %st = ("LAST_PROCESSED_EVENT",0);
                        $STATI{$parameters{"status_id"}}=\%st;
                        $status = \%st;
                }

                #look at the eventlog
                processLog( $parameters{"list"}, $status, $parameters{"types"}
                        , $parameters{"event_ids"}
                        , $parameters{"sources"}
                        , $parameters{"messages"} );
                #Send the State back to the requesting plugin
                $current->send("MATCH_COUNT=$$status{'MATCH_COUNT'}\n") || die "Could not send State";
                $current->send("LAST_MESSAGE=$$status{'LAST_MESSAGE'}\n") || die "Could not send State";
                $current->send("LAST_SOURCE=$$status{'LAST_SOURCE'}\n") || die "Could not send State";
                $current->send("LAST_TIME=$$status{'LAST_TIME'}\n") || die "Could not send State";
                $current->send("LAST_TYPE=$$status{'LAST_TYPE'}\n") || die "Could not send State";
                $current->send("LAST_EVENTID=$$status{'LAST_EVENTID'}\n") || die "Could not send State";
                $current->send("STATE_END\n") || die "Could not send State";
                $current->close() || die "Could not close Socket";
        };
        if($@)
        {
                warn $@;
        }
}