#!/usr/bin/env python
#
# Copyright 2014 Cisco Systems, Inc. 
# 
# Licensed under the Apache License, Version 2.0 (the "License"); 
# you may not use this file except in compliance with the License. 
# You may obtain a copy of the License at 
# http://www.apache.org/licenses/LICENSE-2.0 
# 
# Unless required by applicable law or agreed to in writing, software 
# distributed under the License is distributed on an "AS IS" BASIS, 
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
# See the License for the specific language governing permissions and 
# limitations under the License. 
#
#
# This script automates the installation and uninstallation process of the IMC-Nagios 
# plugin in a linux based Nagios environment.
#
# Usage: 
# Plugin Installation :    ./installer.py
#
# Plugin Uninstallation :  ./installer.py -u or ./installer.py --uninstall
#

import os
import sys
import shlex
import subprocess
import glob
import time
from optparse import OptionParser
import csv
import re

time_tag = time.strftime("%Y%m%d-%H%M%S")
temp_location = "/tmp/nagios_temp_" + time_tag
nagios_home = ""
nagios_install_paths = ["/usr/local/nagios/etc", "/etc/nagios3"]
nagios_restart_install_paths = ["/etc/init.d/nagios", "/etc/init.d/nagios3"]

pkg_filename = glob.glob("*.tar.gz")[0]
install_type = 'imc'


def word_to_upper(word):
    '''
    Method used for converting class and attributes passed
    by end user to the format expected by pythonSDK
    :param word: A string containing class or attribute
    '''
    return (word[0].upper() + word[1:])


class CheckForCommentOrBlank:
    '''
    Overriding the Csv Class for checking configuration file for
    hash(#) and spaces and will ignore those lines
    '''
    def __init__(self, fp):
        self.fp = fp

    def __iter__(self):
        return self

    def next(self):
        line = self.fp.next()
        if not line.strip() or line.startswith('#'):
            return self.next()
        return line

    def close(self):
        self.fp.close()


def read_properties(filename):
    '''
    Reads a given properties file with each line of the format key=value.
    Returns a dictionary containing the pairs.
    :param filename: The name of the file to be read
    '''
    # Check if configuration file exists or not
    if os.path.exists(filename) is False:
        raise Exception("Configuration file '%s' not found. "
                        % filename)
    try:
        file_type = execute_unix_cmd("file " + filename)
        # cisco_imc_nagios.cfg: ASCII English text
        if not (file_type[0] == 0 and re.search("ASCII", file_type[1][0])
                is not None):
            raise Exception('Error occurred while checking plugin '
                            'configuration file format.'
                            'File may contain invalid characters.')
    except Exception, cfg_err:
        raise Exception(str(cfg_err))
    result = dict()
    cfg_file = CheckForCommentOrBlank(open(filename, 'rb'))
    reader = csv.reader(cfg_file, delimiter='=', escapechar='\\',
                        quoting=csv.QUOTE_NONE)
    for row in reader:
        if len(row) != 2:
            raise Exception('Check if configuration properties are in the'
                            ' format PROPERTY_NAME = VALUE.\n' +
                            'Also check the VALUE should be in a single line.')
        result[row[0].strip()] = row[1].strip()
    cfg_file.close()
    return result


def write_to_uninstall(txt):
    global uninstall_filename
    fd_uninstall = open(uninstall_filename, 'a')
    fd_uninstall.write(txt+"\n")
    fd_uninstall.flush()
    fd_uninstall.close()


def uninstall(uninstall_filename):
    print "Starting uninstall process now ..."
    try:
        uninstall_dict = read_properties(uninstall_filename)
    except Exception, err:
        print "Error while trying to read uninstall configuration file '%s' ."\
            % uninstall_filename
        print str(err)
        sys.exit(1)
    service_cmd = uninstall_dict['NAGIOS_CMD']
    # Stop services
    print "As a part of the uninstallation process Nagios service needs to be stopped first."
    answer = get_user_input("Do you want Nagios service to be stopped.(Y/N) ? ","N")
    if answer:
        print "Can not uninstall without stopping service.\nExiting now"
        sys.exit(1)
    print "Stopping Nagios Services now."
    retval = execute_unix_cmd(service_cmd + " stop")[0]
    time.sleep(5)
    if retval == 1:
        print ("Nagios service is already stopped.")
    elif retval != 0 :
        print ("Error occurred while stopping the Nagios service.")
        sys.exit(1)
    else :
        print ("Successfully stopped the Nagios service.")
    # rm -f plugin
    plugin_path = uninstall_dict['PLUGIN_PATH']
    print "Removing monitoring plugin : ", plugin_path
    execute_unix_cmd("rm -f " + plugin_path)
    plugin_cfg_path = uninstall_dict['PLUGIN_CFG_PATH']
    answer = get_user_input("Do you want to keep '%s' plugin config file.(Y/N) ? "
                       % plugin_cfg_path,"Y")
    if not answer:
        print "Your file %s will be removed" % plugin_cfg_path
        print "Removing monitoring plugin : ", plugin_cfg_path
        execute_unix_cmd("rm -f " + plugin_cfg_path)
    else:
        print "Configuration file '%s' is not being removed." % plugin_cfg_path

    if "AUTO_PATH" in uninstall_dict:
        # rm -rf  Cisco Object dir
        cisco_object_dir = os.path.join(nagios_home, "cisco",
                                    install_type + "Objs")
        print "Removing cisco object directory "
        execute_unix_cmd("rm -rf " + cisco_object_dir)

        # rm -f logos
        logos_path = uninstall_dict['LOGOS_PATH']
        logo_list = glob.glob(logos_path+"/CISCO."+str(install_type).upper()+".*")
        for logo in logo_list:
            print "Removing ", logo
            execute_unix_cmd("rm -f " + logo)
        # Remove autodiscovery directory
        autodiscovery_path = uninstall_dict['AUTO_PATH']
        print "\nDo you want to remove the AutoDiscovery directory permanently."
        print 'This will also remove any customization done to the host CSV '
        'and discovery configuration file.'
        answer = get_user_input("\nDo you want to remove Auto-Discovery installation.(Y/N) ? ","N")
        if answer:
            print "Auto-Discovery directory will not be removed."
        else:
            execute_unix_cmd("rm -rf " + autodiscovery_path)
            print "Removed Auto-Discovery '%s' directory." % autodiscovery_path

        nagios_cfg_bkp = uninstall_dict['BKP_NAGIOS']
        if edit_file(nagios_cfg_bkp.rsplit("_")[0],
                     "cfg_dir=%s/cisco/%sObjs" % (nagios_home, install_type)):
            print "Failed to update the file %s.\nExiting now" % nagios_cfg_bkp.rsplit("_")[0]
            sys.exit(1)
        else:
            print "Restored the Nagios nagios.cfg configuration file"
        cgi_cfg_bkp = uninstall_dict['BKP_CGI']
        if edit_file(cgi_cfg_bkp.rsplit("_")[0],
                     "escape_html_tags=0", "escape_html_tags=1"):
            print "Failed to update the file %s.\nExiting now" \
                % cgi_cfg_bkp.rsplit("_")[0]
            sys.exit(1)
        else:
            print "Restored the Nagios cgi.cfg configuration file"
    print "Uninstall completed successfully !! "
    execute_unix_cmd("rm -f " + uninstall_filename)
    # Start services
    print "Re-starting the Nagios service now."
    retval = execute_unix_cmd(service_cmd + " start")[0]
    time.sleep(5)
    if retval != 0 :
        print ("Error occurred while starting the Nagios service.")
        sys.exit(1)
    else :
        print ("Successfully started the Nagios service.")
    sys.exit(0)


def execute_unix_cmd(command):
#     print "Executing cmd : ", command
    args = shlex.split(command.strip('"'))
    p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    cmd_output = p.communicate()
    cmd_return_code = p.wait()
    return (cmd_return_code, cmd_output)


def get_install_path(install_path_list):
    found_paths = []
    counter = 0
    for list_path in install_path_list:
        if os.path.exists(list_path):
            if counter == 0:
                print "Found the following paths :"
            counter += 1
            found_paths.append(list_path)
            print "%s. %s" % (counter, list_path)

    if counter == 0:
        print "No default path found....."
        userPath = raw_input("\nPlease provide an absolute path : ")
        if os.path.exists(userPath):
            return userPath
        else:
            print "Your suggested path not found."
            return None
    else:
        counter += 1
        print "%s. Provide your custom install path" % counter
        custom_input = counter
        try:
            userInput = raw_input("Type in your choice [Default '1'] :")
            if userInput == "" or userInput is None or userInput.strip() == "":
                print "\nAs no input was provided taking option '1' as default."
                return found_paths[0]
            elif int(userInput.strip()) == custom_input:
                userPath = raw_input("\nPlease provide an absolute path : ")
                if os.path.exists(userPath):
                    return userPath
                else:
                    print "\nYour suggested path not found ."
                    return None
            elif len(found_paths) >= int(userInput.strip()):
                return found_paths[int(userInput.strip()) - 1]
            else:
                print "Your input does not match any option ."
                return None
        except Exception:
            print "Your input is invalid."
            return None


def is_writable(directory):
    try:
        tmp_prefix = "write_tester"
        count = 0
        filename = os.path.join(directory, tmp_prefix)
        while(os.path.exists(filename)):
            filename = "{}.{}".format(os.path.join(directory,
                                                   tmp_prefix), count)
            count = count + 1
        f = open(filename, "w")
        f.close()
        os.remove(filename)
        return True
    except Exception:
        return False


def extract_package(package_name, destination_path):
    cmd = "tar zxvf %s -C %s" % (package_name, destination_path)
    result_list = execute_unix_cmd(cmd)
    if not result_list[0]:
        print "Extracted the package '%s'" % package_name
        return 1
    else:
        print "Got some error, please check if the package is present or not."
        return 0


def edit_file(file_path, text_to_find, text_to_replace=""):
    if not options.uninstall:
        backup_orig = file_path + "_" + time_tag
        print "Backing up the original file as : ", backup_orig
        f_name = file_path.rsplit("/", 1)[1].rsplit(".", 1)[0].upper()
        write_to_uninstall("BKP_"+f_name+"=" + backup_orig)
        execute_unix_cmd("cp %s %s" % (file_path, backup_orig))
        if text_to_replace == "":
            if not search_file(file_path, text_to_find):
                text_to_find = text_to_find + "\n"
                try:
                    f = open(file_path, 'a')
                    f.write(text_to_find)
                    f.close()
                except Exception:
                    print "Failed to open file"
                    return 1
        else:
            text_to_replace = text_to_replace + "\n"
            try:
                f = open(file_path, 'r')
                lns = f.readlines()
                f.close()
                f = open(file_path, 'w')
                for line in lns:
                    if line.startswith('#'):
                        f.writelines(line)
                    elif text_to_find in line:
                        line = line.replace(line, text_to_replace)
                        f.writelines(line)
                    else:
                        f.writelines(line)
                f.close()
                return 0
            except Exception:
                print "Failed to open file"
                return 1
    else:
        # Removing text_to_find from file
        if text_to_replace == "":
            text_to_replace = "\n"
        else:
            text_to_replace = text_to_replace + "\n"
        try:
            f = open(file_path, 'r')
            lns = f.readlines()
            f.close()
            f = open(file_path, 'w')
            for line in lns:
                if line.startswith('#'):
                    f.writelines(line)
                elif text_to_find in line:
                    line = line.replace(line, text_to_replace)
                    f.writelines(line)
                else:
                    f.writelines(line)
            f.close()
            return 0
        except Exception:
            print "Failed to open file"
            return 1


def search_file(file_path, text_to_search):
    text_to_search = text_to_search + "\n"
    try:
        f = open(file_path, 'r')
        lns = f.readlines()
        f.close()
        for line in lns:
            if line.startswith('#'):
                continue
            elif text_to_search in line:
                return 1
        return 0
    except Exception:
        print "Failed to open file"
        sys.exit(1)


def get_data_from_file(file_path, text_to_search):
    try:
        f = open(file_path, "r")
        lns = f.readlines()
        f.close()
        for line in lns:
            if line.startswith('#'):
                continue
            elif text_to_search in line:
                return line
        return None
    except Exception:
        print "Failed to open file ", file_path
        return None


def get_user_input(raw_string, default_answer):
    counter = 0
    answer_list_yes = ["y", "yes"]
    answer_list_no = ["n", "no"]
    if default_answer.lower() in answer_list_yes:
        default_answer_list = answer_list_yes
        default_reject_list = answer_list_no
    else:
        default_answer_list = answer_list_no
        default_reject_list = answer_list_yes

    while(1):
        input_ans = raw_input(raw_string + "[%s]" % default_answer)
        answer = input_ans.strip().lower()
        if answer == "" or answer is None or answer in default_answer_list:
            return True
        elif answer in default_reject_list:
            return False
        else:
            counter += 1
            print "You entered a wrong input.Please type in Y/N."
            if counter > 1:
                print "\nEntered wrong input again.\nExiting now."
                sys.exit(1)


# Main Starts here
if not os.geteuid() == 0:
    print "\nRun the script with root user only.\nExiting now"
    sys.exit(1)

current_file_path = os.path.dirname(os.path.realpath(__file__))
package_file = os.path.join(current_file_path, pkg_filename)
parser = OptionParser()
parser.add_option("-f", "--package_file", dest="package_file",
                  default=package_file, type="str",
                  help="Provide the input package file.")
parser.add_option("-u", "--uninstall", dest="uninstall",
                  default=False, action="store_true",
                  help="Uninstall the existing plugin.")
parser.add_option("-p", "--plugin", dest="only_plugin",
                  default=False, action="store_true",
                  help="If set this will only install monitoring plugin.")

(options, cmd_args) = parser.parse_args()
package_file = options.package_file.strip()
only_plugin_flag = options.only_plugin
print "Searching for Nagios installation path..."
nagios_home = get_install_path(nagios_install_paths)
if nagios_home is None or nagios_home == "":
    print "Failed to find Nagios installation.\nExiting now."
    sys.exit(1)
else:
    nagios_cfg_file = os.path.join(nagios_home, "nagios.cfg")
    if not os.path.isfile(nagios_cfg_file):
        print "\nFailed to find the Nagios config file %s."\
            % nagios_cfg_file
        print "This path is not a valid Nagios home.\nExiting now."
        sys.exit(1)
    else:
        print "Nagios installation found at", nagios_home
        if not (is_writable(nagios_home)):
            print "You do not have proper access to this directory %s. \
            Please run the script with root permissions.\nExiting now" % nagios_home
            sys.exit(1)


# Before extracting the package checking if the directory is writable or not.
if not (is_writable("/tmp")):
    print "You do not have proper access to the /tmp directory %s. \
    Please run the script with root permissions.\nExiting now" % current_file_path
    sys.exit(1)
else:
    execute_unix_cmd("rm -rf %s" % temp_location)
    execute_unix_cmd("mkdir -p %s" % temp_location)

name_list = pkg_filename.split("-")
extracted_name = name_list[0] + "_" + name_list[1] + "_" + name_list[2]
install_type = name_list[1]
to_install_plugin_ver = name_list[3].replace(".tar.gz", "")
uninstall_filename = os.path.join(nagios_home, ".cisco_"+install_type +
                                  "_uninstall.txt")

if options.uninstall:
    uninstall(uninstall_filename)
else:
    if os.path.isfile(uninstall_filename):
        try:
            uninstall_dict = read_properties(uninstall_filename)
        except Exception, err:
            print "Error while trying to read uninstall configuration file \
            '%s' ." % uninstall_filename
            print str(err)
            sys.exit(1)
        input_ans = 0
        try:
            existing_plugin_ver = uninstall_dict['PLUGIN_VERSION']
            print "An existing Cisco %s Nagios plugin version %s found." \
                % (install_type.upper(), existing_plugin_ver)
            if to_install_plugin_ver == existing_plugin_ver:
                input_ans = get_user_input('\nDo you want to continue with the same '
                                           'plugin version %s (Y/N) ? '
                                           % existing_plugin_ver, "N")
            elif to_install_plugin_ver < existing_plugin_ver:
                print "You are trying to install an older version of the plugin \n"
                print "Your current plugin version is : ", existing_plugin_ver
                print "You are trying to install an older version : ", to_install_plugin_ver
                input_ans = get_user_input("\nAre you sure you want to continue with \
                    the installation of older version (Y/N) ? ", "N")
        except Exception:
            print "\nError while trying to read uninstall configuration file for version info." 
            print "Removing the file '%s'." % uninstall_filename
        
        if input_ans:
            print "Exiting now."
            sys.exit(1)

        execute_unix_cmd("rm -f %s" % uninstall_filename)

# find the version check for PythonSDK
base_type_name = word_to_upper(install_type)
sdk_name = base_type_name+"Sdk"
try:
    i = __import__(sdk_name)
    ver = i.__version__
except:
    print 'This Nagios plugin requires %s Python SDK to be installed before we can proceed further.\n'\
        'Download the latest Cisco %s Python SDK from the following URL \n'\
        'https://communities.cisco.com/docs/DOC-37174' % (base_type_name, base_type_name)
    print "Aborting the installation"
    sys.exit(1)
print "Found Python %s Sdk version : %s " % (install_type.upper(), ver)
if install_type.strip() == 'imc':
    # check the version of SDK should be larger than 0.7.1
    if ver < '0.7.1':
        print 'This Nagios plugin does not support IMC Python SDK version below 0.7.1.\n'\
           'Download the latest Cisco IMC Python SDK from the following URL \n'\
           'https://communities.cisco.com/docs/DOC-37174'
        print "Aborting the installation"
        sys.exit(1)
else:
    if ver < '0.8.3':
        print 'This Nagios plugin does not support UCSM Python SDK version below 0.8.3.\n'\
            'Download the latest Cisco UCSM Python SDK from the following URL \n'\
            'https://communities.cisco.com/docs/DOC-37174'
        print "Aborting the installation"
        sys.exit(1)

print "\nExtracting the base package... "
if not extract_package(package_file, temp_location):
    print "\nFailed to extract base package %s.\nExiting now" % package_file
    sys.exit(1)

# Copy the plugin to its location and verify its execute permissions
# A valid plugin path will be mentioned in resource.cfg file present @ nagios_home
resource_path_line = get_data_from_file(nagios_cfg_file, "resource_file")
plugin_path = None
if resource_path_line is not None:
    resource_cfg_file = resource_path_line.split("=")[1].strip()
    if not os.path.isfile(resource_cfg_file):
        print "\nFailed to find the Nagios Resource config file %s." % resource_cfg_file
    else:
        plugin_path = get_data_from_file(resource_cfg_file, "USER1")
        if plugin_path is not None:
            plugin_path = plugin_path.split("=")[1].strip()

if plugin_path is None:
    print "\nPath to plugin directory not found "

    plugin_path = raw_input("\nProvide the absolute path to the plugin directory : ")
    plugin_path = plugin_path.strip()
    if plugin_path == "" or plugin_path is None:
        print "Did not provided any Plugin path.\nExiting now."
        sys.exit(1)
    if not os.path.isdir(plugin_path):
        print "The Plugin path provided is not present.\nExiting now."
        sys.exit(1)
    if not is_writable(plugin_path):
        print "\nYou do not have proper access to the provided path. \
            Please run the script with root permissions."
        sys.exit(1)
else:
    plugin_path_in = raw_input('Provide the absolute path to the plugin directory'
                               ' [Default Path \'%s\'] : ' % plugin_path)
    if not plugin_path_in.strip() == "":
        plugin_path = plugin_path_in
    if not os.path.isdir(plugin_path):
        print "The plugin path provided does not exist.\nExiting now."
        sys.exit(1)
    if not is_writable(plugin_path):
        print "The path '%s' is not writable. Please run installer with root permissions.\
            \nExiting now" % plugin_path
        sys.exit(1)

check_plugin_cfg_file = os.path.join(plugin_path, "cisco_%s_nagios.cfg" % install_type)
if os.path.isfile(check_plugin_cfg_file):
    answer = get_user_input("Do you want to backup already present cisco_%s_nagios.cfg file.(Y/N) ?" \
                            % install_type, "Y")
    if not answer:
        print "Your file %s will be overwritten" % check_plugin_cfg_file
    else:
        backup_orig = check_plugin_cfg_file + "_bkup"
        print "Backing up the original file as : ", backup_orig
        execute_unix_cmd("cp %s %s" % (check_plugin_cfg_file, backup_orig))

print "Extracting the monitoring plugin package ..."

plugin_temp_path = os.path.join(temp_location, extracted_name)
monitoring_package_tar = os.path.join(plugin_temp_path, "cisco_nagios_monitoring_plugin.tar.gz")
monitoring_package = os.path.join(plugin_temp_path, "cisco_nagios_monitoring_plugin/*")
if not extract_package(monitoring_package_tar, plugin_temp_path):
    print "Failed to extract Monitoring plugin package.\nExiting now."
    sys.exit(1)

result = os.system("cp -r %s %s" % (monitoring_package, plugin_path))
if result != 0:
    print "Unable to copy the monitoring plugin to Nagios plugin path.\nExiting now."
    sys.exit(1)
else:
    plugin_absolute_path = os.path.join(plugin_path, extracted_name)
    if os.access(plugin_absolute_path, os.X_OK):
        pass
    else:
        execute_unix_cmd("chmod ugo+x %s" % plugin_absolute_path)
write_to_uninstall("PLUGIN_PATH = "+plugin_absolute_path)
write_to_uninstall("PLUGIN_CFG_PATH =" + plugin_absolute_path+".cfg")

print "Searching for Nagios executable service file..."
nagios_restart_cmd_path = get_install_path(nagios_restart_install_paths)
if nagios_restart_cmd_path is None or nagios_restart_cmd_path == "":
    print "\nFailed to find the Nagios executable service file.\nExiting now."
    sys.exit(1)
if not os.path.isfile(nagios_restart_cmd_path):
    print "\nFailed to find the Nagios executable service file %s.\
        \nExiting now." % nagios_restart_cmd_path
    sys.exit(1)

write_to_uninstall("NAGIOS_CMD="+nagios_restart_cmd_path)

print "\nPlugin copied in the Nagios plugin directory and is ready to be used."

# Check if user wants to continue with autodiscovery install.
if only_plugin_flag:
    write_to_uninstall("PLUGIN_VERSION=" + to_install_plugin_ver)
    print "Removing the temporary files crated by the installer"
    execute_unix_cmd("rm -rf %s" % temp_location)
    sys.exit(0)

# Ask user where do we need to put autodiscovery file

autodiscovery_path = raw_input("\nProvide the path for saving \
autodiscovery files. Default Path [ %s ] :" % current_file_path)
if autodiscovery_path == "" or autodiscovery_path is None:
    autodiscovery_path = current_file_path
if not os.path.isdir(autodiscovery_path):
    print "\nThe directory path '%s' provided does not exists.\n \
    Creating this path now." % autodiscovery_path
    execute_unix_cmd("mkdir -p %s" % autodiscovery_path)
if not is_writable(autodiscovery_path):
    print "\nYou do not have proper access to the directory %s.\
        Please run the script with sudo user.\nExiting now." \
        % autodiscovery_path
    sys.exit(1)

check_autodiscovery_cfg_file = os.path.join(autodiscovery_path,
                                            "autodiscovery/NagiosAutoDiscovery%s.cfg" % install_type.upper())
check_autodiscovery_csv_file = os.path.join(autodiscovery_path,
                                            "autodiscovery/%sHostInfo.csv" % install_type.upper())

if os.path.isfile(check_autodiscovery_cfg_file):
    answer = get_user_input("Do you want to backup already present NagiosAutoDiscovery%s.cfg file.(Y/N) ? " \
                            % install_type.upper(), "Y")
    if not answer:
        print "Your file %s will be overwritten" % check_autodiscovery_cfg_file
    else:
        backup_orig = check_autodiscovery_cfg_file + "_bkup"
        print "Backing up the original file as : ", backup_orig
        execute_unix_cmd("cp %s %s" % (check_autodiscovery_cfg_file, backup_orig))

if os.path.isfile(check_autodiscovery_csv_file):
    answer = get_user_input("Do you want to backup already present %sHostInfo.csv file.(Y/N) ? " \
                            % install_type.upper(), "Y")
    if not answer:
        print "Your file %s will be overwritten" % check_autodiscovery_csv_file
    else:
        backup_orig = check_autodiscovery_csv_file + "_bkup"
        print "Backing up the original file as : ", backup_orig
        execute_unix_cmd("cp %s %s" % (check_autodiscovery_csv_file, backup_orig))

print "\nExtracting the Auto-Discovery package...."
autodiscovery_package_tar = os.path.join(plugin_temp_path, "cisco_nagios_autodiscovery_addon.tar.gz")
if not extract_package(autodiscovery_package_tar, autodiscovery_path):
    print "Failed to extract Auto-Discovery package. \nExiting now."
    sys.exit(1)


try:
    contact_group = raw_input('Provide the contact group name for the services and hosts'
                               ' [Default : "admins"] : ')
    if  contact_group == "" or contact_group is None or contact_group.strip() == "":
        print "\nAs no input was provided taking 'admins' as default contact group."
        contact_group = "admins"
    else:
        print "Will be using '%s' as the contact group for all the services and hosts found via autodiscovery" %contact_group
except Exception:
            print "Your input is invalid. Taking 'admins' as default contact group"
            contact_group = "admins"
            
print "\nMaking the required changes in the nagios auto discovery configuration file."

# Now update the autodiscovery files according to user input.
autodiscovery_config_path = os.path.join(autodiscovery_path,
                                         "autodiscovery/NagiosAutoDiscovery" +
                                         str(install_type).upper() + ".cfg")
if not os.path.isfile(autodiscovery_config_path):
    print "\nFailed to find the Auto-Discovery Config file %s.\
        \nExiting now." % autodiscovery_config_path
    sys.exit(1)
if edit_file(autodiscovery_config_path, "NAGIOS_HOME=",
             "NAGIOS_HOME=%s" % nagios_home):
    print "\nFailed to update the %s file .\nExiting now." \
        % autodiscovery_config_path
    sys.exit(1)
else:
    print "Updated %s file.\nUpdated NAGIOS_HOME=%s"\
        % (autodiscovery_config_path, nagios_home)

if edit_file(autodiscovery_config_path, "NAGIOS_RESTART_CMD=",
             "NAGIOS_RESTART_CMD=%s restart" % nagios_restart_cmd_path):
    print "\nFailed to update the %s file . \nExiting now." \
        % autodiscovery_config_path
    sys.exit(1)
else:
    print "Updated %s file. \nUpdated NAGIOS_RESTART_CMD=%s restart" \
        % (autodiscovery_config_path, nagios_restart_cmd_path)


# Edit cgi.cfg file.
cgi_cfg_file = os.path.join(nagios_home, "cgi.cfg")

if not os.path.isfile(cgi_cfg_file):
    print "\nFailed to find the Nagios config file %s.\nExiting now." \
        % cgi_cfg_file
    sys.exit(1)
if edit_file(cgi_cfg_file, "escape_html_tags", "escape_html_tags=0"):
    print "\nFailed to update the %s file.\nExiting now." % cgi_cfg_file
    sys.exit(1)
else:
    print "Updated %s file.\nChanged 'escape_html_tags' attribute from 1 to 0"\
        % cgi_cfg_file

# Edit nagios.cfg file
nagios_cfg_file = os.path.join(nagios_home, "nagios.cfg")

if not os.path.isfile(nagios_cfg_file):
    print "\nFailed to find the Nagios config file %s.\nExiting now." \
        % nagios_cfg_file
    sys.exit(1)
if edit_file(nagios_cfg_file, "cfg_dir=%s/cisco/%sObjs" % (nagios_home,
                                                           install_type)):
    print "\nFailed to update the %s file.\nExiting now." % nagios_cfg_file
    sys.exit(1)
else:
    print "\nUpdated %s file.\nAdded cfg_dir=%s/cisco/%sObjs" \
        % (nagios_cfg_file, nagios_home, install_type)
        
execute_unix_cmd("mkdir -p %s" % os.path.join(nagios_home,"cisco/%sObjs" %install_type))
# Now get the logos directory path from cgi.cfg file.
logo_base_path = get_data_from_file(cgi_cfg_file,"physical_html_path")
if logo_base_path is not None:
    logo_base_path = logo_base_path.split("=")[1].strip() + "/images/logos/"
    autodiscovery_logo_path = os.path.join(autodiscovery_path,
                                       "autodiscovery/payload/logos/*")
    result = os.system("cp %s %s" % (autodiscovery_logo_path, logo_base_path))
    if result != 0:
        print "\nUnable to copy the Logos to Nagios logos directory.\nExiting now."
        sys.exit(1)
    else:
        print "\nCopied logos from %s to %s" % (autodiscovery_logo_path,
                                                logo_base_path)
else:
    print "Unable to find the physical html path in the provided cgi configuration file."

autodiscovery_payload_service_path = os.path.join(autodiscovery_path,"autodiscovery/payload/cisco_%s_service.cfg" %install_type)
autodiscovery_payload_host_path = os.path.join(autodiscovery_path,"autodiscovery/payload/cisco_%s_host.cfg" %install_type)
if edit_file(autodiscovery_payload_host_path, "contact_groups", "contact_groups     %s" %contact_group):
    print "\nFailed to update the %s file.\nExiting now." % autodiscovery_payload_host_path
    sys.exit(1)
else:
    print "Updated %s file.\nUpdated 'contact_groups     %s'\n " %(autodiscovery_payload_host_path,contact_group)

if edit_file(autodiscovery_payload_service_path, "contact_groups", "contact_groups     %s" %contact_group):
    print "\nFailed to update the %s file.\nExiting now." % autodiscovery_payload_service_path
    sys.exit(1)
else:
    print "\nUpdated %s file.\nUpdated 'contact_groups     %s' \n" %(autodiscovery_payload_service_path,contact_group)


print "\nYou can access Auto Discovery from %s/autodiscovery \n" % autodiscovery_path
write_to_uninstall("LOGOS_PATH="+logo_base_path)
write_to_uninstall("AUTO_PATH="+autodiscovery_path+"/autodiscovery/")
write_to_uninstall("PLUGIN_VERSION=" + to_install_plugin_ver)

# Now remove the temp files created by installer
print "Removing the temporary files created by the installer"
execute_unix_cmd("rm -rf %s" % temp_location)
