#!/usr/bin/perl -w

use warnings;
use strict;

# Debugging
our $DEBUG = 0;
our $PRESERVE = 0;  # 0|1 if the testenv should be deleted at the end or kept

use Test::More tests => 25 + 1; # 'no_plan'; 
use Test::Cmd;
use Test::Output;
use Test::NoWarnings;
use Test::LongString max => 800;
use Readonly;

# Constants
Readonly our $EMPTY => q{};
Readonly our $BLANK => q{ };

# Constants for Testresults
Readonly our $PASS  => 0;
Readonly our $FAIL  => 2;

# Nagios Status Codes
Readonly our $OK     => 0;
Readonly our $WARN   => 1;
Readonly our $CRIT   => 2;

our $PROG_NAME = 'wrap_multi.pl'; # Name of the script we want to test
our $PROG = "bin/$PROG_NAME";   
our $WORKDIR = 'temporary_test_env';

# HINT: should be commented-out before final release ($PRESERVE  must be set to 0 before!)
#system("rm -rf /Users/ingolantschner/Perl/Projekte/multi-label-wrapper/wrap-multi/$WORKDIR");
my $test = Test::Cmd->new(
    prog => "$PROG", 
    workdir => "$WORKDIR", 
    verbose => $DEBUG
);
ok($test, "create Test::Cmd object. Note: If failed check if workdir ($WORKDIR) already exists.");
if ($PRESERVE) {$test->preserve()}


# =============================
# = Definition of test-cases  =
# =============================

=pod

=head1 TESTS for WRAP_MULTI.PL

There 2 test whoch use the standard Nagios::Plugin Module to output and 3 tests 
which print exactly what the Nagios-docs say. In the latter there are some 
logical errors but they are taken from the source (and they will not hurt in
thi context).

The wrapper-script should work fine with alle 5 cases and pass these tests.

=cut

my $cases; # Ref for Testcases-Definitions
my $id;

# Simple check: single instance and single-line
$id = 'check_stuff';
$cases->{'stuff'}{'check_name'} = prepare_check_stuff("$id");
$cases->{'stuff'}{'label'} = 'label';
$cases->{'stuff'}{'prefix'} = $EMPTY;
$cases->{'stuff'}{'result'} = $OK;
$cases->{'stuff'}{'stdout'} =   
    qq{STUFF OK - stuff checked | ${id}::multi_label::instances=1} . $BLANK
  . q{stuff::check_stuff::label=10GB;15;18;0;20} . "\n";

# Pseudo-check_disk with multi-line and multi-instances
$id = 'check_disk';
$cases->{'disk'}{'check_name'} = prepare_check_disk("$id");
$cases->{'disk'}{'label'} = 'free_space';
$cases->{'disk'}{'prefix'} = q{fs-};
$cases->{'disk'}{'result'} = $OK;
$cases->{'disk'}{'stdout'} =   
    qq{DISK OK - free space: / 3326 MB (56%); | ${id}::multi_label::instances=4} . "\n"
  . q{/ 15272 MB (77%);} . "\n"
  . q{/boot 68 MB (69%);} . "\n"
  . q{/home 69357 MB (27%);} . "\n"
  . q{/var/log 819 MB (84%); | }
  . q{fs-/::check_disk::free_space=15272MB;80;95;0;5958}        . $BLANK
  . q{fs-/boot::check_disk::free_space=68MB;80;95;0;5958}      . $BLANK
  . q{fs-/home::check_disk::free_space=69357MB;80;95;0;5958}   . $BLANK
  . q{fs-/var/log::check_disk::free_space=818MB;80;95;0;5958}  . "\n";

# Cases described in Nagios.org-dok
# http://nagios.sourceforge.net/docs/3_0/pluginapi.html

# Case 1: One line of output (text only)
$id = 'print_case1';
$cases->{'case1'}{'check_name'} = prepare_case1("$id");
$cases->{'case1'}{'label'} = 'case_1';
$cases->{'case1'}{'prefix'} = $EMPTY;
$cases->{'case1'}{'result'} = $OK;
$cases->{'case1'}{'stdout'} = q{DISK OK - free space: / 3326 MB (56%);}
   . qq{ | ${id}::multi_label::instances=0} . "\n";

# Case 2: One line of output (text and perfdata)
$id = 'print_case2';
$cases->{'case2'}{'check_name'} = prepare_case2("$id");
$cases->{'case2'}{'label'} = 'some_space';
$cases->{'case2'}{'prefix'} = $EMPTY;
$cases->{'case2'}{'result'} = $OK;
$cases->{'case2'}{'stdout'} = 
    q{DISK OK - free space: / 3326 MB (56%); | }
    . qq{${id}::multi_label::instances=1} . $BLANK 
    . qq{/::${id}::some_space=2643MB;5948;5958;0;5968} . "\n";
# No newlines have ben added! (Still single-line)

# Case 3: Multiple lines of output (text and perfdata)
$id = 'print_case3';
$cases->{'case3'}{'check_name'} = prepare_case3("$id");
$cases->{'case3'}{'label'} = 'some_space';
$cases->{'case3'}{'prefix'} = $EMPTY;
$cases->{'case3'}{'result'} = $OK;
$cases->{'case3'}{'stdout'} = 
    q{DISK OK - free space: / 3326 MB (56%); |} . $BLANK
    . qq{${id}::multi_label::instances=4} . $BLANK
    . qq{/::${id}::some_space=2643MB;5948;5958;0;5968} . "\n"
    . q{/ 15272 MB (77%);} . "\n"  
    . q{/boot 68 MB (69%);} . "\n"
    . q{/home 69357 MB (27%);} . "\n"
    . q{/var/log 819 MB (84%); |} . $BLANK 
    . qq{/boot::${id}::some_space=68MB;88;93;0;98} . "\n" . $BLANK
    . qq{/home::${id}::some_space=69357MB;253404;253409;0;253414} . $BLANK
    . qq{/var/log::${id}::some_space=818MB;970;975;0;980} . "\n"
    ;
# Newlines have been added! (as according to the nagios.org-docs)

# Bug 2010-04-21 reported by MP/Bacher
# check_disk seems to write anything into one line (which is NOT the problem)
# DISK CRITICAL - free space: / 5469 MB (35% inode=92%); /dev 3991 MB (99% inode=99%); /opt/openitc 324567 MB (84% inode=99%); /var/lib/mysql 374046 MB (96% inode=99%);| /=9840MB;3225;1612;0;16128 /dev=0MB;798;399;0;3991 /opt/openitc=58006MB;80609;40304;0;403047 /var/lib/mysql=13292MB;81613;40806;0;408068

# Problem was, that check_disk does not have an extension so the 
# filename-parsing failed

# This test creates a script with no extension ( see sub prepare_bug2104() )
$id = 'check_bug2104';
$cases->{"$id"}{'check_name'} = prepare_bug2104("$id");
$cases->{"$id"}{'label'} = 'used';
$cases->{"$id"}{'prefix'} = 'FS-';
$cases->{"$id"}{'result'} = $CRIT;
$cases->{"$id"}{'stdout'} = 
    q{DISK CRITICAL - free space: / 5469 MB (35% inode=92%); /dev 3991 MB (99% inode=99%); /opt/openitc 324567 MB (84% inode=99%); /var/lib/mysql 374046 MB (96% inode=99%);}
    .  q{ |}
    .  q{ check_bug2104::multi_label::instances=4}
    . qq{ FS-/::${id}::used=9840MB;3225;1612;0;16128}
    . qq{ FS-/dev::${id}::used=0MB;798;399;0;3991}
    . qq{ FS-/opt/openitc::${id}::used=58006MB;80609;40304;0;403047}
    . qq{ FS-/var/lib/mysql::${id}::used=13292MB;81613;40806;0;408068}
    . "\n"
    ;

# Silly Path (dots and double dots for extension ('path.with.dots/script..pl'))
$id = 'silly_path';
$cases->{"$id"}{'check_name'} = prepare_silly_path("$id");
$cases->{"$id"}{'label'} = 'used';
$cases->{"$id"}{'prefix'} = 'FS-';
$cases->{"$id"}{'result'} = $OK;
$cases->{"$id"}{'stdout'} = 
    q{SILLY PATH OK - nothing found}
    .  q{ |}
    .  q{ silly_path::multi_label::instances=1}
    . qq{ FS-p1::${id}::used=100;0;0;1}
    . "\n"
    ;

# check without ext but with path in front (/some/path/check)
$id = 'path_but_no_ext';
$cases->{"$id"}{'check_name'} = prepare_path_but_no_ext("$id");
$cases->{"$id"}{'label'} = 'used';
$cases->{"$id"}{'prefix'} = 'FS-';
$cases->{"$id"}{'result'} = $OK;
$cases->{"$id"}{'stdout'} = 
    q{NO EXT OK - nothing found}
    .  q{ |}
    .  q{ path_but_no_ext::multi_label::instances=1}
    . qq{ FS-p1::${id}::used=100;0;0;1}
    . "\n"
    ;



# ========
# = Test =
# ========

foreach my $case (keys %{$cases}) {
    my $script_name = $cases->{$case}{'check_name'};
    my $label = $cases->{$case}{'label'};
    my $prefix = $cases->{$case}{'prefix'};
    my $expected_exit = $cases->{$case}{'result'};
    my $expected_stdout = $cases->{$case}{'stdout'};
    my $exit = $test->run(args => qq{'$script_name' '--arg=pseudo' $label $prefix} );
    is($exit >> 8 , $expected_exit, "$case: " . 'returnvalue es expected');
    is($test->stderr, $EMPTY, "$case: " . 'no output to stderr');
    is_string($test->stdout, "$expected_stdout", "$case: " . 'stdout as expected');
}

sub prepare_check_disk {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

# ================================================
# = This is is a pseudo-check just for testing!! =
# ================================================

use Nagios::Plugin;

my $p = Nagios::Plugin->new(
    usage => 'Just for testing',
    version => '1.0',
    url => 'http://netapp-monitoring.info/',
);

$p->set_thresholds(
  warning => 80,
  critical => 95
);

$p->add_perfdata(
     label   => '/',
     value   => 15272,
     uom     => 'MB',
     threshold   => $p->threshold(),
     min     =>  0,
     max     => 5958,
);
$p->add_perfdata(
     label   => '/boot',
     value   => 68,
     uom     => 'MB',
     threshold   => $p->threshold(),
     min     =>  0,
     max     => 5958,
);
$p->add_perfdata(
     label   => '/home',
     value   => 69357,
     uom     => 'MB',
     threshold   => $p->threshold(),
     min     =>  0,
     max     => 5958,
);
$p->add_perfdata(
     label   => '/var/log',
     value   => 818,
     uom     => 'MB',
     threshold   => $p->threshold(),
     min     =>  0,
     max     => 5958,
);

$p->nagios_exit(
     return_code => 0,
     message => 'free space: / 3326 MB (56%); ' . "\n"
                . q{/ 15272 MB (77%);}     . "\n"
                . q{/boot 68 MB (69%);}    . "\n"
                . q{/home 69357 MB (27%);} . "\n"
                . q{/var/log 819 MB (84%);}
                 ,
);


END_HERE


    my $fn = $test->workpath("$name" . q{.pl});
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get chenged its permissions'};
    return $fn;
}

sub prepare_check_stuff {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

use Nagios::Plugin;

my $p = Nagios::Plugin->new(
    usage => 'Just for testing',
    version => '1.0',
    url => 'http://netapp-monitoring.info/',
);

$p->set_thresholds(
  warning => 15,
  critical => 18
);

$p->add_perfdata(
     label   => 'stuff',
     value   => 10,
     uom     => 'GB',
     threshold   => $p->threshold(),
     min     =>  0,
     max     => 20,
);

print STDERR 'harharr - this should be discarded' . "\n";

$p->nagios_exit(
     return_code => 0,
     message => 'stuff checked',
);

END_HERE


    my $fn = $test->workpath("$name" . q{.pl});
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get chenged its permissions'};
    return $fn;
}

sub prepare_case1 {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

print STDERR 'harharr - this should be discarded' . "\n";

print q{DISK OK - free space: / 3326 MB (56%);};
print "\n";

exit 0;

END_HERE

    my $fn = $test->workpath("$name" . q{.pl});
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get changed its permissions'};
    return $fn;
}

sub prepare_case2 {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

print STDERR 'harharr - this should be discarded' . "\n";

print q{DISK OK - free space: / 3326 MB (56%); | /=2643MB;5948;5958;0;5968};
print "\n";

exit 0;

END_HERE

    my $fn = $test->workpath("$name" . q{.pl});
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get changed its permissions'};
    return $fn;
}

sub prepare_case3 {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

print STDERR 'harharr - this should be discarded' . "\n";

print q{DISK OK - free space: / 3326 MB (56%); | /=2643MB;5948;5958;0;5968
/ 15272 MB (77%);
/boot 68 MB (69%);
/home 69357 MB (27%);
/var/log 819 MB (84%); | /boot=68MB;88;93;0;98
/home=69357MB;253404;253409;0;253414 
/var/log=818MB;970;975;0;980
};

print "\n";

exit 0;

END_HERE

    my $fn = $test->workpath("$name" . q{.pl});
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get changed its permissions'};
    return $fn;
}

sub prepare_bug2104 {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

print q{DISK CRITICAL - free space: / 5469 MB (35% inode=92%); /dev 3991 MB (99% inode=99%); /opt/openitc 324567 MB (84% inode=99%); /var/lib/mysql 374046 MB (96% inode=99%);| /=9840MB;3225;1612;0;16128 /dev=0MB;798;399;0;3991 /opt/openitc=58006MB;80609;40304;0;403047 /var/lib/mysql=13292MB;81613;40806;0;408068};

print "\n";

exit 2;

END_HERE

    my $fn = $test->workpath("$name");
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get changed its permissions'};
    return $fn;
}

sub prepare_silly_path {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

print q{SILLY PATH OK - nothing found | p1=100;0;0;1};

print "\n";

exit 0;

END_HERE

    $test->subdir('path.with dots and.blanks');
    my $fn = $test->workpath("path.with dots and.blanks/$name..pl");
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get changed its permissions'};
    return $fn;
}

sub prepare_path_but_no_ext {
    my $name = shift;
(my $file_content = << 'END_HERE');
#!/usr/bin/perl -w
use warnings;
use strict;

print q{NO EXT OK - nothing found | p1=100;0;0;1};

print "\n";

exit 0;

END_HERE

    $test->subdir('path_to_script');
    my $fn = $test->workpath("path_to_script/$name");
    
    open my $fh, ">", $fn 
        or die "Cant open $fn for writing: $!\n";
    print $fh $file_content;
    close $fh or die "Can not close $fn: $!\n";
    my $cnt = chmod 0755,"$fn";
    if ($cnt != 1) { die 'Only one file should get changed its permissions'};
    return $fn;
}
__END__
=pod

=head1 COPYRIGHT and LICENSE

Same as the tested perl-script.


=cut

