#
# gothix::area - simple curses-based windows
# (too simple to be named windows, hence areas..)
#
# Copyright Stig H. Jacobsen / Gothix 2004-2005
#
# TODO
# ----
#  - support for titles (in window frame)
#
#
#


# --- POD prologue --------------------------------------------------------

=pod 

=head1 gothix::area

gothix::area - simple Curses-based windows.

=head1 SYNOPSIS

   use gothix::xcurses;
   use gothix::area;

   curses_init();
   $a = new gothix::area(name=>'headers', x=>0, y=>2, width=>$COLS, height=>1);
   $a->print(0, 0, "Key  Data   Comment");
   term_refresh();
   read_key();
   curses_shutdown();


=head1 DESCRIPTION

Features:

=over 4

=item Rectangular sections of screen space

=item No title, no word wrap, no mouse support, no ...

=back


=head1 Exported symbols

None.


=cut


# === gothix::area ========================================================

package gothix::area;

use strict;
use warnings;
use Exporter;
use Carp;
use Curses;

eval "use Data::Dumper;";     #debug
if ($@) {
   sub Dumper {
      return "dumper n/a";
   }
}

use gothix::oobase;
use gothix::xcurses;

our @ISA = qw( Exporter gothix::oobase );
our @EXPORT = qw( &area_default_options &resizer );


# --- Globals -------------------------------------------------------------

our $rootwin = undef;

my %All_Areas = ();

my @areaOptions = (
   'name', 'x', 'y', 'width', 'height', 'frame', 'attr', 'parent', 'activate', 'attr_title',
   'right_justify');

my %DefaultOptions = (
   'attr'         => A_NORMAL,
   'attr_title'   => A_BOLD,
);

my $dummy = 1; #module truth return


# --- Construction --------------------------------------------------------

=pod

=head1 Constructor and initialization

This will create a one-line/half-line area in the upper-right corner of the
screen:

   my $area = new nsc::area(name => 'clock', x => $COLS/2, y => 0, width => $COLS/2, height => 1);

Parameters - most have defaults:

=over 4

=item C<name> 

Used to name the area - mostly for debugging purposes.


=item C<x> 

Absolute X-position of the left side of the area.


=item C<y> 

Absolute Y-position of the top side of the area. If this is 
negative, then it is relative to the bottom of the screen, so that -1 will be
the last line, -2 is the second-last line, etc.


=item C<width> 

The width of the area in characters - i.e. the number of
characters that can be written horizontally within the area.

=item C<height>

Height of the area - if this is negative, then leave C<abs(height)> lines free
at the bottom of the screen. I.e. use -1 to use rest of screen, except for the
last screen line.

=item C<attr>

This is optional and is the default attribute when clearing and
printing to the area. Inherited from parent, if not supplied.

=item C<frame>

This is optional - if set and true, this area will have a frame drawn around
it. If bit 1 (& 0x02) is set, the windows name will be used as title.

=item C<parent>

This is optional - if set, then the new area will be a sub-area of 'parent'.
If this argument isn't supplied, the root area is used as the parent.

=back


=cut


sub new {

   my $class = shift;
   my $this = {};
   bless $this, $class;
   my %args = @_;

   #can't be done at module initialization, since curses isn't up ($COLS, $LINES not defined)
   if (!defined( $rootwin )) {
      $rootwin = new gothix::area::rootwin();
   }

   $this->debug_init( defined($ENV{AREA_DEBUG}) );

   foreach my $param (@areaOptions) {
      if (defined($args{$param})) {
         my $val = $args{$param};
         $this->d("got '$val' .. ");
         confess "$this->{name}: Param $param is not defined" unless defined($val);
         $this->{$param} = $val;
         $this->d("set $param to '$this->{$param}'");
      } elsif (defined($DefaultOptions{$param})) {
         my $val = $DefaultOptions{$param};
         $this->{$param} = $val;
         $this->d("set $param to default value '$this->{$param}'");
      }
   }

   $this->{activate} = 1;
   $this->{frame_width} = 0;
   $this->{frame_width} = 1 if $this->{frame};

   # --- apply defaults ---

   $this->{orig_height} = $this->{height};
   $this->{orig_width} = $this->{width};
   $this->{orig_y} = $this->{y};

   $this->{parent} = $rootwin unless $this->{parent};
   $this->{x} = 0 unless defined($this->{x});
   $this->{y} = 0 unless defined($this->{y});
   $this->{width} = $this->{parent}->{width} - $this->{x} unless $this->{width};

   $this->{'attr'} = $this->{parent}->{attr}
      unless $this->{attr};

   # --- params done with ---

   $this->set_size();

   $All_Areas{$this} = $this;

   $this->{open} = 1;
   $this->{drawn} = 0;

   if ($this->{activate}) {
      $this->clr();
      $this->activate();
      $this->d("created and activated");
   } else {
      $this->d("created, not activated");
   }

   # --- all done ---

   if ($this->{debug}) {
      $this->dump_object();
      $this->print(0, 0, "new():$this->{name}");
   }

   return $this;

} #new


# --- Destroyers ----------------------------------------------------------

sub DESTROY {

   my $this = shift;
   $this->d("DESTROY");
   $this->close();

} #DESTROY


sub close {

   my $this = shift;
   return unless $this->{open};
   $this->d("close");
   $this->{open} = 0;

} #close


# --- Methods -------------------------------------------------------------

=pod

=head1 Class and object methods

(what can be done with/to the object)

=cut


# --- assert()-------------------------------------------------------------



# --- activate() ----------------------------------------------------------
# Sets the area as active (front, on-top area)
# Can be called multiple times, does not disrupt content...

sub activate {

   my $this = shift;

   $this->d("activate");

   $this->draw_frame()
      if ($this->{frame});
   $this->{drawn} = 1;

} #activate


# --- set_size() ----------------------------------------------------------

sub set_size {

   my $this = shift;

   if ($this->debug()) {
      my $termsize = sprintf('l=%d,c=%d', term_lines(), term_cols());
      $this->d("set_size($termsize): now lines=$this->{height} cols=$this->{width} @ $this->{x},$this->{y}\n");
   }

   if ($this->{orig_y} < 0) {    #relative to bottom of screen?
      $this->{y} = $this->{parent}->{height} - -$this->{orig_y};
      $this->d("set_size($this->{name}: defined y=$this->{y}\n");
      #print STDERR "$this->{name}: defined y=$this->{y}\n";
   }
   if (defined($this->{orig_height}) && $this->{orig_height} < 0) {    #relative to bottom of parent?
      $this->{height} = $this->{parent}->{height} - $this->{y} - -$this->{orig_height};
      #print STDERR "$this->{name}: defined/1 height=$this->{height}\n";
   } elsif (! defined($this->{height}) ) {
      $this->{height} = $this->{parent}->{height} - $this->{y};
      #print STDERR "$this->{name}: defined/2 height=$this->{height}\n";
   }

   if (defined($this->{orig_width})) {
      my $maxw = $this->{parent}->{width} - $this->{x};
      if (($this->{width} = $this->{orig_width}) > $maxw) {
         $this->{width} = $maxw;
      }
   }

   $this->d("NOW y=$this->{y} height=$this->{height}");

   confess "$this->{name}: Area is too wide ($this->{width} @ $this->{x}"
      if (($this->{x} + $this->{width}) > $this->{parent}->{width});
   confess "$this->{name}: Area is not wide enough"
      if ($this->{width} < 1);
   confess "$this->{name}: Area is not wide enough to have space for frame"
      if ( ($this->{width} - (2*$this->{frame_width})) < 1);

   confess "$this->{name}: Area is too tall ($this->{height}), parent dump = " . Dumper($this->{parent})
      if (($this->{y} + $this->{height}) > $this->{parent}->{height});
   confess "$this->{name}: Area is not tall enough"
      if ($this->{height} < 1);
   confess "$this->{name}: Area is not tall enough at $this->{height} lines to have space for frame"
      if ( ($this->{height} - (2*$this->{frame_width})) < 1);

   # fix curx/cury as needed

   $this->{curx} = 0
      if ( !defined( $this->{curx} ) ||
           ( $this->{curx} < 0 ) ||
           ( $this->{curx} > $this->maxx() ) );

   $this->{cury} = 0
      if ( !defined( $this->{cury} ) ||
           ( $this->{cury} < 0 ) ||
           ( $this->{cury} > $this->maxy() ) );

   $this->d("set_size(): area size NOW lines=$this->{height} cols=$this->{width} @ $this->{x},$this->{y}\n");

} #set_size


# --- raw_gotoxy() --------------------------------------------------------

#internal use only, does NOT update curx/cury
sub raw_gotoxy {

   my($this,$x,$y) = @_;

   $this->lfatal("area::raw_gotoxy($x,$y): x ($x) is outside bounds")
      if ($x < 0 || $x >= $this->{width});
   $this->lfatal("area::raw_gotoxy($x,$y): y ($y) is outside bounds")
      if ($y < 0 || $y >= $this->{height});

   $this->{parent}->gotoxy($x + $this->{x}, $y + $this->{y});

} #raw_gotoxy


# --- gotoxy() ------------------------------------------------------------

sub gotoxy {

   my($this,$x,$y) = @_;

   $this->lfatal("area::gotoxy($x,$y): x ($x) is outside bounds")
      if ($x < 0 || $x > $this->maxx());
   $this->lfatal("area::gotoxy($x,$y): y ($y) is outside bounds, maxy=" . $this->maxy())
      if ($y < 0 || $y > $this->maxy());

   $this->{curx} = $x;
   $this->{cury} = $y;

   my $nx = $x + $this->{frame_width};
   my $ny = $y + $this->{frame_width};

   $this->raw_gotoxy($nx, $ny);

} #gotoxy


# --- print() -------------------------------------------------------------

=pod

=head2 print()

Prints string to area. Args: x, y, text, attr

x and y are within the area. attr is optional - the area default
attribute is used if absent.

If x is negative, then it is relative to the right side of the area
- i.e. print this right justified.

If text starts with '!', then the 'y' line in the area is cleared
before printing text (without the !).

Returns number of characters printed, "cat" = 3..

=cut


sub print {
      
   my ($this,$x, $y, $text, $attr) = @_;

   $attr = $this->{attr} unless $attr;

   # --- Checks ---

   defined($text) || confess "more args, please";

   if ($x > $this->maxx()) {
      $this->d("outside horizontal bound ($x), " . substr($text, 0, 10));
      return;
   }
   if ($y > $this->maxy()) {
      $this->d("outside vertical bound, " . substr($text, 0, 10));
      return;
   }

   # --- Text adjustment as required ---
   
   my $local_rjust = 0;
   if ($x < 0) {
      $local_rjust = 1;
      $x = -$x - 1;
   }

   if ($text =~ s/[\r\n]/ /g) {
      $this->d("removed cr or nl char(s): " . substr($text, 0, 20));
   }

   if (substr($text, 0, 1) eq '!') {
      $this->clr_line($y);
      $text = substr($text, 1);
   }

   my $maxwidth = $this->maxx() + 1 - $x;
   my $textlen = length($text);
   if ($textlen > $maxwidth) {
      $this->d("truncating to $maxwidth: '$text'");
      $text = substr($text, 0, $maxwidth);
      $textlen = $maxwidth;
   }

   # --- justify ---
   my $xoffs = 0;
   if ($local_rjust || $this->{right_justify}) {
      my $i = $this->maxx() + 1 - length($text);
      if ($i > 0) {
         # $text = (' ' x $i) . $text;       #spacing instead of positioning
         $xoffs = $i;
      }
   }

   # --- ready to print ... ---

   attron($attr);
   $this->gotoxy($x + $xoffs, $y);
   addstr($text);
   attroff($attr);

   $this->{curx} = ($this->{curx} > $this->maxx()) ?
                      $this->maxx() :
                         $x + $xoffs + $textlen;
   $this->{cury} = $y;

   return length($text);

} #print


# --- cprint() ------------------------------------------------------------

=pod

=head2 cprint()

Same thing as L<C<print()>> except that this clears the remainder
of the output line.

=cut

#TODO: this implementation sucks
sub cprint {

   my ($this, $x, $y, $text, @extra) = @_;

   return $this->print($x, $y, '!' . $text, @extra);

} #cprint


# --- raw_print() ---------------------------------------------------------
# internal print, uses raw_gotoxy

sub raw_print {
      
   my ($this,$x, $y, $text) = @_;

   confess("no text?") unless $text;

   $this->raw_gotoxy($x, $y);
   addstr($text);

} #raw_print


# --- ttyout() ------------------------------------------------------------
# Not *that* much tty over this, as it only handles \n's and wrapping...
# TODO: Doesn't scroll!!
sub ttyout {
      
   my ($this, $text, $attr) = @_;

   $attr = $this->{attr} unless $attr;

   while ((my $i = index($text, "\n")) >= 0) {
      my $line = substr($text, 0, $i);
      #print STDERR "x=$this->{curx} y=$this->{cury} line=$line\n";
      $i++;
      if ($i > length($text)) {
         $text = ''; #eos
      } else {
         $text = substr($text, $i);
      }
      my $s = $line;
      do {
         if ($s ne '') {
            my $os = $s;
            my $left = $this->maxx() + 1 - $this->{curx};
            if (length($s) > $left) {           #wrap needed?
               $os = substr($s, 0, $left);
               $s = substr($s, $left);
            } else {
               $s = ''; #end-of-string..
            }
            $this->print($this->{curx}, $this->{cury}, $os);
         }
         $this->{cury}++;
         $this->{curx} = 0;
         #TODO: scroll instead of wrap..
         $this->{cury} = 0 if ($this->{cury} > $this->maxy());
      } while ($s ne '');
   }
   $this->print($this->{curx}, $this->{cury}, $text, $attr) if ($text ne '');

} #ttyout


# --- clr_line()---------------------------------------------------------
# TODO: optimize, use native curses calls to clear stuff

sub clr_line {

   my ($this,$lineno) = @_;

   my $fill = $this->{debug} ? '_' : ' ';

   $this->print(0, $lineno,
                $fill x ($this->maxx() + 1),
                $this->{'attr'});
   $this->gotoxy(0,$lineno);

} #clr_line


# --- clr() -------------------------------------------------------------
# don't rename this to clear(), it conflicts...
sub clr {

   my ($this) = @_;

   for (my $line=0; $line <= $this->maxy(); $line++) {
      $this->d("clear! {y}=$this->{y}  maxy=%d   aline=%d", $this->maxy(), $line);
      $this->clr_line($line);
   }
   $this->gotoxy(0,0);

} #clr


# --- Return info about area size -----------------------------------------

# --- Columns ---

sub maxx {

   my ($this) = @_;

   return ($this->{width} - 1) - ($this->{frame_width} * 2);

} #maxx

sub cols() {
   my $this = shift;
   return $this->maxx() + 1;
} #cols


# --- Lines ---

sub maxy {

   my ($this) = @_;

   my $res = ($this->{height} - 1) - ($this->{frame_width} * 2);

   $this->d("maxy($this->{name}) returning $res"
#               . ', object = ' . Dumper($this)
           );

   return $res;

} #maxy

sub lines() {
   my $this = shift;
   return $this->maxy() + 1;
} #lines


# --- .. ------------------------------------------------------------------
# Not really for public consumption
# NOTE: Uses raw_gotoxy() for coordinates..
sub raw_hline {

   my ($this, $x, $y, $len) = @_;

   $this->raw_gotoxy($x, $y);
   hline(0, $len);

} #raw_hline

sub raw_vline {

   my ($this, $x, $y, $len) = @_;

   $this->raw_gotoxy($x, $y);
   vline(0, $len);

} #raw_vline


# --- draw_frame() --------------------------------------------------------

sub draw_frame {

   my($this) = @_;

   return unless $this->{frame};

   $this->d("draw_frame()");

   #print STDERR "draw frame,attr=", $this->{attr}, "\n";
   attron($this->{attr});

   my $lw = $this->{width} - (2*$this->{frame_width});
   $this->raw_hline(1, 0, $lw, $this->{attr});
   $this->raw_hline(1, $this->{height} - $this->{frame_width}, $lw);

   $this->raw_gotoxy(0, 0);
   addch(ACS_ULCORNER);
   $this->raw_gotoxy($this->{width}-$this->{frame_width}, 0);
   addch(ACS_URCORNER);

   $lw = $this->{height} - (2*$this->{frame_width});
   $this->raw_vline(0, 1, $lw, $this->{attr});
   $this->raw_vline($this->{width} - (2 * $this->{frame_width}) + 1, 1, $lw);

   $this->raw_gotoxy(0, $this->{height}-$this->{frame_width});
   addch(ACS_LLCORNER);
   $this->raw_gotoxy($this->{width}-$this->{frame_width}, $this->{height}-1);
   addch(ACS_LRCORNER);

   attroff($this->{attr});

   if ($this->{frame} & 0x02) {     #title in frame?
      attron($this->{attr_title});
      my $s = '[ ' . $this->{name} . ' ]';
      $this->raw_print(2, 0, $s);
      attroff($this->{attr_title});
   }

} #draw_frame


# --- Aux subs ------------------------------------------------------------

=pod

=head1 Auxillary subroutines

(included, useful subs that aren't object methods)

=cut

#fatal() that includes object
sub lfatal {
   my($this) = @_;
   gothix::area::fatal(
                       "\n***** FATAL ERROR IN area.pm *****\n\n" .
                       "Dumping object:\n\n" .
                       Dumper($this) . "\n" . 
                       "\n***** FATAL ERROR IN area.pm *****\n\n"
                      );
}


#walk keys of hash...
sub clr_all {
   foreach my $a (keys %All_Areas) {
      $All_Areas{$a}->clr();
   }
} #clr_all


# --- resizer() -----------------------------------------------------------

sub resizer {

   return unless defined( $rootwin );     #Not running yet?

   $rootwin->set_size();
   foreach my $a (keys %All_Areas) {
      $All_Areas{$a}->d("** resizing area **");
      $All_Areas{$a}->set_size();
   }

} #resizer


# --- area_default_options() ----------------------------------------------

=pod

=head2 area_default_options() 

This sets default options for gothix::area objects created in the
future. Arguments are key => value pairs, i.e. C<attr => A_UNDERLINE>.

=cut

sub area_default_options {

   my %args = @_;

   foreach my $param (@areaOptions) {
      if (defined($args{$param})) {
         $DefaultOptions{$param} = $args{$param};
      }
   }

} #area_default_options


# === Root window class ===================================================

package gothix::area::rootwin;

use strict;
use warnings;
use Carp;

use gothix::oobase;
use gothix::misc qw( tdif );
use gothix::xcurses;
use gothix::area;

our @ISA = qw( gothix::area gothix::oobase );


# --- new() ---------------------------------------------------------------

sub new {

   my $class = shift;
   my $this = {};
   bless $this, $class;

   $this->debug_init( defined($ENV{AREA_DEBUG}) );

   $this->{name} = 'rootwin';
   $this->set_size();

   $this->{attr} = $dfl_attribute;
   $this->{frame_width} = 0;

   return $this;

} #new root


# --- raw_gotoxy() --------------------------------------------------------

#overrides ::area method 
sub raw_gotoxy {

   my($this,$x,$y) = @_;

   confess "rootwin::gotoxy($x,$y) - X is out of range, max=" . $this->maxx()
      if (($x < 0) || ($x > $this->maxx()));
   confess "rootwin::gotoxy($x,$y) - Y is out of range, max=" . $this->maxy()
      if (($x < 0) || ($y > $this->maxy()));

   abs_gotoxy($x, $y);

} #raw_gotoxy


# --- set_size() ----------------------------------------------------------

#overrides ::area method 
sub set_size {

   my $this = shift;

   $this->{x} = 0;                
   $this->{y} = 0;                
   $this->{width} = term_cols();  
   $this->{height} = term_lines();

   if ($this->debug()) {
      my $termsize = sprintf('l=%d,c=%d', term_lines(), term_cols());
      $this->d("set_size($termsize): ROOTWIN size now lines=$this->{height} cols=$this->{width}\n");
   }

} #set_size


# === POD trailer =========================================================

=pod 

=head1 Object description

Additional keys that can be set in the object:

=over 4

=item right_justify

If this is set (to any true value), then all output to the area via
print() will be right-justified.

=item debug

Set to any true value to cause object to print info to stderr.

=back

Example: C<$area->{debug} = 1;>  .. sets debugging on for C<$area>.

=cut


=head1 ENVIRONMENT

Set C<AREA_DEBUG> to any value to get debug output on stderr. This will set the
'debug' key in all objects.


=head1 DIAGNOSTICS

Uses L<gothix::oobase> for diagnostics.


=head1 AUTHOR(S)

Written by Stig H. Jacobsen, perl.area@gothix.biz, 18-Jun-2004


=head1 SEE ALSO

L<nsc>
L<Term::Console>
L<gothix::xcurses>


=head1 COPYRIGHT

Copyright Gothix / Stig H. Jacobsen 2004-2005 - input@gothix.biz


=cut


# --- .. ------------------------------------------------------------------

__END__

# $Id: area.pm,v 1.14 2004/11/13 16:36:34 gothix Exp $
# vim:aw:
# $HeadURL$
