# gothix::xcurses - glue on top of Curses.pm
# 30-Jun-2004/shj from nsc.pl
#
# --- TODO ----------------------------------------------------------------
#  - inline applicable subs - see man perlsub, /inline
#

package gothix::xcurses;

use strict;
use warnings;
use Carp;
use Curses;

use Exporter;

use gothix::misc;


# --- Code ----------------------------------------------------------------

our(@ISA, @EXPORT, $VERSION);
@ISA = qw( Exporter );


# --- Globals -------------------------------------------------------------

#private: &mvaddch &mvaddstr &at
@EXPORT = qw( 
               KEY_RESIZE
               $dfl_attribute
               &aat
               &abs_gotoxy     
               &bell
               &clear_screen
               &colour2attr  
               &colour_init
               &curses_init
               &curses_shutdown
               &fatal
               &mono2attr  
               &read_key     
               &term_redraw
               &scopy
               &term_set_size
               &term_attroff
               &term_attron
               &term_cols
               &term_has_colours
               &term_lines
               &term_refresh
            );
$VERSION = 0.01;


# --- Constants -----------------------------------------------------------

use constant KEY_RESIZE => 410;   #This keyname is not exported by Curses.pm :-(


# --- Globals -------------------------------------------------------------

my $curses_initialized = 0;

my ($fake_cols, $fake_lines) = (undef,undef);

my $msg_fatal = "***** FATAL ERROR *****";

my %attrMono = (
   'bold'      => A_BOLD,
   'blink'     => A_BLINK,
   'normal'    => A_NORMAL,
   'reverse'   => A_REVERSE,
   'dim'       => A_DIM,
   'underline' => A_UNDERLINE,
);

my %attrColour = (
   'white'     => COLOR_WHITE,
   'blue'      => COLOR_BLUE,
   'red'       => COLOR_RED,
   'yellow'    => COLOR_YELLOW,
   'green'     => COLOR_GREEN,
   'black'     => COLOR_BLACK,
   'magenta'   => COLOR_MAGENTA,
   'cyan'      => COLOR_CYAN,
   'default'   => -1,               #ncurses specific
);

#our $save_suspend_sigh = undef;
our $dfl_attribute = A_NORMAL;


# --- Publics -------------------------------------------------------------

sub curses_init {

   return if $curses_initialized;

   #sbs/dSafe/scon gets into trouble if we don't save and restore
   #(because apparently Curses.pm doesn't)
   #DOESNT HELP!! 08-May-2005/shj
   #bug recipe:
   #  scon.pl
   #  lconsole
   #  q
   #  ctrl-z to suspend scon.pl
   #  fg to resume
   #  (curses has taken control and redrawn stuff)
   #$save_suspend_sigh = $SIG{'STOP'};

   initscr(); cbreak(); noecho();
   nonl();
#&stdscr=warns, $stdscr=errs
#   intrflush(&stdscr, $FALSE);
#   keypad(&stdscr, $TRUE);

#   #wrap in eval, as this is ncurses specific
#   eval "use_default_colors()";

#attron(A_NORMAL);    #doesn't anyone else do this?

   colour_init() if term_has_colours();

   $curses_initialized = 1;

} #curses_init
   

sub curses_shutdown {

   return unless $curses_initialized;

   #erase();
   abs_gotoxy(0, 0); #neither works... #$LINES);
   term_refresh();
   endwin();

   #$SIG{'STOP'} = defined($save_suspend_sigh) ? $save_suspend_sigh : undef;

   $curses_initialized = 0;

} #curses_shutdown


#if perl stops, don't require stty sane et al
END {
   curses_shutdown() if $curses_initialized;
}


sub fatal {

   my ($message) = @_;

   if ($curses_initialized) {
      #popup($message);
      clear_screen();
      abs_gotoxy(0, term_lines()-1);
      curses_shutdown();
      }
   confess "\n\n$msg_fatal\n$message\nAborted.\n";

} #fatal


# --- Curses replacement functions ----------------------------------------
#
#
sub mvaddch {
   my($y,$x,$ch) = @_;
   move($y,$x);
   addch($ch);
}

sub mvaddstr {
   my($y,$x,$str) = @_;
   move($y,$x);
   addstr($str);
}


# --- term_redraw() -------------------------------------------------------

sub term_redraw {

   redrawwin(curscr);

} #term_redraw


# --- scopy() -------------------------------------------------------------
# Copy screen characters & attributes to string & array.

sub scopy {

   my($x, $y, $n) = @_;

   #return mvinnstr($y, $x, $n);

   my $cres = '';
   my @ares = ();
   while ($n > 0) {
      $n--;
      #my $ch = mvinch($y,$x);
      abs_gotoxy($x, $y);
      #my $ch = inch();
      my $ch;
      $ch = inch(&stdscr());
      $cres .= chr(ord($ch) & &A_CHARTEXT);
      push(@ares, ord($ch) & &A_ATTRIBUTES);
   }

   return wantarray ? ($cres, \@ares) : $cres;

} #scopy


# --- bell() --------------------------------------------------------------

=head2 C<bell()>

Beeps.

=cut

sub bell {

   beep();

} #bell


# --- Low level "graphics" ------------------------------------------------

sub abs_gotoxy {
   my($x,$y) = @_;

   fatal("gotoxy($x,$y): X out of range!") unless (($x >= 0) && ($x < term_cols()));
   fatal(sprintf("gotoxy($x,$y): Y out of range! (term_lines=%d)", term_lines()))
      unless (($y >= 0) && ($y < term_lines()));

   move($y,$x);

}


# read_key(5)  reads a key within 5 seconds
# wait indefinitely, if timeout value isn't given
sub read_key {

   term_refresh();            #in case coder forgot

   if (defined($_[0])) {
      timeout(1000*$_[0]);
   } else {
      timeout(-1);
   }
   my $res = getch();

   if (defined($res) && ($res eq '-1')) {
      return undef;
   } else {
      return $res;
   }

} #read_key


# at(int x, int y, const char *str)
sub at {

   my($x,$y,$str) = @_;

   if (substr($str,0,1) eq '!') {         #blank rest of line?
      $str = substr($str, 1);
      $str .= (' ' x ($COLS - length($str) - $x));
   } elsif (substr($str,0,1) eq '>') {    #relative to right side of display?
      $str = substr($str, 1);
      $x = $COLS - length($str) - $x;
      }

   #please control yourself or I will (we never wrap)
   if (length($str) > ($COLS-$x)) {
      #dm("truncated '$str'");
      $str = substr($str, 0, ($COLS-$x));
   }

   mvaddstr($y,$x,$str);

   return length($str);

} #at


# aat(int x, int y, attrib, const char *str)
sub aat {
  attron($_[2]);
  at($_[0], $_[1], $_[3]);
  attroff($_[2]);
  return length($_[3]);
}


# --- Mono stuff ----------------------------------------------------------

# Return Curses attribute value (for attron()) for named color
#  $a = mono2attr('reverse');   #returns perhaps A_REVERSE
#  attron($a);
sub mono2attr {

   my ($attr) = @_;

   my $val = $attr;
   $val =~ s/\s+//g;
   my @attrs = split(/\+/, $val);

   my $res = undef;
   foreach my $as (@attrs) {
      fatal("Unknown monochrome attribute '$as'")
         unless defined($attrMono{$as});
      my $a = $attrMono{$as};
      #$a = $attrMono{$a} if defined($attrMono{$a});    #indirect, i.e. mono.heading = 'normal'
      if (defined($res)) {
         $res |= $a;
      } else {
         $res = $a;
      }
   }

   return $res;

} #mono2attr


# --- Colour configuration ------------------------------------------------

{
#current colour pair
my $pairno;

my %myColourPairs;          #cache

sub colour_init {

   start_color();
   %myColourPairs = ();
   #colorpairs starts at 1
   $pairno = 1;

} #colour_init
   

# colour2attr()
# As mono2attr(), but for colour attributes
sub colour2attr {

   my ($attr) = @_;

   defined($attr) || fatal("colour2attr: attribute not defined");
   fatal("colour2attr: attribute empty")
      unless ($attr ne '');

   return $myColourPairs{$attr} if defined($myColourPairs{$attr});

   # $val = 'white on blue', optionally '+ bold'
   my $val = $attr;

   my $cpair = ++$pairno;
   if ($cpair >= COLOR_PAIRS) {
      confess "Out of curses colour-pairs, this=$cpair";
   }

   my ($fore,$back);

   if (!(($fore,$back) = split(/\s+on\s+/, $val))) {
      $fore = $val;
   }
   $back = 'default' unless $back;

   my @plusses = ();
   my @temp;
   ($fore,@temp) = split(/[\s+]+/, $fore);
   push(@plusses, @temp);
   ($back,@temp) = split(/[\s+]+/, $back);
   push(@plusses, @temp);

   $attrColour{$fore} = $attrMono{$fore}
      unless defined($attrColour{$fore});
   $attrColour{$back} = $attrMono{$back}
      unless defined($attrColour{$back});
   
   #print STDERR "CPAIR = $cpair  BACK = $back   FORE = $fore\n";
   defined($attrColour{$fore}) || fatal("Foreground colour '$fore' is unknown");
   defined($attrColour{$back}) || fatal("Background colour '$back' is unknown");

   init_pair($cpair, $attrColour{$fore}, $attrColour{$back});
   my $res = COLOR_PAIR($cpair);
   foreach my $plus (@plusses) {
      if (!defined($attrMono{$plus})) {
         print STDERR "KEYS: ", keys %attrMono, "!\n";
         foreach my $m (sort keys %attrMono) {
            print STDERR "$m => $attrMono{$m}\n";
         }
         fatal("Unknown +attribute '$plus' ($attr)");
      }
      my $a = $attrMono{$plus};
      $res |= $a;
   }

   $myColourPairs{$attr} = $res;         #cache for next time
   return $res;

} #colour2attr

} #brace


# --- Blanks screen with the named attribute ------------------------------

sub clear_screen {

   my ($attr) = @_;

   $attr = A_NORMAL unless $attr;
   bkgd($attr | ord(' '));
   erase();

} #clear_screen


# --- term_set_size() -----------------------------------------------------
# Call with (undef,undef) to return to reality
sub term_set_size {

   my ($cols,$lines) = @_;

   $fake_cols = $cols;
   $fake_lines = $lines;

} #set_term_size


# --- term_xxx() ----------------------------------------------------------

sub term_cols {

   return defined($fake_cols) ? $fake_cols : $COLS;

} #term_cols


sub term_lines {

   return defined($fake_lines) ? $fake_lines : $LINES;

} #term_lines


sub term_refresh {

   refresh();

} #term_refresh


sub term_has_colours {

   return has_colors();

} #term_has_colours


sub term_attron {

   my ($attr) = @_;

   return attron($attr);

} #term_attron


sub term_attroff {

   my ($attr) = @_;

   return attroff($attr);

} #term_attroff


# --- Positive module init ------------------------------------------------

my $this_module_initialized_successfully = 42;


# =========================================================================
#                           A L L   D O N E  
# =========================================================================

__END__


# vim:aw:
# $Id: xcurses.pm,v 1.9 2004/11/13 16:36:34 gothix Exp $
# $HeadURL$
