#
# nsc_nagios.pm -- Nagios v1/2 interface module for nsc
#
# Copyright (C) Stig H. Jacobsen & Gothix 2000-2005
#
# E-mail: nsc@gothix.biz
#


package nsc::statuslog;

use strict;
use warnings;

use IO::File;
use Fcntl qw( SEEK_SET );

use gothix::misc qw( &inode &mtime );
use gothix::oobase;

our @ISA = qw( gothix::oobase );


# --- Module initialization -----------------------------------------------

# Maps Nagios2 numeric states to appropriate (Nagios1) labels
# http://nagiosplug.sourceforge.net/developer-guidelines.html#RETURNCODES
our @v2_SstateMap = (
   'OK',
   'WARNING',
   'CRITICAL',
   'UNKNOWN',
);

our @v2_HstateMap = (
   'UP',
   'DOWN',
   'UNKNOWN',
);

# Maps Nagios2 status.log types to Nagios1/nsc types
# Also used for ignoring unknown/unused types
our $v2_typeMap = {
   'host'      => 'HOST',
   'service'   => 'SERVICE',
   'info'      => 'INFO',
   'program'   => 'PROGRAM',
}; 


our $module_truth_return = 42;


# === v1 methods ==========================================================

sub _v1_parser {

   my $this = shift;

   # '# Nagios 1.1 Status File'
   if ($this->{comments}->[0] =~ /^#\s+(\S+)\s+(\S+)\sStatus/i) {
      $this->{application} = $1;
      $this->{version} = $2;
   }

   my @res = ();
   foreach my $line (@{ $this->{lines} }) {
      my @SVC = split(/[;\n]/, $line);
      my $row = {};

      return $this->_srERR(__PACKAGE__ . "::v1_parser(): Format unrecognized: \"$SVC[0]\"")
         if (!defined( $SVC[0] ) or $SVC[0] !~ /^\[(\d+)] (.*)$/);

      $row->{'tm'} = $1;
      $row->{'type'} = $2;

      if ($row->{'type'} eq 'SERVICE') {
         #[1078349554] SERVICE;dax;IMAP mail;CRITICAL;3/3;HARD;1078349453;1078349573;ACTIVE;1;1;1;1064879368;0;CRITICAL;0;0;0;11514636;1078338644;864;1;0;0;1;0;0.00;0;1;1;1;(Return code of 127 is out of bounds - plugin may be missing)
         ($row->{'node'},
         $row->{'service'},
         $row->{'status'},
         $row->{'attempts'},
         $row->{'updated'},
         $row->{changed},
         $row->{'ack'}) = (@SVC[1..4],$SVC[6], $SVC[12], $SVC[13]);
         $row->{'info'} = $SVC[$#SVC];
      } elsif ($row->{'type'} eq 'HOST') {
         # [1078349554] HOST;localhost;UP;1078349454;1064880057;0;11513957;529;0;0;0;1;1;1;1;0;0.00;0;1;1;PING OK - Packet loss = 0%, RTA = 0.26 ms
         $row->{'node'} = $SVC[1];
         $row->{'status'} = $SVC[2];
         $row->{updated} = $SVC[3];
         $row->{changed} = $SVC[4];
         $row->{ack} = $SVC[5];           #is this it?
         $row->{'info'} = $SVC[$#SVC];
         #TODO: which column is ack!?
      } elsif ($row->{'type'} eq 'PROGRAM') {
         # [1078349554] PROGRAM;1076394004;2714;1;1078349539;1078009200;1;1;1;1;0;0;1;0
         $this->{uptime} = $SVC[1];
         $this->{appl_status} = $SVC[3];
      }
      #ignore other row types
      
      if ($this->{filter_sub} && (! $this->{filter_sub}->($row))) {
         next;    #filter ate it
      }

      push(@res, $row);
   }

   return \@res;

} #_v1_parser


# === v2 methods ==========================================================

sub _v2_parse_entry {

   my($this, $typ, $cref) = @_;

   $this->d("v2::_parse_entry '$typ', '${$cref}'");

   return undef                               #Ignore unknown/unused types
      unless defined( $v2_typeMap->{$typ}) ;

   my $entry = {};
   $entry->{type} = $v2_typeMap->{$typ};

   foreach my $line (split(/\n\s*/, ${$cref})) {
#      $line =~ s/(^\s+|\s+$)//gs;
      next if ($line eq '');        #not sure where these come from
      my ($kw,$val) = split(/=/, $line, 2);
      $entry->{$kw} = $val;
   }

   # Convert to nsc internal format
   
   my $row = {};
   $row->{type} = $v2_typeMap->{$typ};

   if ($row->{'type'} eq 'SERVICE') {
      $row->{updated}   = $entry->{last_check};
      $row->{changed}   = $entry->{last_state_change};
      $row->{attempts}  = $entry->{current_attempt} . '/' . $entry->{max_attempts};
      $row->{node}      = $entry->{host_name};
      $row->{service}   = $entry->{service_description};
      $row->{status}    = $v2_SstateMap[ $entry->{current_state} ];
      $row->{info}      = $entry->{plugin_output};
      $row->{ack}       = $entry->{problem_has_been_acknowledged};
#      if ($row->{service} =~ /Otto/i) {
#         use Data::Dumper;
#         print STDERR "RAW v1 entry from status.log:\n", Dumper($entry), "\n.\n";
#         print STDERR "nsc row:\n", Dumper($row), "\n.\n";
#      }
   } elsif ($row->{'type'} eq 'HOST') {
      $row->{updated}   = $entry->{last_check};
      $row->{changed}   = $entry->{last_state_change};
      $row->{node}      = $entry->{host_name};
      $row->{info}      = $entry->{plugin_output};
      $row->{ack}       = $entry->{problem_has_been_acknowledged};
      $row->{status}    = $v2_HstateMap[ $entry->{current_state} ];
   } elsif ($row->{'type'} eq 'INFO') {
      $this->{application} = 'Nagios';
      $this->{version}     = $entry->{version};
   } elsif ($row->{'type'} eq 'PROGRAM') {
      $this->{uptime}      = $entry->{program_start};
      $this->{appl_status} = 1;      #file exists, so its running..
   } else {
      #otherwise ignore, but we shouldn't get here ever due to the typemapping
      dm("unrecognized type '$row->{'type'}'");
      }

   return $row;

} #_v2_parse_entry


# -------------------------------------------------------------------------

sub _v2_parser {

   my $this = shift;
   my $lines = join('', @{ $this->{lines} });    #one long string now

   my @res = ();

   while ($lines =~ /^(.*) \n([a-z]+) \s+ \{ (.*) \} (.*)$/xs) {
      my ($pre,$typ,$content,$post) = ($1,$2,$3,$4);
      $lines = $pre . $post;
      my $row = $this->_v2_parse_entry($typ, \$content);
      if ($this->{filter_sub} && (! $this->{filter_sub}->($row))) {
         next;
      }
      push(@res, $row);
   }

   #debug check, should be empty..
   if ($this->{debug}) {
      $lines =~ s/\s+//g;
      $this->d("remaining lines = '$lines'");
   }

   return \@res;

} #_v2_parser


# === Version independent methods =========================================

sub new {

   my $class = shift;
   my $this = {};
   bless $this, $class;

   $this->debug_init();

   $this->{filename} = shift;    #logical name, if remote

   ($this->{copy_cmd},$this->{copy_interval}) = (shift,shift);
   $this->{filter_sub} = shift;

   ($this->{remote_host},$this->{remote_file}) = $this->is_remote();

   $this->_srOK();
   return $this;

} #new


# -------------------------------------------------------------------------

sub DESTROY {

   my $this = shift;

   close($this->{fob})
      if $this->{fob};

   unlink($this->{temp_status})
      if $this->{temp_status};

   unlink($this->{copy_log})
      if $this->{copy_log};

} #DESTROY


# -------------------------------------------------------------------------

sub is_remote {

   my $this = shift;

   return undef unless $this->{filename};
   $this->{filename} =~ /^([^\s\/:]{2,}):(.*)$/ || return undef;

   return wantarray() ? ($1,$2) : 1;

} #is_remote


# -------------------------------------------------------------------------

sub _detect_format {

   my $this = shift;

   foreach (@{ $this->{lines} }[0..9]) {
      next if (/^\s*$/);
      return 1 if (/^\[\d{8,}\] [A-Z]{3,};/);   # v1: [1100780937] PROGRAM;1100694447;7201
      return 2 if (/^\s*\S{3,}\s+\{/);          # v2: 'host {'
   }
   return undef;

} #_detect_format


# -------------------------------------------------------------------------

sub _local_logname {

   my $this = shift;

   return $this->{temp_status} 
      if $this->{remote_file};

   # Local status.log:

   if (! -e $this->{filename}) {
      # try status.sav, in case nagios is stopped
      my $xfn = $this->{filename};
      $xfn =~ s!\.log$!.sav!i;
      return $xfn
         if (-e $xfn);
   }

   return $this->{filename};

} #_local_logname


# -------------------------------------------------------------------------

sub _open {

   my $this = shift;

   if ($this->{fob}) {           #wants to reopen?
      $this->{fob}->close();
   }

   $this->{fob} = new IO::File('<' . $this->_local_logname());
   return $this->_srERR("$! on open of " . $this->_local_logname())
      unless $this->{fob};

   # Save the files inode#, so that if Nagios creates a new status.log
   # (instead of rewriting the old) we will detect it
   $this->{_status_inode} = inode($this->_local_logname());
#   print STDERR "got $! after stat(2)\n"
#      unless $this->{_status_inode};

   return 1;

} #_open


# -------------------------------------------------------------------------

sub _remote_sync {

   my $this = shift;

   return $this->_srERR('No remote copy command defined')
      unless $this->{copy_cmd};

   return $this->_srERR("File is not remote! ($this->{filename})")
      unless $this->{remote_host};

   $this->{copy_interval} = 30
      unless $this->{copy_interval};

   if (!defined( $this->{temp_status} )) {
      $this->{temp_status} = "/tmp/status.log.$$";
      $this->{copy_log}    = "/tmp/status.log.$$.copylog";
   }

   if (defined( $this->{last_remote_sync} )) {
      return $this->_srOK()         #just return happily, if it isn't time yet
         unless (time() > ($this->{last_remote_sync} + $this->{copy_interval}));
   }

   $this->{last_remote_sync} = time();

   my $cmd = $this->{copy_cmd};
   $cmd =~ s/%H/$this->{remote_host}/g;
   $cmd =~ s/%F/$this->{remote_file}/g;
   $cmd =~ s/%D/$this->{temp_status}/g;
   $cmd =~ s/%L/$this->{copy_log}/g;

   $this->d("execute copy command '$cmd'");

   system($cmd);
   my $res = $?;

   #TODO: umask 0600 before remote cp?
   chmod(0600, $this->{temp_status});     #better than nothing

   if ($res == 0 and
       -e $this->{temp_status} and
       -s $this->{temp_status} > 0) {
      return $this->_srOK();
   } else {
      delete $this->{last_remote_sync};
      return $this->_srERR("Copy of $this->{remote_host}:$this->{remote_file} failed ($res)");
   }

} #_remote_sync


# -------------------------------------------------------------------------

sub _real_load {

   my $this = shift;

   return $this->_srERR("File not opened(?!)") unless $this->{fob};      #open it first, please

   # --- Wipe old data ---

   #init counts for host states
   $this->{hState} = {};         
   foreach my $k ('UP','DOWN','UNREACHABLE', 'ACK') {
      $this->{hState}->{$k} = 0;
   }

   #init counts for service states
   $this->{sState} = {};         
   foreach my $k ('OK', 'CRITICAL', 'WARNING', 'UNKNOWN', 'PENDING', 'HOST DOWN', 'ACK') {
      $this->{sState}->{$k} = 0;
   }

   $this->{num_svc_unacknowledged} = 
      $this->{num_host_unacknowledged} = 0;

   # --- Load it ---

   $this->{fob}->seek(0, SEEK_SET);    #rewind, also clears eof condition + discards old data(?)

   my $tm_load = mtime($this->_local_logname());

   my $fob = $this->{fob};          #Perl v5.8.6 doesn't grok  <$this->{fob}> ..
   my @lines = <$fob>;

   return $this->_srERR("No data read from file")
      if (@lines <= 0);

   @{ $this->{comments} } = grep(/^#/, @lines);     #v1 format needs these
   {
      my @res = grep(!/^#/, @lines);
      $this->{lines} = \@res;
   }

   if (! defined($this->{format}) ) {          #check format of file?
      $this->{format} = $this->_detect_format();
      return $this->_srERR("Unrecognized file-format in $this->{filename}")
         unless defined( $this->{format} );
      if (1 == $this->{format}) {
         $this->{parser_sub} = \&_v1_parser;
      } elsif (2 == $this->{format}) {
         $this->{parser_sub} = \&_v2_parser;
      } else {
         return $this->_srERR("Unsupported file-format in $this->{filename}");
      }
   }

   $this->{_tm_loaded} = $tm_load;     #only change if success

   return 1;

} #_real_load


# -------------------------------------------------------------------------

sub time_loaded {

   my $this = shift;

   return undef
      unless $this->{_tm_loaded};

   return $this->{_tm_loaded};

} #time_loaded


# -------------------------------------------------------------------------

sub dump_states {

   my $this = shift;

   print STDERR "hState:\n";
   foreach my $k (sort keys %{$this->{hState}}) {
      print STDERR "  $k => $this->{hState}->{$k}\n";
   }
      
   print STDERR "sState:\n";
   foreach my $k (sort keys %{$this->{sState}}) {
      print STDERR "  $k => $this->{sState}->{$k}\n";
   }
      
} #dump_states


# -------------------------------------------------------------------------

sub dump_rows {

   my $this = shift;

   foreach my $row (@{$this->{rows}}) {
      print STDERR "$row->{type}:\n";
      foreach my $kw (sort keys %{$row}) {
         print STDERR "  '$kw' => '$row->{$kw}'\n";
      }
   }
      
} #dump_rows


# -------------------------------------------------------------------------
#Returns true, if status.log has changed (stat(2).mtime) since last time we loaded it.
sub changed {

   my $this = shift;

   $this->_remote_sync() if $this->{remote_host};

   my $fn = $this->_local_logname();
   my $mt = mtime($fn);
   return $this->_srERR("No such file ($fn)") unless $mt;            #no such file

   return 1
      unless $this->{_tm_loaded};      #it has changed, if it isn't loaded
   
   $this->_srOK();
   return ($mt > $this->{_tm_loaded}) ? 1 : undef;

} #changed


# -------------------------------------------------------------------------
#Returns reference to array with hashrefs
#undef for errors, errormessage in $this->{error}
sub load {

   my $this = shift;

   $this->{appl_status} = 0;         #assume that we fail

   if ($this->{remote_host}) {
      $this->_remote_sync() || return undef;
   }

   #TEMP-TEST-TODO: force reopen, otherwise we just get old data..., 16-May-2005/shj
   $this->_open() || return undef;  

   #stat() the file, see if the inode changed....
#   if (!defined( $this->{fob} )) {
#      $this->_open() || return undef;
#   } elsif ($this->{_status_inode} != inode($this->_local_logname())) {
#      #print STDERR "inode changed! reopening....\n";
#      $this->_open() || return undef;
#   } else {
#      1; #printf(STDERR "file is open, inode is unchanged $this->{_status_inode}");
#   }

   $this->_real_load() || return undef;

   $this->{rows} = $this->{parser_sub}->($this);   #parse it

   # --- count states & acknowledged stuff
   foreach my $row (@{ $this->{rows} }) {
      if ($row->{'type'} eq 'SERVICE') {
         $this->{sState}->{$row->{status}}++;
         if ($row->{ack}) {
            $this->{sState}->{ACK}++;
         } elsif ($row->{status} ne 'OK') {
            $this->{num_svc_unacknowledged}++;
         }
      } elsif ($row->{'type'} eq 'HOST') {
         $this->{hState}->{$row->{status}}++;
         if ($row->{ack}) {
            $this->{hState}->{ACK}++;
         } elsif ($row->{status} ne 'UP') {
            $this->{num_host_unacknowledged}++;
         }
      }
   }

   $this->{application} = '?appl?'
      unless $this->{application};
   $this->{version} = '?ver?'
      unless $this->{version};

   # env OOBASE_DEBUG=1 ./APPL.pl
   $this->dump_object() if $this->{debug};

   $this->_srOK();
   return $this->{rows};

} #load


# =========================================================================

__END__

# # Nagios 1.0b6 Status File
# [1041516783] PROGRAM;1037463573;27646;1;1041516768;1041116400;1;1;1;1;0;0;1;0
#from nagions 1.06b6 src: snprintf(new_program_string,sizeof(program_string)-1,
#      "[%lu] PROGRAM;%lu;%d;%d;%lu;%lu;%d;%d;%d;%d;%d;%d;%d;%d\n",
#  0   (unsigned long)current_time,
#  1   (unsigned long)_program_start,
#  2   _nagios_pid,
#  3   _daemon_mode,
#  4   (unsigned long)_last_command_check,
#  5   (unsigned long)_last_log_rotation,
#  6   _enable_notifications,
#  7   _execute_service_checks,
#  8   _accept_passive_service_checks,
#  9   _enable_event_handlers,
# 10   _obsess_over_services,
# 11   _enable_flap_detection,
# 12   _enable_failure_prediction,
# 13   _process_performance_data);

# $Id: nsc_nagios.pm 2876 2005-05-18 13:36:39Z shj $
# vim:aw:
