/*
 * testio - Program/Nagios Plugin for checking response time of filesystem
 * version: 0.1.2
 * Author: Klemen Andreuzzi, klemen.andreuzzi@arnes.si
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Copy of GNU General Public License is available at
 * http://www.gnu.org/copyleft/gpl.html
 *
 * Violations of this license should be reported to
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <aio.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <string.h>

int timespec_milliseconds(struct timespec *a); 
void printHelp();

int main( int argc, char *argv[] )
{
		
	if(argc == 1) {
		printHelp();
		return 0;
	}

	//misc
	int i=0;
	int c;
	short mode=0; //read = 0, write = 1

	//time bool
	short test_critical;
	short test_warning;
	
	//time
	struct timespec start, end, current;

  	//buffer size
        unsigned int BUFF_SIZE=1000000;

	//file descriptor
	int file=-2;
	char * filename;

	//time miliseconds
	int warning=0, critical=0;
	
	//parse arguments	
	while ((c = getopt (argc, argv, ":h:iof:w:c:s:")) != -1) {	
		switch(c) {
			case 'h':
				printHelp();
				return 0;
				break;
			case 'i':
				mode=0;
				break;
			case 'o':
				mode=1;
				break;
			case 'f':
				if(strlen(optarg) > 254) {
					printf("testio UNKNOWN filename not OK\n");
                                        return 3;
				}
				filename = malloc (sizeof(char) * strlen(optarg));
				strcpy(filename,optarg);
				file = open(filename, O_RDWR|O_CREAT|O_NONBLOCK, S_IRUSR|S_IWUSR);
				break;
			case 'w':
				warning = atoi(optarg);
				if (warning < 1) {
					 printf("testio UNKNOWN warning value not OK\n");
       				         return 3;
				}
				break;
			case 'c':
				critical = atoi(optarg);
				if (critical < 1) {
                                         printf("testio UNKNOWN critical value not OK\n");
                                         return 3;
                                }
				break;
			case 's':
				BUFF_SIZE = atoi(optarg);
				if (BUFF_SIZE < 1) {
                                         printf("testio UNKNOWN size of file not OK\n");
                                         return 3;
                                }
				break;
			default:
				printHelp();
				return 0;
		}
	}	

	if ((warning == 0) || (critical == 0) || (warning > critical)) {
		printf("testio UNKNOWN critical and warning values not OK\n");
		return 3;
	}

	if (file == -2) {
		printf("testio UNKNOWN specify filename \n");
		return 3;
	}
	if (file == -1) {
		printf("testio CRITICAL cant open file %s \n",filename);
		return 1;
	}	


	//if in write mode
if (mode == 1) {	
	// create the buffer and fills it
	char* buffer = (char *) malloc ((BUFF_SIZE)*sizeof(char *));
	memset(buffer,'X',BUFF_SIZE);
	
	// create the control block structure
	struct aiocb cb;

	//fills struct with zeros
	bzero( (char *)&cb, sizeof(struct aiocb) );	

	//fills the sturct with values
	cb.aio_nbytes = BUFF_SIZE;
	cb.aio_fildes = file;
	cb.aio_offset = 0;
	cb.aio_buf = buffer;

	//sends kernel command to write to file
	int ret = aio_write( &cb );

	//checks if kernel got the call
  	if (ret < 0) {
		printf("testio WARNING kernel didnt recive file write call, file %s\n",filename);
		return 1;
	}


	//gets current time and sets warning and critical time
	clock_gettime(CLOCK_REALTIME, &start);

	//test = 0
	test_critical=0;
        test_warning=0;	

	//waits until its done or time elapsed
	do {
		clock_gettime(CLOCK_REALTIME, &current);
		int mili = timespec_milliseconds(&current) - timespec_milliseconds(&start);
		if (mili > critical) {
			test_critical = 1;
		} else if (mili >= warning) {
			test_warning = 1;
		}

	} while((aio_error(&cb) == EINPROGRESS) && (test_critical == 0));

	//checks if it is realy done
	if ((aio_return( &cb ) == BUFF_SIZE) && (test_critical == 0) && (test_warning == 0)) {
		printf("testio OK time  %dms, file %s\n",timespec_milliseconds(&current) - timespec_milliseconds(&start),filename);
		close(file);
	        free(buffer);
		unlink(filename);
		return 0;
	} else if ((aio_return( &cb ) == BUFF_SIZE) && (test_warning == 1))  {
		printf("testio WARNING time  %dms, file %s\n",timespec_milliseconds(&current) - timespec_milliseconds(&start),filename);
                close(file);
                free(buffer);
		unlink(filename);
                return 1;
	} else {
		//if not done yet, terminate request
		ret = aio_cancel(file,&cb);
		struct aiocb *cblist[1] = {&cb};
		end.tv_sec=2;
		end.tv_nsec=0;
		ret = aio_suspend(cblist,1,&end);
		if (ret == 0) {
			printf("testio CRITICAL taking too long, but job caceled, file %s \n",filename);
			close(file);
	                free(buffer);
			free(filename);
			unlink(filename);
       		        return 2;
		} else {
			printf("testio CRITICAL taking too long and job still running, file %s \n",filename);
			return 2;
		}
	
	} 
} else { 
	//in read mode
	// create the buffer
        char* buffer = (char *) malloc ((BUFF_SIZE)*sizeof(char *));

        // create the control block structure
        struct aiocb cb;

        //fills struct with zeros
        bzero( (char *)&cb, sizeof(struct aiocb) );

        //fills the sturct with values
        cb.aio_nbytes = BUFF_SIZE;
        cb.aio_fildes = file;
        cb.aio_offset = 0;
        cb.aio_buf = buffer;

        //sends kernel command to read from file
        int ret = aio_read( &cb );

	//checks if kernel got the call
        if (ret < 0) {
                printf("testio WARNING kernel didnt recive file read call, file %s\n",filename);
                return 1;
        }

        //gets current time and sets warning and critical time
        clock_gettime(CLOCK_REALTIME, &start);

        //test = 0
        test_critical=0;
        test_warning=0;

	//waits until its done or time elapsed
        do {
                clock_gettime(CLOCK_REALTIME, &current);
                int mili = timespec_milliseconds(&current) - timespec_milliseconds(&start);
                if (mili > critical) {
                        test_critical = 1;
                } else if (mili >= warning) {
                        test_warning = 1;
                }

        } while((aio_error(&cb) == EINPROGRESS) && (test_critical == 0));


	//checks if it is realy done
        if ((aio_return( &cb ) == BUFF_SIZE) && (test_critical == 0) && (test_warning == 0)) {
                printf("testio OK time  %dms, file %s\n",timespec_milliseconds(&current) - timespec_milliseconds(&start),filename);
                close(file);
                free(buffer);
                return 0;
        } else if ((aio_return( &cb ) == BUFF_SIZE) && (test_warning == 1))  {
                printf("testio WARNING time  %dms, file %s\n",timespec_milliseconds(&current) - timespec_milliseconds(&start),filename);
                close(file);
                free(buffer);
                return 1;
        } else {
                //if not done yet, terminate request
                ret = aio_cancel(file,&cb);
                struct aiocb *cblist[1] = {&cb};
                end.tv_sec=2;
                end.tv_nsec=0;
                ret = aio_suspend(cblist,1,&end);
                if (ret == 0) {
                        printf("testio CRITICAL taking too long, but job caceled, file %s \n",filename);
                        close(file);
                        free(buffer);
                        free(filename);
                        return 2;
                } else {
                        printf("testio CRITICAL taking too long and job still running, file %s \n",filename);
                        return 2;
                }

        }
	
}
	printf("testio UNKNOWN error \n");
	return 3;
}

void printHelp() {
	printf("\nUSAGE: testio [-i OR -o] [-f <file> -w <warning> -c <critical>] [OPTIONAL -s <buffer_size>]\n");
	printf("\t<-i OR -o> check read OR write.\n");
        printf("\t<file> file on partition to check.\n");
	printf("\t\tIf in read mode, then file should be at least buffer_size bytes (file is not deleted)\n");
	printf("\t\tIf in write mode, file should not exist (file is deleted after check)\n");
        printf("\t<warning> warning time in miliseconds\n");
        printf("\t<critical> critical time in miliseconds\n");
        printf("\t<buffer_size> size of file in bytes (default 1000000)\n");
	printf("\t\tWARNING buffer_size must fit in to your memmory!!!\n");
}

int timespec_milliseconds(struct timespec *a)
{
    return a->tv_sec*1000 + a->tv_nsec/1000000;
}
